# MaxMind Portfolio Website

Professional portfolio and content management system for MaxMind - Innovation Technology, a digital solutions company based in Morocco.

## 📋 Table of Contents

- [Overview](#overview)
- [Features](#features)
- [Tech Stack](#tech-stack)
- [Project Structure](#project-structure)
- [Installation](#installation)
- [Configuration](#configuration)
- [CMS Administration](#cms-administration)
- [API Documentation](#api-documentation)
- [Frontend Pages](#frontend-pages)
- [Security](#security)
- [Database Schema](#database-schema)
- [Contact Form](#contact-form)
- [SEO Optimization](#seo-optimization)

## 🎯 Overview

This project is a full-featured portfolio website with a custom-built Content Management System (CMS). It showcases digital projects, blog articles, and company services while providing a powerful backend for content management with multi-language support.

**Live Site:** [portfolio.maxmind.ma](https://portfolio.maxmind.ma)

## ✨ Features

### Frontend Features
- **Responsive Design** - Mobile-first approach with Tailwind CSS
- **Interactive Animations** - GSAP ScrollTrigger animations and smooth transitions
- **Hero Carousel** - Dynamic image carousel for the hero section
- **Portfolio Showcase** - Filterable project gallery with categories
- **Blog System** - SEO-optimized blog articles with rich content
- **Project Types** - Categorized projects (Web Development, Mobile Apps, E-commerce, UI/UX)
- **Timeline Section** - "Small Wins One Step at a Time" process timeline
- **Contact Form** - Integrated form with PHPMailer and Google Sheets integration
- **Social Integration** - Floating social media buttons
- **Multi-language Meta Tags** - Enhanced SEO for French and Arabic content
- **Sitemap Generation** - Dynamic XML sitemap for search engines

### CMS Features
- **Custom SQLite CMS** - Lightweight, file-based database
- **Category Management** - Flexible content categorization system
- **Dynamic Fields** - Support for multiple field types:
  - Text, Textarea, Number, Email, URL, Date
  - Checkbox, Image, File, Media
  - Rich Text Editor with character limits
  - Foreign Keys, Arrays, JSON, Links
- **Media Manager** - Upload and manage images/files with metadata
- **Multi-language Support** - Built-in translation system for content
- **User Authentication** - JWT-based authentication for admin access
- **API Key Protection** - Secure API endpoints with token validation
- **Search Functionality** - Searchable content across categories
- **Slug Generation** - SEO-friendly URLs for content
- **Export/Import** - Database export and configuration management

## 🛠 Tech Stack

### Backend
- **PHP 7.4+** - Server-side scripting
- **SQLite3** - Lightweight database
- **PDO** - Database abstraction layer
- **JWT** - JSON Web Tokens for authentication
- **PHPMailer** - Email handling for contact forms

### Frontend
- **HTML5/CSS3** - Modern web standards
- **JavaScript (ES6+)** - Client-side interactivity
- **Tailwind CSS** - Utility-first CSS framework
- **GSAP** - Animation library with ScrollTrigger
- **Lucide Icons** - Modern icon library

### Third-Party Services
- **Google Sheets API** - Contact form submissions storage
- **PHPMailer** - SMTP email delivery

## 📁 Project Structure

```
PORTFOLIO.MAXMIND.MA-main/
├── admin/                          # CMS Administration
│   ├── cms.php                     # Main CMS application (API + Frontend)
│   ├── config.php                  # Configuration and API key
│   ├── cms.db                      # SQLite database
│   ├── uploads/                    # Uploaded media files
│   └── .env                        # Environment variables (API key)
│
├── backend/                        # Backend PHP logic for frontend
│   ├── backend.php                 # Core backend functions
│   ├── portfolio.php               # Portfolio data fetching
│   ├── projects.php                # Project utilities
│   ├── blog.php                    # Single blog article
│   └── blogs.php                   # Blog listing
│
├── css/                           # Stylesheets
│   ├── index.css                  # Homepage styles
│   ├── portfolio.css              # Portfolio page styles
│   ├── projects.css               # Project listing styles
│   └── projectType.css            # Project type filtering
│
├── js/                            # JavaScript files
│   ├── index.js                   # Homepage animations and interactions
│   ├── carousell.js               # Carousel functionality
│   ├── blogs.js                   # Blog page scripts
│   ├── projects.js                # Project filtering
│   └── portfolio.php              # Dynamic portfolio data loader
│
├── mail/                          # Email handling
│   ├── submit-form.php            # Contact form processor
│   ├── PHPMailer/                 # PHPMailer library
│   ├── config/connect.php         # Database connection for forms
│   └── glorypushtosheets-*.json   # Google Sheets API credentials
│
├── imgs/                          # Image assets
│   └── LOGO_MAXMIND_BLACK.png     # Company logo
│
├── index.php                      # Homepage
├── portfolio.php                  # Portfolio showcase page
├── project.php                    # Individual project detail page
├── projectsType.php               # Projects filtered by type
├── blog.php                       # Single blog article page
├── blogs.php                      # Blog listing page
├── header.php                     # Site header component
├── footer.php                     # Site footer component
├── header-component.php           # Reusable header
├── hero-section.php               # Hero section component
├── social-floating.php            # Floating social buttons
├── sitemap.php                    # Dynamic sitemap generator
├── robots.txt                     # Search engine directives
└── README.md                      # This file
```

## 🚀 Installation

### Prerequisites
- **XAMPP** (Apache + PHP 7.4+ + SQLite support)
- **Git** (for version control)
- **Web Browser** (Chrome, Firefox, Safari, Edge)

### Steps

1. **Clone the repository**
   ```bash
   cd c:\xampp\htdocs
   git clone <repository-url> PORTFOLIO.MAXMIND.MA-main
   ```

2. **Navigate to project**
   ```bash
   cd PORTFOLIO.MAXMIND.MA-main\PORTFOLIO.MAXMIND.MA-main
   ```

3. **Set up environment variables**
   - Create `.env` file in `admin/` directory:
   ```
   API_KEY=your_secure_api_key_here
   ```

4. **Configure database permissions**
   ```bash
   # Ensure admin directory is writable for SQLite
   chmod 755 admin/
   chmod 666 admin/cms.db
   ```

5. **Start XAMPP**
   - Start Apache server
   - Access: `http://localhost/PORTFOLIO.MAXMIND.MA-main/PORTFOLIO.MAXMIND.MA-main/`

6. **Access CMS**
   - Navigate to: `http://localhost/PORTFOLIO.MAXMIND.MA-main/PORTFOLIO.MAXMIND.MA-main/admin/cms.php`
   - Default credentials will be created on first run

## ⚙️ Configuration

### API Key Setup

The API key is stored in `admin/.env`:

```env
API_KEY=2250c47a4397a41fe5ff8f2c2a92b240ddf46aaeded8ffddb4f29f02ffef8720
```

This key is required for all API requests. It's loaded by [admin/config.php](admin/config.php#L4-L18).

### Database Configuration

Database settings are defined in [admin/cms.php](admin/cms.php#L4-L6):

```php
define('DB_TYPE', 'sqlite');
define('DB_NAME', 'cms.db');
define('JWT_SECRET', 'your-jwt-secret-key');
```

### Email Configuration

PHPMailer settings are in [mail/submit-form.php](mail/submit-form.php). Configure SMTP settings:

```php
$mail->Host = 'smtp.gmail.com';
$mail->Port = 587;
$mail->Username = 'your-email@gmail.com';
$mail->Password = 'your-app-password';
```

## 🎛 CMS Administration

### Accessing the CMS

**URL:** `/admin/cms.php`

The CMS provides a complete content management interface with:

- **Dashboard** - Overview of content statistics
- **Categories** - Manage content types (Projects, Blogs, Hero Section, etc.)
- **Fields** - Define custom fields for each category
- **Records** - Create and edit content entries
- **Media Library** - Upload and manage files
- **Languages** - Multi-language content support
- **Users** - User management and authentication

### Authentication

The CMS uses two authentication methods:

1. **JWT Authentication** - For logged-in CMS users
   - Token stored in browser after login
   - Automatically included in requests

2. **API Key Authentication** - For external API access
   - Must include `X-API-TOKEN` header
   - Defined in `.env` file

### Content Management

#### Creating a Category

1. Navigate to **Categories** section
2. Click **"Add Category"**
3. Enter category name (e.g., "Projects", "Blog Posts")
4. Add custom fields:
   - **Field Name** - Identifier (e.g., "title", "description")
   - **Data Type** - Choose from available types
   - **Required** - Mark if mandatory
   - **Options** - Additional settings (e.g., max_chars for rich_paragraph)

#### Adding Content Records

1. Select a **Category**
2. Click **"Add Record"**
3. Fill in all fields
4. Add translations (if multi-language enabled)
5. Generate slug for SEO-friendly URLs
6. Save record

#### Rich Text Editor

The CMS includes a custom rich text editor with:
- **Character limit enforcement** - Set maximum characters per field
- **Formatting tools** - Bold, italic, lists, links
- **Real-time character counter** - Visual feedback
- **Paste protection** - Prevents exceeding limits

Configuration in [admin/cms.php](admin/cms.php#L9993-L10072):
```javascript
fieldData.field_options = {
    max_chars: 500  // Set character limit
};
```

## 📡 API Documentation

### Base URL
```
http://localhost/PORTFOLIO.MAXMIND.MA-main/PORTFOLIO.MAXMIND.MA-main/admin/cms.php
```

### Authentication

All API requests require authentication via one of:

1. **API Key Header**
   ```
   X-API-TOKEN: your_api_key_here
   ```

2. **Bearer Token** (for authenticated users)
   ```
   Authorization: Bearer <jwt_token>
   ```

### Endpoints

#### Authentication

**POST** `/login`
```json
{
  "email": "user@example.com",
  "password": "password"
}
```

Response:
```json
{
  "success": true,
  "token": "jwt_token_here",
  "user": {
    "id": 1,
    "name": "User Name",
    "email": "user@example.com"
  }
}
```

#### Categories

**GET** `/categories` - List all categories

**GET** `/categories/{id}` - Get specific category

**POST** `/categories` - Create new category
```json
{
  "name": "Category Name"
}
```

#### Records

**GET** `/records/{category_id}` - Get all records for category

**GET** `/Records/{record_id}` - Get specific record with all field values

**POST** `/records` - Create new record
```json
{
  "category_id": 1,
  "slug": "my-record-slug",
  "values": {
    "field_id_1": "value1",
    "field_id_2": "value2"
  }
}
```

**PUT** `/records/{record_id}` - Update record

**DELETE** `/records/{record_id}` - Delete record

#### Fields

**GET** `/fields/{category_id}` - Get all fields for category

**POST** `/fields` - Create new field
```json
{
  "category_id": 1,
  "name": "field_name",
  "data_type": "text",
  "is_required": 1,
  "field_options": "{\"max_chars\": 100}"
}
```

#### File Uploads

**POST** `/upload` - Upload file (multipart/form-data)

**GET** `/files` - List all uploaded files

#### Special Endpoints

**GET** `/hero-section` - Get hero section data

**GET** `/searchable-fields/{category_id}` - Get searchable fields

**GET** `/schema` - Get complete database schema

### API Request Example

```javascript
const API_KEY = '2250c47a4397a41fe5ff8f2c2a92b240ddf46aaeded8ffddb4f29f02ffef8720';

fetch('http://localhost/admin/cms.php/categories', {
  headers: {
    'Content-Type': 'application/json',
    'X-API-TOKEN': API_KEY
  }
})
.then(res => res.json())
.then(data => console.log(data));
```

## 🌐 Frontend Pages

### Homepage (`index.php`)

Features:
- Dynamic hero carousel with CMS-managed images
- Services section with animated timeline
- Portfolio showcase
- Blog highlights
- Contact form
- GSAP animations on scroll

Key files:
- [index.php](index.php) - Main page structure
- [js/index.js](js/index.js) - Animations and interactions
- [css/index.css](css/index.css) - Styles

### Portfolio Page (`portfolio.php`)

Displays individual portfolio items with:
- Project title and description
- Image gallery
- Technologies used
- Client information
- Related projects

### Projects Listing (`projectsType.php`)

Filterable project gallery:
- Filter by project type (Web, Mobile, E-commerce, UI/UX)
- Search functionality
- Pagination
- Responsive grid layout

### Blog Pages

**Blog Listing** (`blogs.php`)
- Grid layout of blog posts
- Category filtering
- Featured articles
- Read more links

**Single Blog** (`blog.php`)
- Full article content with rich text
- Meta tags for SEO
- Social sharing
- Related articles

## 🔒 Security

### API Protection

All API endpoints are protected with API key validation in [admin/cms.php](admin/cms.php#L36-L57):

```php
if ($isApiRequest) {
    $apiToken = $_GET['token'] ?? $_POST['token'] ?? $_SERVER['HTTP_X_API_TOKEN'] ?? '';
    $authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? '';

    $hasBearerToken = preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches);
    $hasValidApiKey = ($apiToken === $API_TOKEN);

    if (!$hasValidApiKey && !$hasBearerToken) {
        http_response_code(403);
        echo json_encode(["error" => "Unauthorized", "message" => "Invalid API key."]);
        exit();
    }
}
```

### Frontend vs API Requests

The system intelligently differentiates between:
- **Frontend requests** - Direct page access (no auth required)
- **API requests** - Programmatic data access (auth required)

Detection logic in [admin/cms.php](admin/cms.php#L19-L34).

### Input Sanitization

Contact form includes protection against:
- SQL injection
- XSS attacks
- HTML entity encoding

Implementation in [mail/submit-form.php](mail/submit-form.php#L17-L36).

### Password Security

- Passwords hashed with `password_hash()` (bcrypt)
- JWT tokens for session management
- Secure token validation

## 🗄 Database Schema

### Tables Overview

**users**
- Stores CMS administrator accounts
- Columns: `id`, `name`, `email`, `password`, `created_at`

**categories**
- Content type definitions (Projects, Blogs, etc.)
- Columns: `id`, `user_id`, `name`, `created_at`

**category_fields**
- Custom field definitions for each category
- Columns: `id`, `category_id`, `name`, `data_type`, `is_required`, `default_value`, `field_options`, `foreign_key_table`, `foreign_key_column`, `created_at`

**category_records**
- Actual content entries
- Columns: `id`, `category_id`, `slug`, `created_at`

**category_record_values**
- Field values for each record
- Columns: `id`, `record_id`, `field_id`, `value`

**uploaded_files**
- Media library files
- Columns: `id`, `filename`, `original_name`, `file_type`, `file_size`, `upload_path`, `width`, `height`, `tags`, `slug`, `original_url`, `created_at`

**languages**
- Supported languages for translations
- Columns: `id`, `code`, `name`, `is_default`, `created_at`

**record_translations**
- Multi-language content versions
- Columns: `id`, `record_id`, `language_id`, `field_id`, `value`, `created_at`, `updated_at`

Schema definition: [admin/cms.php](admin/cms.php#L143-L228)

### Field Types Supported

- `text` - Single-line text input
- `textarea` - Multi-line text
- `number` - Numeric values
- `email` - Email addresses
- `url` - Website URLs
- `date` - Date picker
- `checkbox` - Boolean values
- `image` - Image uploads
- `file` - File uploads
- `media` - Media gallery
- `rich_paragraph` - Rich text editor with character limits
- `foreign_key` - Relations to other tables
- `array` - Multiple values
- `json` - JSON data
- `link` - Hyperlinks

## 📧 Contact Form

### Features
- Client-side validation
- Server-side sanitization
- PHPMailer email delivery
- Google Sheets integration for lead tracking
- AJAX submission (no page reload)

### Implementation

Form processor: [mail/submit-form.php](mail/submit-form.php)

Frontend handler: [js/index.js](js/index.js) (initializeContactForm)

### Google Sheets Integration

Contact form submissions are automatically saved to Google Sheets using the Google Sheets API.

Credentials file: `mail/glorypushtosheets-740d2d7d6e8a.json`

## 🔍 SEO Optimization

### Features Implemented

1. **Meta Tags**
   - Open Graph tags for social media sharing
   - Twitter Card meta tags
   - Structured data markup

2. **Dynamic Sitemaps**
   - XML sitemap generation: [sitemap.php](sitemap.php)
   - Robots.txt configuration: [robots.txt](robots.txt)

3. **SEO-Friendly URLs**
   - Slug generation for all content
   - Clean URL structure
   - Canonical URLs

4. **Performance Optimization**
   - Lazy loading images
   - Minified CSS/JS (production)
   - GSAP animation optimization

5. **Multi-language Support**
   - French and Arabic meta tags
   - Language-specific content
   - Hreflang tags

### Sitemap Structure

The dynamic sitemap includes:
- Homepage
- All portfolio items
- All blog posts
- Static pages (mentions légales, privacy policy)
- Project type pages

## 🎨 Animations & Interactions

### GSAP ScrollTrigger

Animations implemented in [js/index.js](js/index.js):

- **Hero Section** - Fade-in on load
- **Services Section** - Scroll-triggered reveals
- **Timeline Section** - Sequential appearance (lines 652-671)
- **Portfolio Carousel** - Smooth transitions
- **Process Steps** - Stagger animations

### Timeline Visibility Fix

Recent fix for "Small Wins One Step at a Time" section:
- **Issue:** Timeline items only visible on hover
- **Solution:** Removed conflicting GSAP animation overriding CSS opacity
- **Implementation:** [js/index.js:652-653](js/index.js#L652-L653)

```javascript
// Process steps are visible by default (no animation needed)
// The CSS already handles the visibility
```

CSS configuration: [css/index.css:943-952](css/index.css#L943-L952)

## 🤝 Contributing

This is a private portfolio project. For inquiries, contact MaxMind.

## 📄 License

Proprietary - MaxMind © 2024

## 📞 Contact

**MaxMind - Innovation Technology**

- Website: [maxmind.ma](https://maxmind.ma)
- Portfolio: [portfolio.maxmind.ma](https://portfolio.maxmind.ma)
- Email: contact@maxmind.ma
- Location: Casablanca, Morocco

---

**Built with ❤️ by MaxMind Team**
