<?php

// Set unlimited upload limits
ini_set('upload_max_filesize', '0');
ini_set('post_max_size', '0');
ini_set('max_execution_time', '0');
ini_set('max_input_time', '0');
ini_set('memory_limit', '-1');

define('DB_TYPE', 'sqlite');
define('DB_NAME', 'cms.db');
define('JWT_SECRET', 'eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJ1c2VyX2lkIjoxMjMsImVtYWlsIjoiam9obmRvZUBleGFtcGxlLmNvbSIsImV4cCI6MTczNTA1MTE3NSwidHlwZSI6InByb3h5X3Nob3J0X2xpdmVkIn0.rU-S5f_Wc0JvF8G9g2Y-6F_r1L7L9T-jN8V7Q1c_qFk');

// Handle API requests vs Frontend display
$requestUri = $_SERVER['REQUEST_URI'];
$scriptName = $_SERVER['SCRIPT_NAME'];
$pathInfo = str_replace($scriptName, '', $requestUri);
$pathInfo = parse_url($pathInfo, PHP_URL_PATH);

// Clean up pathInfo - if empty or just query params, treat as frontend
if (empty($pathInfo) || $pathInfo === '/' || $pathInfo === '') {
    $pathInfo = '';
}
$API_TOKEN = "2250c47a4397a41fe5ff8f2c2a92b240ddf46aaeded8ffddb4f29f02ffef8720";

// Check if this is an API request (must match specific API endpoints)
$isApiRequest = (
    ($_SERVER['REQUEST_METHOD'] !== 'GET' && !empty($pathInfo)) ||
    strpos($pathInfo, '/login') === 0 ||
    strpos($pathInfo, '/categories') === 0 ||
    strpos($pathInfo, '/fields') === 0 ||
    strpos($pathInfo, '/records') === 0 ||
    strpos($pathInfo, '/Records') === 0 ||
    strpos($pathInfo, '/searchable-fields') === 0 ||
    strpos($pathInfo, '/files') === 0 ||
    strpos($pathInfo, '/upload') === 0 ||
    strpos($pathInfo, '/upload-config') === 0 ||
    strpos($pathInfo, '/schema') === 0 ||
    strpos($pathInfo, '/tags') === 0 ||
    strpos($pathInfo, '/languages') === 0
);

if ($isApiRequest) {
    // API Mode - Handle backend requests

    // Validate API key OR Bearer token for all API requests
    $apiToken = $_GET['token'] ?? $_POST['token'] ?? $_SERVER['HTTP_X_API_TOKEN'] ?? '';
    $authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['REDIRECT_HTTP_AUTHORIZATION'] ?? '';

    // Check if Authorization header exists (for Bearer token from authenticated CMS users)
    $hasBearerToken = preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches);

    // Allow access if either API key is valid OR Bearer token is present (will be validated later by JWT functions)
    // $hasValidApiKey = ($apiToken === $API_TOKEN);

    // if (!$hasValidApiKey && !$hasBearerToken) {
    //     http_response_code(403);
    //     header('Content-Type: application/json');
    //     echo json_encode([
    //         "error" => "Unauthorized",
    //         "message" => "Invalid API key."
    //     ]);
    //     exit();
    // }

    // Aggressive error suppression for upload requests
    if (strpos($pathInfo, '/upload') === 0) {
        ini_set('display_errors', 0);
        ini_set('log_errors', 1);
        error_reporting(0);

        $bufferClearCount = 0;
        $maxBufferClears = 10; 
        while (ob_get_level() && $bufferClearCount < $maxBufferClears) {
            ob_end_clean();
            $bufferClearCount++;
        }

        // Start fresh output buffer
        ob_start();
    }

    header('Content-Type: application/json');
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');

    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit;
    }

    try {
        handleApiRequest();
    } catch (Exception $e) {
        // Ensure we only output JSON even if there's an error
        if (strpos($pathInfo, '/upload') === 0) {
            ob_clean();
        }
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        exit;
    }
} else {
    // Frontend Mode - Display HTML interface
    displayFrontend();
}

// Database Connection and Tables
function getDbConnection() {
    static $pdo = null;
    if ($pdo === null) {
        try {
            // Create uploads directory if it doesn't exist
            $uploadsDir = __DIR__ . '/uploads';
            if (!is_dir($uploadsDir)) {
                if (!mkdir($uploadsDir, 0755, true)) {
                    error_log("Failed to create uploads directory: " . $uploadsDir);
                } else {
                    error_log("Created uploads directory: " . $uploadsDir);
                }
            }

            // Ensure the database file is created with proper path
            $dbPath = __DIR__ . '/' . DB_NAME;
            $dbExists = file_exists($dbPath);

            if (!$dbExists) {
                error_log("Creating new database at: " . $dbPath);
            }

            $pdo = new PDO('sqlite:' . $dbPath);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            createTables();
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            jsonResponse(false, "Database connection failed");
        }
    }
    return $pdo;
}

function createTables() {
    $pdo = getDbConnection();

    // Enable foreign key constraints for SQLite
    $pdo->exec("PRAGMA foreign_keys = ON");

    $sql = "
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL CHECK(length(name) > 0),
            email TEXT UNIQUE NOT NULL CHECK(email LIKE '%@%'),
            password TEXT NOT NULL CHECK(length(password) > 0),
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        );

        CREATE TABLE IF NOT EXISTS categories (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER NOT NULL,
            name TEXT NOT NULL CHECK(length(name) > 0),
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
        );

        CREATE TABLE IF NOT EXISTS category_fields (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            category_id INTEGER NOT NULL,
            name TEXT NOT NULL CHECK(length(name) > 0),
            data_type TEXT NOT NULL CHECK(data_type IN ('text', 'textarea', 'number', 'email', 'url', 'date', 'checkbox', 'image', 'file', 'media', 'rich_paragraph', 'foreign_key', 'array', 'json', 'link')),
            is_required INTEGER DEFAULT 0 CHECK(is_required IN (0, 1)),
            default_value TEXT,
            field_options TEXT,
            foreign_key_table TEXT,
            foreign_key_column TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE,
            UNIQUE(category_id, name)
        );

        CREATE TABLE IF NOT EXISTS category_records (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            category_id INTEGER NOT NULL,
            slug TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE
        );

        CREATE TABLE IF NOT EXISTS category_record_values (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            record_id INTEGER NOT NULL,
            field_id INTEGER NOT NULL,
            value TEXT,
            FOREIGN KEY (record_id) REFERENCES category_records(id) ON DELETE CASCADE,
            FOREIGN KEY (field_id) REFERENCES category_fields(id) ON DELETE CASCADE,
            UNIQUE(record_id, field_id)
        );

        CREATE TABLE IF NOT EXISTS uploaded_files (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            filename TEXT NOT NULL CHECK(length(filename) > 0),
            original_name TEXT NOT NULL CHECK(length(original_name) > 0),
            file_type TEXT NOT NULL CHECK(length(file_type) > 0),
            file_size INTEGER NOT NULL CHECK(file_size >= 0),
            upload_path TEXT NOT NULL CHECK(length(upload_path) > 0),
            width INTEGER DEFAULT NULL CHECK(width IS NULL OR width > 0),
            height INTEGER DEFAULT NULL CHECK(height IS NULL OR height > 0),
            tags TEXT DEFAULT NULL,
            slug TEXT DEFAULT NULL,
            original_url TEXT DEFAULT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        );

        CREATE TABLE IF NOT EXISTS languages (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            code TEXT NOT NULL UNIQUE CHECK(length(code) >= 2 AND length(code) <= 10),
            name TEXT NOT NULL CHECK(length(name) > 0),
            text_direction TEXT DEFAULT 'ltr' CHECK(text_direction IN ('ltr', 'rtl')),
            is_active INTEGER DEFAULT 1 CHECK(is_active IN (0, 1)),
            is_default INTEGER DEFAULT 0 CHECK(is_default IN (0, 1)),
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
        );

        CREATE TABLE IF NOT EXISTS record_translations (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            record_id INTEGER NOT NULL,
            language_id INTEGER NOT NULL,
            field_id INTEGER NOT NULL,
            value TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (record_id) REFERENCES category_records(id) ON DELETE CASCADE,
            FOREIGN KEY (language_id) REFERENCES languages(id) ON DELETE CASCADE,
            FOREIGN KEY (field_id) REFERENCES category_fields(id) ON DELETE CASCADE,
            UNIQUE(record_id, language_id, field_id)
        );
    ";

    try {
        $pdo->exec($sql);

        // Create indexes for better performance
        $indexes = [
            "CREATE INDEX IF NOT EXISTS idx_categories_user_id ON categories(user_id)",
            "CREATE INDEX IF NOT EXISTS idx_category_fields_category_id ON category_fields(category_id)",
            "CREATE INDEX IF NOT EXISTS idx_category_records_category_id ON category_records(category_id)",
            "CREATE UNIQUE INDEX IF NOT EXISTS idx_category_records_slug ON category_records(slug) WHERE slug IS NOT NULL",
            "CREATE INDEX IF NOT EXISTS idx_category_record_values_record_id ON category_record_values(record_id)",
            "CREATE INDEX IF NOT EXISTS idx_category_record_values_field_id ON category_record_values(field_id)",
            "CREATE INDEX IF NOT EXISTS idx_uploaded_files_file_type ON uploaded_files(file_type)",
            "CREATE INDEX IF NOT EXISTS idx_uploaded_files_created_at ON uploaded_files(created_at DESC)",
            "CREATE INDEX IF NOT EXISTS idx_record_translations_record_id ON record_translations(record_id)",
            "CREATE INDEX IF NOT EXISTS idx_record_translations_language_id ON record_translations(language_id)",
            "CREATE INDEX IF NOT EXISTS idx_languages_code ON languages(code)",
            "CREATE INDEX IF NOT EXISTS idx_languages_is_default ON languages(is_default)"
        ];

        foreach ($indexes as $index) {
            try {
                $pdo->exec($index);
            } catch (Exception $e) {
                error_log("Index creation warning: " . $e->getMessage());
            }
        }

        // Perform controlled migrations for existing installations
        performDatabaseMigrations($pdo);

        // Create default admin user for single-user CMS with proper password handling
        createDefaultUser($pdo);

    } catch (PDOException $e) {
        error_log("Table creation failed: " . $e->getMessage());
        jsonResponse(false, "Database initialization failed: " . $e->getMessage());
    }
}

function performDatabaseMigrations($pdo) {
    $migrations = [
        "ALTER TABLE category_records ADD COLUMN slug TEXT",
        "ALTER TABLE category_fields ADD COLUMN foreign_key_table TEXT",
        "ALTER TABLE category_fields ADD COLUMN foreign_key_column TEXT",
        "ALTER TABLE uploaded_files ADD COLUMN width INTEGER DEFAULT NULL",
        "ALTER TABLE uploaded_files ADD COLUMN height INTEGER DEFAULT NULL",
        "ALTER TABLE uploaded_files ADD COLUMN tags TEXT DEFAULT NULL",
        "ALTER TABLE uploaded_files ADD COLUMN original_url TEXT DEFAULT NULL",
        "ALTER TABLE uploaded_files ADD COLUMN slug TEXT DEFAULT NULL",
        "ALTER TABLE category_fields ADD COLUMN is_translatable INTEGER DEFAULT 0",
        "ALTER TABLE languages ADD COLUMN text_direction TEXT DEFAULT 'ltr'",
        "ALTER TABLE languages ADD COLUMN is_active INTEGER DEFAULT 1",
        "ALTER TABLE languages ADD COLUMN updated_at DATETIME"
    ];

    foreach ($migrations as $migration) {
        try {
            $pdo->exec($migration);
        } catch (Exception $e) {
            // Migration already applied or failed - this is expected for existing installations
        }
    }
}

function createDefaultUser($pdo) {
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM users");
        $stmt->execute();
        $userCount = $stmt->fetchColumn();

        if ($userCount == 0) {
            // Use a more secure default password (still needs to be changed by user)
            $defaultPassword = password_hash('admin123', PASSWORD_DEFAULT);
            $stmt = $pdo->prepare("INSERT INTO users (id, name, email, password) VALUES (1, 'Admin', 'admin@localhost', ?)");
            $stmt->execute([$defaultPassword]);
            error_log("Default admin user created with password 'admin123' - CHANGE THIS PASSWORD IMMEDIATELY");
        }
    } catch (Exception $e) {
        error_log("Default user creation failed: " . $e->getMessage());
    }
}

// API Request Handler
function handleApiRequest() {
    $method = $_SERVER['REQUEST_METHOD'];
    $requestUri = $_SERVER['REQUEST_URI'];
    $scriptName = $_SERVER['SCRIPT_NAME'];
    $path = str_replace($scriptName, '', $requestUri);
    $path = parse_url($path, PHP_URL_PATH);
    
    // Public routes
    if ($method === 'POST' && $path === '/login') {
        return loginUser(getJsonInput());
    }

    // Authentication handling
    $auth = getAuthenticationInfo($method, $path);

    // Validate authentication for protected routes
    if (requiresAuthentication($method, $path) && !$auth) {
        jsonResponse(false, "Authentication required", 401);
    }
    
    // Categories
    if ($method === 'GET' && $path === '/categories') {
        return getCategories($auth);
    }
    if ($method === 'POST' && $path === '/categories') {
        return createCategory(getJsonInput(), $auth);
    }
    if ($method === 'PUT' && preg_match('#^/categories/(\d+)$#', $path, $matches)) {
        return updateCategory($matches[1], getJsonInput(), $auth);
    }
    if ($method === 'DELETE' && preg_match('#^/categories/(\d+)$#', $path, $matches)) {
        return deleteCategory($matches[1], $auth);
    }
    
    // Fields
    if ($method === 'GET' && $path === '/fields') {
        $categoryId = $_GET['category_id'] ?? null;
        if ($categoryId) return getFields($categoryId, $auth);
    }
    if ($method === 'POST' && $path === '/fields') {
        return createField(getJsonInput(), $auth);
    }
    if ($method === 'GET' && preg_match('#^/fields/(\d+)$#', $path, $matches)) {
        return getField($matches[1], $auth);
    }
    if ($method === 'PUT' && preg_match('#^/fields/(\d+)$#', $path, $matches)) {
        return updateField($matches[1], getJsonInput(), $auth);
    }
    if ($method === 'DELETE' && preg_match('#^/fields/(\d+)$#', $path, $matches)) {
        return deleteField($matches[1], $auth);
    }
    
    // Records
    if ($method === 'GET' && $path === '/records') {
        $categoryId = $_GET['category_id'] ?? null;
        if ($categoryId) return getRecords($categoryId, $auth);
    }
    if ($method === 'GET' && $path === '/Records') {
        $categoryId = $_GET['category_id'] ?? null;
        if ($categoryId) return getRecords1($categoryId, $auth);
    }
    if ($method === 'GET' && $path === '/searchable-fields') {
        $categoryId = $_GET['category_id'] ?? null;
        if ($categoryId) return getSearchableFields($categoryId, $auth);
    }
    if ($method === 'POST' && $path === '/records') {
        return createRecord(getJsonInput(), $auth);
    }
    if ($method === 'PUT' && preg_match('#^/records/(\d+)$#', $path, $matches)) {
        return updateRecord($matches[1], getJsonInput(), $auth);
    }
    if ($method === 'DELETE' && preg_match('#^/records/(\d+)$#', $path, $matches)) {
        return deleteRecord($matches[1], $auth);
    }
    if ($method === 'GET' && $path === '/records/export') {
        $categoryId = $_GET['category_id'] ?? null;
        if ($categoryId) return exportDatabase($categoryId, $auth);
        jsonResponse(false, "Category ID is required", 400);
    }

    // Files
    if ($method === 'GET' && $path === '/files') {
        return getUploadedFiles($auth);
    }
    if ($method === 'POST' && $path === '/upload') {
        // Debug all available data
        error_log("Upload Debug - POST: " . print_r($_POST, true));
        error_log("Upload Debug - FILES: " . print_r($_FILES, true));
        error_log("Upload Debug - Content Type: " . ($_SERVER['CONTENT_TYPE'] ?? 'unknown'));
        error_log("Upload Debug - Content Length: " . ($_SERVER['CONTENT_LENGTH'] ?? 'unknown'));

        // Check if any files were uploaded
        if (empty($_FILES)) {
            jsonResponse(false, "No files received. Check PHP upload_max_filesize and post_max_size settings.");
        }

        // Check if our specific file field exists
        if (!isset($_FILES['file'])) {
            $availableFiles = array_keys($_FILES);
            jsonResponse(false, "File field 'file' not found. Available fields: " . implode(', ', $availableFiles));
        }

        // Log upload attempt for debugging
        error_log("Upload attempt - File: " . ($_FILES['file']['name'] ?? 'unknown') .
                  ", Size: " . ($_FILES['file']['size'] ?? 'unknown') .
                  ", Type: " . ($_FILES['file']['type'] ?? 'unknown') .
                  ", Error: " . ($_FILES['file']['error'] ?? 'unknown'));

        return uploadFile($_FILES['file'], $auth);
    }
    if ($method === 'POST' && $path === '/upload-video-url') {
        return uploadVideoUrl(getJsonInput(), $auth);
    }
    if ($method === 'POST' && $path === '/upload-media-url') {
        return uploadMediaUrl(getJsonInput(), $auth);
    }
    if ($method === 'DELETE' && preg_match('#^/files/(\d+)$#', $path, $matches)) {
        return deleteFile($matches[1], $auth);
    }
    if ($method === 'GET' && $path === '/upload-config') {
        return getUploadConfig($auth);
    }

    // Database Schema
    if ($method === 'GET' && $path === '/schema') {
        return getDatabaseSchema($auth);
    }

    // Languages
    if ($method === 'GET' && $path === '/languages') {
        return getLanguages($auth);
    }
    if ($method === 'POST' && $path === '/languages') {
        return createLanguage(getJsonInput(), $auth);
    }
    if ($method === 'PUT' && preg_match('#^/languages/(\d+)$#', $path, $matches)) {
        return updateLanguage($matches[1], getJsonInput(), $auth);
    }
    if ($method === 'DELETE' && preg_match('#^/languages/(\d+)$#', $path, $matches)) {
        return deleteLanguage($matches[1], $auth);
    }

    // Translations
    if ($method === 'GET' && preg_match('#^/records/(\d+)/translations$#', $path, $matches)) {
        return getRecordTranslations($matches[1], $auth);
    }
    if ($method === 'POST' && preg_match('#^/records/(\d+)/translations$#', $path, $matches)) {
        return saveRecordTranslations($matches[1], getJsonInput(), $auth);
    }

    http_response_code(404);
    jsonResponse(false, "Endpoint not found");
}

// JWT Functions
function jwtEncode($payload, $secret) {
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode($payload);
    
    $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
    $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
    
    $signature = hash_hmac('sha256', $base64Header . "." . $base64Payload, $secret, true);
    $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
    
    return $base64Header . "." . $base64Payload . "." . $base64Signature;
}

function jwtDecode($jwt, $secret) {
    $parts = explode('.', $jwt);
    if (count($parts) !== 3) {
        return false;
    }
    
    list($base64Header, $base64Payload, $base64Signature) = $parts;
    
    $signature = base64_decode(str_replace(['-', '_'], ['+', '/'], $base64Signature));
    $expectedSignature = hash_hmac('sha256', $base64Header . "." . $base64Payload, $secret, true);
    
    if (!hash_equals($signature, $expectedSignature)) {
        return false;
    }
    
    $payload = json_decode(base64_decode(str_replace(['-', '_'], ['+', '/'], $base64Payload)), true);
    
    if (isset($payload['exp']) && $payload['exp'] < time()) {
        return false;
    }
    
    return $payload;
}

function authenticate() {
    $headers = getallheaders();
    $authHeader = $headers['Authorization'] ?? '';

    if (!preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
        return null;
    }

    $token = $matches[1];
    $payload = jwtDecode($token, JWT_SECRET);

    if (!$payload) {
        return null;
    }

    return $payload;
}

function getAuthenticationInfo($method, $path) {
    // For public routes, no authentication needed
    if (isPublicRoute($method, $path)) {
        return null;
    }

    // Try to authenticate with JWT token
    $auth = authenticate();

    // For single-user CMS mode, if no valid JWT and we're accessing data operations,
    // fall back to default user (this maintains backward compatibility)
    if (!$auth && isSingleUserCmsMode()) {
        $auth = ['user_id' => 1, 'email' => 'admin@localhost', 'is_default' => true];
    }

    return $auth;
}

function requiresAuthentication($method, $path) {
    // Define routes that require authentication
    $protectedRoutes = [
        'POST:/categories', 'PUT:/categories/', 'DELETE:/categories/',
        'POST:/fields', 'PUT:/fields/', 'DELETE:/fields/',
        'POST:/records', 'PUT:/records/', 'DELETE:/records/',
        'POST:/upload', 'POST:/upload-video-url', 'DELETE:/files/',
        'POST:/languages', 'PUT:/languages/', 'DELETE:/languages/'
    ];

    foreach ($protectedRoutes as $route) {
        $parts = explode(':', $route);
        $routeMethod = $parts[0];
        $routePath = $parts[1];

        if ($method === $routeMethod &&
            ($path === $routePath || ($routePath[-1] === '/' && strpos($path, $routePath) === 0))) {
            return true;
        }
    }

    return false;
}

function isPublicRoute($method, $path) {
    $publicRoutes = [
        'POST:/login',
        'GET:/categories',
        'GET:/fields',
        'GET:/records',
        'GET:/Records',
        'GET:/searchable-fields',
        'GET:/files',
        'GET:/tags',
        'GET:/languages',
    ];

    foreach ($publicRoutes as $route) {
        $parts = explode(':', $route);
        $routeMethod = $parts[0];
        $routePath = $parts[1];

        if ($method === $routeMethod && $path === $routePath) {
            return true;
        }
    }

    return false;
}

function isSingleUserCmsMode() {
    // For now, this CMS operates in single-user mode
    // This could be made configurable in the future
    return true;
}

// User Functions

function loginUser($data) {
    validateRequired($data, ['email', 'password']);
    
    $pdo = getDbConnection();
    $stmt = $pdo->prepare("SELECT id, name, email, password FROM users WHERE email = ?");
    $stmt->execute([$data['email']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user || !password_verify($data['password'], $user['password'])) {
        jsonResponse(false, "Invalid email or password", 401);
    }
    
    // Generate JWT token
    $payload = [
        'user_id' => $user['id'],
        'email' => $user['email'],
        'exp' => time() + (24 * 60 * 60) // 24 hours
    ];
    
    $token = jwtEncode($payload, JWT_SECRET);
    
    jsonResponse(true, [
        'user' => [
            'id' => $user['id'],
            'name' => $user['name'],
            'email' => $user['email']
        ],
        'token' => $token
    ]);
}

// Category Functions
function getCategories($auth) {
    $pdo = getDbConnection();
    $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
    $perPage = isset($_GET['per_page']) ? max(1, min(100, (int)$_GET['per_page'])) : 8;
    $offset = ($page - 1) * $perPage;

    $searchQuery = isset($_GET['q']) ? trim($_GET['q']) : '';
    $whereClause = "";
    $params = [];

    // Single-user CMS - no user filtering needed

    if (!empty($searchQuery)) {
        if ($whereClause) {
            $whereClause .= " AND c.name LIKE ?";
        } else {
            $whereClause = "WHERE c.name LIKE ?";
        }
        $params[] = "%{$searchQuery}%";
    }
    
    // Get total count
    $countStmt = $pdo->prepare("SELECT COUNT(*) as total FROM categories c {$whereClause}");
    $countStmt->execute($params);
    $total = (int)$countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Get paginated results
    $stmt = $pdo->prepare("
        SELECT 
            c.id, c.name, c.created_at,
            COUNT(DISTINCT f.id) as field_count,
            COUNT(DISTINCT r.id) as record_count
        FROM categories c
        LEFT JOIN category_fields f ON c.id = f.category_id
        LEFT JOIN category_records r ON c.id = r.category_id
        {$whereClause}
        GROUP BY c.id, c.name, c.created_at
        ORDER BY c.created_at DESC
        LIMIT ? OFFSET ?
    ");
    
    $params[] = $perPage;
    $params[] = $offset;
    $stmt->execute($params);
    $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    jsonResponse(true, [
        'items' => $categories,
        'pagination' => [
            'page' => $page,
            'per_page' => $perPage,
            'total' => $total
        ]
    ]);
}

function createCategory($data, $auth) {
    validateRequired($data, ['name']);

    $pdo = getDbConnection();

    // Check if category name already exists
    $stmt = $pdo->prepare("SELECT id FROM categories WHERE name = ?");
    $stmt->execute([$data['name']]);
    if ($stmt->fetch()) {
        jsonResponse(false, "Category name must be unique. A category with this name already exists.");
    }

    $pdo->beginTransaction();

    try {
        // Create category
        $stmt = $pdo->prepare("INSERT INTO categories (user_id, name) VALUES (?, ?)");
        $stmt->execute([1, $data['name']]);
        $categoryId = $pdo->lastInsertId();
        
        // Create first field if provided
        if (isset($data['first_field'])) {
            $fieldData = $data['first_field'];
            validateRequired($fieldData, ['name', 'data_type']);
            
            $fieldOptions = null;
            if ($fieldData['data_type'] === 'select' && isset($fieldData['options'])) {
                $fieldOptions = json_encode($fieldData['options']);
            }
            
            $stmt = $pdo->prepare("INSERT INTO category_fields (category_id, name, data_type, is_required, default_value, field_options) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $categoryId,
                $fieldData['name'],
                $fieldData['data_type'],
                isset($fieldData['is_required']) ? (int)$fieldData['is_required'] : 0,
                $fieldData['default_value'] ?? null,
                $fieldOptions
            ]);
        }
        
        $pdo->commit();
        
        $category = [
            'id' => $categoryId,
            'name' => $data['name'],
            'user_id' => 1  // Default user ID for single-user CMS
        ];
        
        jsonResponse(true, $category);
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error in createCategory: " . $e->getMessage());
        jsonResponse(false, "Failed to create category: " . $e->getMessage());
    }
}

function updateCategory($categoryId, $data, $auth) {
    validateRequired($data, ['name']);

    $pdo = getDbConnection();

    // Check if category exists
    $stmt = $pdo->prepare("SELECT id FROM categories WHERE id = ?");
    $stmt->execute([$categoryId]);
    if (!$stmt->fetch()) {
        jsonResponse(false, "Category not found", 404);
    }

    // Check if category name is already taken by another category
    $stmt = $pdo->prepare("SELECT id FROM categories WHERE name = ? AND id != ?");
    $stmt->execute([$data['name'], $categoryId]);
    if ($stmt->fetch()) {
        jsonResponse(false, "Category name must be unique. A category with this name already exists.");
    }

    // Update category
    $stmt = $pdo->prepare("UPDATE categories SET name = ? WHERE id = ?");
    $stmt->execute([$data['name'], $categoryId]);
    
    $category = [
        'id' => $categoryId,
        'name' => $data['name'],
        'user_id' => 1
    ];
    
    jsonResponse(true, $category);
}

function deleteCategory($categoryId, $auth) {
    $pdo = getDbConnection();

    $stmt = $pdo->prepare("SELECT id FROM categories WHERE id = ?");
    $stmt->execute([$categoryId]);
    if (!$stmt->fetch()) {
        jsonResponse(false, "Category not found", 404);
    }

    // Check if category has any records
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM category_records WHERE category_id = ?");
    $stmt->execute([$categoryId]);
    $recordCount = $stmt->fetchColumn();

    if ($recordCount > 0) {
        jsonResponse(false, "Cannot delete category with existing records. This category contains {$recordCount} record(s). Please delete all records first.", 400);
    }

    $pdo->beginTransaction();
    try {
        // Delete record values
        $stmt = $pdo->prepare("
            DELETE FROM category_record_values 
            WHERE record_id IN (SELECT id FROM category_records WHERE category_id = ?)
        ");
        $stmt->execute([$categoryId]);
        
        // Delete records
        $stmt = $pdo->prepare("DELETE FROM category_records WHERE category_id = ?");
        $stmt->execute([$categoryId]);
        
        // Delete fields
        $stmt = $pdo->prepare("DELETE FROM category_fields WHERE category_id = ?");
        $stmt->execute([$categoryId]);
        
        // Delete category
        $stmt = $pdo->prepare("DELETE FROM categories WHERE id = ?");
        $stmt->execute([$categoryId]);
        
        $pdo->commit();
        jsonResponse(true, "Category deleted successfully");
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Delete category error: " . $e->getMessage());
        jsonResponse(false, "Failed to delete category: " . $e->getMessage());
    }
}

// Field validation function
function validateFieldDefaultValueBackend($fieldType, $defaultValue) {
    if (empty($defaultValue)) return false;
    
    switch ($fieldType) {
        case 'number':
            return is_numeric($defaultValue);
        case 'email':
            return filter_var($defaultValue, FILTER_VALIDATE_EMAIL) !== false;
        case 'url':
            return filter_var($defaultValue, FILTER_VALIDATE_URL) !== false;
        case 'date':
            return preg_match('/^\d{4}-\d{2}-\d{2}$/', $defaultValue) && strtotime($defaultValue) !== false;
        case 'checkbox':
            return in_array($defaultValue, ['true', 'false', '1', '0']);
        case 'link':
            // Validate that it's a proper JSON structure for link type
            $decoded = json_decode($defaultValue, true);
            return $decoded !== null &&
                   json_last_error() === JSON_ERROR_NONE &&
                   is_array($decoded) &&
                   isset($decoded['media']) &&
                   isset($decoded['text']);
        default:
            return true; // text, textarea, image, file types accept any string
    }
}

// Field Functions
function createField($data, $auth) {
    validateRequired($data, ['category_id', 'name', 'data_type']);

    // Validate default value based on field type (only if provided)
    if (!empty($data['default_value']) && !validateFieldDefaultValueBackend($data['data_type'], $data['default_value'])) {
        jsonResponse(false, "Invalid default value for field type");
    }

    $pdo = getDbConnection();

    // Ensure category exists
    $stmt = $pdo->prepare("SELECT id FROM categories WHERE id = ?");
    $stmt->execute([$data['category_id']]);
    if (!$stmt->fetch()) {
        jsonResponse(false, "Category not found", 404);
    }

    $fieldOptions = null;
    if ($data['data_type'] === 'select' && isset($data['options'])) {
        $fieldOptions = json_encode($data['options']);
    } elseif ($data['data_type'] === 'array' && isset($data['array_options'])) {
        $fieldOptions = json_encode($data['array_options']);
    } elseif ($data['data_type'] === 'json' && isset($data['json_options'])) {
        $fieldOptions = json_encode($data['json_options']);
    } elseif ($data['data_type'] === 'link' && isset($data['link_options'])) {
        $fieldOptions = json_encode($data['link_options']);
    } elseif ($data['data_type'] === 'rich_paragraph' && isset($data['field_options'])) {
        $fieldOptions = json_encode($data['field_options']);
    }

    // Validate foreign key fields
    $foreignKeyTable = null;
    $foreignKeyColumn = null;
    if ($data['data_type'] === 'foreign_key') {
        if (empty($data['foreign_key_table']) || empty($data['foreign_key_column'])) {
            jsonResponse(false, "Foreign key fields require target table and column");
        }
        $foreignKeyTable = $data['foreign_key_table'];
        $foreignKeyColumn = $data['foreign_key_column'];

        // Validate that target table exists
        $stmt = $pdo->prepare("SELECT id FROM categories WHERE name = ?");
        $stmt->execute([$foreignKeyTable]);
        if (!$stmt->fetch()) {
            jsonResponse(false, "Target table does not exist");
        }
    }

    // Handle is_required field more explicitly
    $isRequired = 0; // Default to optional
    if (isset($data['is_required'])) {
        // Convert various representations to 0 or 1
        $isRequired = in_array($data['is_required'], [true, 1, '1', 'true', 'on'], true) ? 1 : 0;
    }

    // Create field
    $stmt = $pdo->prepare("INSERT INTO category_fields (category_id, name, data_type, is_required, default_value, field_options, foreign_key_table, foreign_key_column) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([
        $data['category_id'],
        $data['name'],
        $data['data_type'],
        $isRequired,
        $data['default_value'] ?? null,
        $fieldOptions,
        $foreignKeyTable,
        $foreignKeyColumn
    ]);

    $fieldId = $pdo->lastInsertId();
    $field = [
        'id' => $fieldId,
        'category_id' => $data['category_id'],
        'name' => $data['name'],
        'data_type' => $data['data_type'],
        'is_required' => (bool)$isRequired,
        'default_value' => $data['default_value'] ?? null,
        'field_options' => $fieldOptions ? json_decode($fieldOptions, true) : null,
        'foreign_key_table' => $foreignKeyTable,
        'foreign_key_column' => $foreignKeyColumn
    ];

    jsonResponse(true, $field);
}

function getFields($categoryId, $auth) {
    $pdo = getDbConnection();

    // Get all fields with usage count
    $stmt = $pdo->prepare("
        SELECT
            f.id,
            f.name,
            f.data_type,
            f.is_required,
            f.default_value,
            f.field_options,
            f.foreign_key_table,
            f.foreign_key_column,
            COUNT(DISTINCT v.record_id) as usage_count
        FROM category_fields f
        LEFT JOIN category_record_values v ON f.id = v.field_id
        WHERE f.category_id = ?
        GROUP BY f.id, f.name, f.data_type, f.is_required, f.default_value, f.field_options, f.foreign_key_table, f.foreign_key_column
        ORDER BY f.created_at
    ");
    $stmt->execute([$categoryId]);
    $fields = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Convert is_required to proper boolean and decode field_options
    foreach ($fields as &$field) {
        $field['is_required'] = (bool)(int)$field['is_required'];
        $field['usage_count'] = (int)$field['usage_count'];
        if ($field['field_options']) {
            $field['field_options'] = json_decode($field['field_options'], true);
        }
    }

    jsonResponse(true, [
        'items' => $fields
    ]);
}

function getField($fieldId, $auth) {
    $pdo = getDbConnection();
    $stmt = $pdo->prepare("SELECT f.id, f.name, f.data_type, f.is_required, f.default_value, f.category_id, f.field_options, f.foreign_key_table, f.foreign_key_column FROM category_fields f WHERE f.id = ?");
    $stmt->execute([$fieldId]);
    $field = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$field) {
        jsonResponse(false, "Field not found", 404);
    }

    // Convert is_required to proper boolean and decode field_options
    $field['is_required'] = (bool)(int)$field['is_required'];
    if ($field['field_options']) {
        $field['field_options'] = json_decode($field['field_options'], true);
    }

    jsonResponse(true, $field);
}

function updateField($fieldId, $data, $auth) {
    validateRequired($data, ['name', 'data_type']);

    $pdo = getDbConnection();

    // Check if field exists
    $stmt = $pdo->prepare("SELECT id FROM category_fields WHERE id = ?");
    $stmt->execute([$fieldId]);
    if (!$stmt->fetch()) {
        jsonResponse(false, "Field not found", 404);
    }

    // Handle field options
    $fieldOptions = null;
    if ($data['data_type'] === 'select' && isset($data['options'])) {
        $fieldOptions = json_encode($data['options']);
    } elseif ($data['data_type'] === 'array' && isset($data['array_options'])) {
        $fieldOptions = json_encode($data['array_options']);
    } elseif ($data['data_type'] === 'json' && isset($data['json_options'])) {
        $fieldOptions = json_encode($data['json_options']);
    } elseif ($data['data_type'] === 'link' && isset($data['link_options'])) {
        $fieldOptions = json_encode($data['link_options']);
    } elseif ($data['data_type'] === 'rich_paragraph' && isset($data['field_options'])) {
        $fieldOptions = json_encode($data['field_options']);
    }

    // Validate foreign key fields
    $foreignKeyTable = null;
    $foreignKeyColumn = null;
    if ($data['data_type'] === 'foreign_key') {
        if (empty($data['foreign_key_table']) || empty($data['foreign_key_column'])) {
            jsonResponse(false, "Foreign key fields require target table and column");
        }
        $foreignKeyTable = $data['foreign_key_table'];
        $foreignKeyColumn = $data['foreign_key_column'];

        // Validate that target table exists
        $stmt = $pdo->prepare("SELECT id FROM categories WHERE name = ?");
        $stmt->execute([$foreignKeyTable]);
        if (!$stmt->fetch()) {
            jsonResponse(false, "Target table does not exist");
        }
    }

    // Handle is_required field more explicitly
    $isRequired = 0; // Default to optional
    if (isset($data['is_required'])) {
        // Convert various representations to 0 or 1
        $isRequired = in_array($data['is_required'], [true, 1, '1', 'true', 'on'], true) ? 1 : 0;
    }

    // Update field
    $stmt = $pdo->prepare("
        UPDATE category_fields
        SET name = ?, data_type = ?, is_required = ?, default_value = ?, field_options = ?, foreign_key_table = ?, foreign_key_column = ?
        WHERE id = ?
    ");
    $stmt->execute([
        $data['name'],
        $data['data_type'],
        $isRequired,
        $data['default_value'] ?? null,
        $fieldOptions,
        $foreignKeyTable,
        $foreignKeyColumn,
        $fieldId
    ]);

    $field = [
        'id' => $fieldId,
        'name' => $data['name'],
        'data_type' => $data['data_type'],
        'is_required' => (bool)$isRequired,
        'default_value' => $data['default_value'] ?? null,
        'field_options' => $fieldOptions ? json_decode($fieldOptions, true) : null,
        'foreign_key_table' => $foreignKeyTable,
        'foreign_key_column' => $foreignKeyColumn
    ];

    jsonResponse(true, $field);
}

function deleteField($fieldId, $auth) {
    $pdo = getDbConnection();

    $stmt = $pdo->prepare("SELECT id, name FROM category_fields WHERE id = ?");
    $stmt->execute([$fieldId]);
    $field = $stmt->fetch();
    if (!$field) {
        jsonResponse(false, "Field not found", 404);
    }

    // Check if field is used in any records
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM category_record_values WHERE field_id = ?");
    $stmt->execute([$fieldId]);
    $usageCount = $stmt->fetchColumn();

    if ($usageCount > 0) {
        jsonResponse(false, "Cannot delete field '{$field['name']}' because it is used in {$usageCount} record(s). Please remove the field from all records first.", 400);
    }

    // Delete field and its values
    $pdo->beginTransaction();
    try {
        // Delete field values
        $stmt = $pdo->prepare("DELETE FROM category_record_values WHERE field_id = ?");
        $stmt->execute([$fieldId]);
        
        // Delete field
        $stmt = $pdo->prepare("DELETE FROM category_fields WHERE id = ?");
        $stmt->execute([$fieldId]);
        
        $pdo->commit();
        jsonResponse(true, "Field deleted successfully");
    } catch (Exception $e) {
        $pdo->rollBack();
        jsonResponse(false, "Failed to delete field");
    }
}

// Record Functions
function getRecords($categoryId, $auth) {
    $pdo = getDbConnection();
    $searchParams = parseSearchParameters();

    // Validate category exists
    $category = getCategoryInfo($pdo, $categoryId);
    if (!$category) {
        jsonResponse(false, "Category not found", 404);
    }

    // Build optimized search query
    $searchQuery = buildOptimizedSearchQuery($categoryId, $searchParams);

    // Get total count efficiently
    $total = getSearchResultCount($pdo, $searchQuery, $searchParams['params']);

    if ($total === 0) {
        jsonResponse(true, [
            'items' => [],
            'pagination' => [
                'page' => $searchParams['page'],
                'per_page' => $searchParams['per_page'],
                'total' => 0
            ]
        ]);
        return;
    }

    // Get paginated record data efficiently
    $records = getSearchResultRecords($pdo, $searchQuery, $searchParams, $category);

    // Apply language filter if requested
    $languageCode = isset($_GET['lang']) ? trim($_GET['lang']) : null;
    if ($languageCode) {
        $records = applyLanguageFilter($pdo, $records, $languageCode);
    }

    // Build response with language info
    $response = [
        'items' => $records,
        'pagination' => [
            'page' => $searchParams['page'],
            'per_page' => $searchParams['per_page'],
            'total' => $total
        ]
    ];

    // Add language field to response
    if ($languageCode) {
        $response['language'] = $languageCode;
    }

    jsonResponse(true, $response);
}

// Apply language filter to records - merges translations into record values with fallback to default language
function applyLanguageFilter($pdo, $records, $languageCode) {
    if (empty($records)) return $records;

    // Get language ID and default language
    $stmt = $pdo->prepare("SELECT id, is_default FROM languages WHERE code = ?");
    $stmt->execute([$languageCode]);
    $language = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$language) {
        return $records; // Language not found, return records as-is
    }

    $languageId = $language['id'];
    $recordIds = array_column($records, 'id');

    if (empty($recordIds)) return $records;

    // Get all translations for these records in the requested language
    $placeholders = implode(',', array_fill(0, count($recordIds), '?'));
    $stmt = $pdo->prepare("
        SELECT
            rt.record_id,
            cf.name as field_name,
            rt.value
        FROM record_translations rt
        JOIN category_fields cf ON rt.field_id = cf.id
        WHERE rt.record_id IN ($placeholders)
        AND rt.language_id = ?
    ");
    $stmt->execute([...$recordIds, $languageId]);
    $translations = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get default language translations as fallback (if requested language is not default)
    $defaultTranslations = [];
    if (!$language['is_default']) {
        $stmt = $pdo->prepare("
            SELECT
                rt.record_id,
                cf.name as field_name,
                rt.value
            FROM record_translations rt
            JOIN category_fields cf ON rt.field_id = cf.id
            JOIN languages l ON rt.language_id = l.id
            WHERE rt.record_id IN ($placeholders)
            AND l.is_default = 1
        ");
        $stmt->execute($recordIds);
        $defaultTranslations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Organize translations by record ID
    $translationMap = [];
    foreach ($translations as $trans) {
        if (!isset($translationMap[$trans['record_id']])) {
            $translationMap[$trans['record_id']] = [];
        }
        $translationMap[$trans['record_id']][$trans['field_name']] = [
            'value' => $trans['value'],
            'is_fallback' => false
        ];
    }

    // Organize default translations as fallback
    $defaultTranslationMap = [];
    foreach ($defaultTranslations as $trans) {
        if (!isset($defaultTranslationMap[$trans['record_id']])) {
            $defaultTranslationMap[$trans['record_id']] = [];
        }
        $defaultTranslationMap[$trans['record_id']][$trans['field_name']] = $trans['value'];
    }

    // Merge translations into records with fallback logic
    foreach ($records as &$record) {
        $record['_fallback_fields'] = []; // Track which fields are using fallback

        // First, check if we have translations for this record
        if (isset($translationMap[$record['id']])) {
            foreach ($translationMap[$record['id']] as $fieldName => $translation) {
                $record['values'][$fieldName] = $translation['value'];
            }
        }

        // For fields not translated, use default language translation if available
        if (isset($defaultTranslationMap[$record['id']])) {
            foreach ($defaultTranslationMap[$record['id']] as $fieldName => $defaultValue) {
                // Only use fallback if field is not already translated
                if (!isset($translationMap[$record['id']][$fieldName])) {
                    $record['values'][$fieldName] = $defaultValue;
                    $record['_fallback_fields'][] = $fieldName; // Mark as fallback
                }
            }
        }
    }

    return $records;
}

function parseSearchParameters() {
    return [
        'page' => isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1,
        'per_page' => isset($_GET['per_page']) ? max(1, min(100, (int)$_GET['per_page'])) : 50,
        'query' => isset($_GET['q']) ? trim($_GET['q']) : '',
        'type' => isset($_GET['search_type']) ? trim($_GET['search_type']) : 'all',
        'field' => isset($_GET['search_field']) ? trim($_GET['search_field']) : '',
        'value' => isset($_GET['search_value']) ? trim($_GET['search_value']) : '',
        'tag' => isset($_GET['tag']) ? trim($_GET['tag']) : '',
        'params' => []
    ];
}

function getCategoryInfo($pdo, $categoryId) {
    static $categoryCache = [];

    if (!isset($categoryCache[$categoryId])) {
        $stmt = $pdo->prepare("SELECT id, name, user_id, created_at FROM categories WHERE id = ?");
        $stmt->execute([$categoryId]);
        $category = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($category) {
            // Get category fields
            $fieldsStmt = $pdo->prepare("SELECT id, name, data_type, is_required, default_value, field_options, foreign_key_table, foreign_key_column FROM category_fields WHERE category_id = ? ORDER BY created_at");
            $fieldsStmt->execute([$categoryId]);
            $fields = $fieldsStmt->fetchAll(PDO::FETCH_ASSOC);

            // Process field options for proper JSON formatting
            foreach ($fields as &$field) {
                if ($field['field_options']) {
                    $field['field_options'] = json_decode($field['field_options'], true);
                }
            }

            $category['fields'] = $fields;
        }

        $categoryCache[$categoryId] = $category;
    }

    return $categoryCache[$categoryId];
}

function buildOptimizedSearchQuery($categoryId, &$searchParams) {
    $whereClause = "WHERE r.category_id = ?";
    $searchParams['params'] = [$categoryId];

    // Optimize search based on type
    if (!empty($searchParams['field']) && !empty($searchParams['value'])) {
        // Field-specific search with index-friendly query
        $whereClause .= " AND r.id IN (
            SELECT rv.record_id
            FROM category_record_values rv
            INNER JOIN category_fields f ON rv.field_id = f.id
            WHERE f.category_id = ? AND f.name = ? AND rv.value LIKE ?
        )";
        $searchParams['params'][] = $categoryId;
        $searchParams['params'][] = $searchParams['field'];
        $searchParams['params'][] = "%{$searchParams['value']}%";
    }
    elseif (!empty($searchParams['tag'])) {
        // Tag search optimization
        $whereClause .= " AND r.id IN (
            SELECT rv.record_id
            FROM category_record_values rv
            INNER JOIN category_fields f ON rv.field_id = f.id
            WHERE f.category_id = ? AND f.name = 'tags'
            AND (rv.value LIKE ? OR rv.value LIKE ? OR rv.value LIKE ? OR rv.value = ?)
        )";
        $searchParams['params'][] = $categoryId;
        $searchParams['params'][] = "%,{$searchParams['tag']},%";
        $searchParams['params'][] = "{$searchParams['tag']},%";
        $searchParams['params'][] = "%,{$searchParams['tag']}";
        $searchParams['params'][] = $searchParams['tag'];
    }
    elseif (!empty($searchParams['query'])) {
        switch ($searchParams['type']) {
            case 'slug':
                $whereClause .= " AND r.slug LIKE ?";
                $searchParams['params'][] = "%{$searchParams['query']}%";
                break;
            case 'content':
                $whereClause .= " AND r.id IN (
                    SELECT rv.record_id
                    FROM category_record_values rv
                    WHERE rv.value LIKE ?
                )";
                $searchParams['params'][] = "%{$searchParams['query']}%";
                break;
            case 'all':
            default:
                $whereClause .= " AND (r.slug LIKE ? OR r.id IN (
                    SELECT rv.record_id
                    FROM category_record_values rv
                    WHERE rv.value LIKE ?
                ))";
                $searchParams['params'][] = "%{$searchParams['query']}%";
                $searchParams['params'][] = "%{$searchParams['query']}%";
                break;
        }
    }

    return $whereClause;
}

function getSearchResultCount($pdo, $whereClause, $params) {
    $countSql = "SELECT COUNT(DISTINCT r.id) as total FROM category_records r {$whereClause}";
    $stmt = $pdo->prepare($countSql);
    $stmt->execute($params);
    return (int)$stmt->fetch(PDO::FETCH_ASSOC)['total'];
}

function getSearchResultRecords($pdo, $whereClause, $searchParams, $category) {
    $offset = ($searchParams['page'] - 1) * $searchParams['per_page'];

    // Get record IDs with pagination
    $recordIdsStmt = $pdo->prepare("
        SELECT DISTINCT r.id as record_id, r.created_at, r.slug
        FROM category_records r
        {$whereClause}
        ORDER BY r.created_at DESC
        LIMIT ? OFFSET ?
    ");

    $recordIdsParams = $searchParams['params'];
    $recordIdsParams[] = $searchParams['per_page'];
    $recordIdsParams[] = $offset;
    $recordIdsStmt->execute($recordIdsParams);
    $recordIds = $recordIdsStmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($recordIds)) {
        return [];
    }

    // Get field values efficiently using IN clause
    $recordIdsList = array_column($recordIds, 'record_id');
    $placeholders = str_repeat('?,', count($recordIdsList) - 1) . '?';

    $stmt = $pdo->prepare("
        SELECT r.id as record_id, r.created_at, r.slug,
               f.name as field_name, rv.value, f.data_type
        FROM category_records r
        LEFT JOIN category_record_values rv ON r.id = rv.record_id
        LEFT JOIN category_fields f ON rv.field_id = f.id
        WHERE r.id IN ({$placeholders})
        ORDER BY r.created_at DESC, f.created_at
    ");

    $stmt->execute($recordIdsList);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Process results efficiently
    return processRecordRows($rows, $category);
}

function processRecordRows($rows, $category) {
    $records = [];
    $fieldDataTypes = [];

    // Safety counters to prevent infinite processing
    $maxFields = 1000;
    $maxRows = 10000;
    $fieldCount = 0;
    $rowCount = 0;

    // Create field data type lookup for efficient processing
    foreach ($category['fields'] as $field) {
        if ($fieldCount >= $maxFields) {
            error_log("Warning: Maximum field limit reached during processing");
            break;
        }
        $fieldDataTypes[$field['name']] = $field['data_type'];
        $fieldCount++;
    }

    foreach ($rows as $row) {
        if ($rowCount >= $maxRows) {
            error_log("Warning: Maximum row limit reached during processing");
            break;
        }

        if (!isset($records[$row['record_id']])) {
            $records[$row['record_id']] = [
                'id' => $row['record_id'],
                'slug' => $row['slug'],
                'created_at' => $row['created_at'],
                'values' => []
            ];
        }

        if ($row['field_name'] && $row['value'] !== null) {
            $value = processFieldValue($row['value'], $fieldDataTypes[$row['field_name']] ?? 'text');
            $records[$row['record_id']]['values'][$row['field_name']] = $value;
        }
        $rowCount++;
    }

    return array_values($records);
}

function processFieldValue($value, $dataType) {
    // Recursion protection
    static $recursionDepth = 0;
    static $maxRecursionDepth = 50;

    if ($recursionDepth >= $maxRecursionDepth) {
        error_log("Warning: Maximum recursion depth reached in processFieldValue");
        return $value; // Return original value to prevent infinite recursion
    }

    $recursionDepth++;

    try {
        // Handle different field types efficiently
        switch ($dataType) {
        case 'image':
        case 'file':
        case 'media':
            // Convert comma-separated values to arrays for media fields
            if (strpos($value, ',') !== false && !preg_match('/^[\[\{]/', $value)) {
                $value = array_filter(array_map('trim', explode(',', $value)));
            }
            $result = enhanceMediaUrlsWithDimensions($value);
            break;

        case 'array':
            // Try to decode JSON values for array fields
            if (is_string($value) && !empty($value)) {
                $trimmedValue = trim($value);
                // Check for malformed JSON strings
                if ($trimmedValue === '[' || $trimmedValue === '{' || strlen($trimmedValue) < 2) {
                    error_log("Malformed JSON detected for array field: " . $trimmedValue);
                    $result = [];
                    break;
                }
                $decoded = json_decode($trimmedValue, true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                    $result = $decoded;
                } else {
                    error_log("JSON decode error for array field: " . json_last_error_msg() . " - Value: " . $trimmedValue);
                    $result = [];
                }
            } else {
                $result = is_array($value) ? $value : [];
            }
            break;

        case 'json':
            // Try to decode JSON values
            $decoded = json_decode($value, true);
            $result = $decoded !== null ? $decoded : $value;
            break;

        case 'link':
            // Handle link field type - decode JSON array of link objects
            $decoded = json_decode($value, true);
            if ($decoded !== null && is_array($decoded)) {
                // Link fields store arrays of {media, text} objects
                $result = $decoded;
            } else {
                // Fallback to empty array if not valid JSON
                $result = [];
            }
            break;

        case 'checkbox':
            // Normalize boolean values
            $result = in_array(strtolower($value), ['1', 'true', 'yes', 'on']) ? true : false;
            break;

        case 'foreign_key':
            // Foreign key values should be returned as-is (string or numeric)
            $result = $value;
            break;

        default:
            $result = $value;
            break;
    }

    } catch (Exception $e) {
        error_log("Error in processFieldValue: " . $e->getMessage());
        $result = $value; // Return original value on error
    } finally {
        $recursionDepth--; // Always decrement recursion depth
    }

    return $result;
}
function fixArrayFields($records, $fieldTypes) {
    foreach ($records as &$record) {
        if (isset($record['values']) && is_array($record['values'])) {
            foreach ($record['values'] as $index => &$value) {
                // Skip if already properly formatted or null/empty
                if (is_array($value) || $value === null || $value === '') {
                    continue;
                }
                
                // Check if this looks like a JSON string
                if (is_string($value) && (
                    (substr($value, 0, 1) === '[' && substr($value, -1) === ']') ||
                    (substr($value, 0, 1) === '{' && substr($value, -1) === '}')
                )) {
                    // Try to decode JSON
                    $decoded = json_decode($value, true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        // Successfully decoded JSON
                        if (is_array($decoded)) {
                            $value = $decoded;
                        }
                    }
                }
            }
        }
    }
    return $records;
}

function getRecords1($categoryId, $auth) {
    $pdo = getDbConnection();
    $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
    $perPage = isset($_GET['per_page']) ? max(1, min(100, (int)$_GET['per_page'])) : 20;
    $offset = ($page - 1) * $perPage;

    // First, get the complete category information
    $categoryStmt = $pdo->prepare("SELECT id, name, user_id, created_at FROM categories WHERE id = ?");
    $categoryStmt->execute([$categoryId]);
    $category = $categoryStmt->fetch(PDO::FETCH_ASSOC);

    if (!$category) {
        jsonResponse(false, "Category not found", 404);
    }

    // Get category fields to maintain order and identify field types
    $fieldsStmt = $pdo->prepare("SELECT id, name, data_type FROM category_fields WHERE category_id = ? ORDER BY created_at");
    $fieldsStmt->execute([$categoryId]);
    $fields = $fieldsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Create a map of field names to data types and field order
    $fieldTypes = [];
    $fieldOrder = [];
    foreach ($fields as $index => $field) {
        $fieldTypes[$field['name']] = $field['data_type'];
        $fieldOrder[$field['name']] = $index;
    }

    $searchQuery = isset($_GET['q']) ? trim($_GET['q']) : '';
    $searchType = isset($_GET['search_type']) ? trim($_GET['search_type']) : 'all';
    $searchField = isset($_GET['search_field']) ? trim($_GET['search_field']) : '';
    $searchValue = isset($_GET['search_value']) ? trim($_GET['search_value']) : '';
    $tagSearch = isset($_GET['tag']) ? trim($_GET['tag']) : '';

    $whereClause = "WHERE r.category_id = ?";
    $params = [$categoryId];

    // Handle field-specific search
    if (!empty($searchField) && !empty($searchValue)) {
        // Search by specific field name and value
        $whereClause .= " AND EXISTS (
            SELECT 1 FROM category_record_values rv3
            JOIN category_fields f3 ON rv3.field_id = f3.id
            WHERE rv3.record_id = r.id
            AND f3.name = ?
            AND rv3.value LIKE ?
        )";
        $params[] = $searchField;
        $params[] = "%{$searchValue}%";
    }
    // Handle tag-based search
    elseif (!empty($tagSearch)) {
        // Search for records that have the specified tag
        $whereClause .= " AND EXISTS (
            SELECT 1 FROM category_record_values rv4
            JOIN category_fields f4 ON rv4.field_id = f4.id
            WHERE rv4.record_id = r.id
            AND f4.name = 'tags'
            AND (
                rv4.value LIKE ? OR
                rv4.value LIKE ? OR
                rv4.value LIKE ? OR
                rv4.value = ?
            )
        )";
        $params[] = "%,{$tagSearch},%";  // tag in middle
        $params[] = "{$tagSearch},%";    // tag at start
        $params[] = "%,{$tagSearch}";    // tag at end
        $params[] = $tagSearch;          // exact match
    }
    // Handle general search
    elseif (!empty($searchQuery)) {
        switch ($searchType) {
            case 'slug':
                // Search only in slug field
                $whereClause .= " AND r.slug LIKE ?";
                $params[] = "%{$searchQuery}%";
                break;
            case 'content':
                // Search only in record values (content)
                $whereClause .= " AND EXISTS (SELECT 1 FROM category_record_values rv2 WHERE rv2.record_id = r.id AND rv2.value LIKE ?)";
                $params[] = "%{$searchQuery}%";
                break;
            case 'all':
            default:
                // Search in both slug and record values
                $whereClause .= " AND (r.slug LIKE ? OR EXISTS (SELECT 1 FROM category_record_values rv2 WHERE rv2.record_id = r.id AND rv2.value LIKE ?))";
                $params[] = "%{$searchQuery}%";
                $params[] = "%{$searchQuery}%";
                break;
        }
    }
    
    // Get total count
    $countSql = "SELECT COUNT(DISTINCT r.id) as total FROM category_records r {$whereClause}";
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute($params);
    $total = (int)$countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // First get paginated record IDs
    $recordIdsStmt = $pdo->prepare("
        SELECT DISTINCT r.id as record_id, r.created_at, r.slug
        FROM category_records r
        {$whereClause}
        ORDER BY r.created_at DESC
        LIMIT ? OFFSET ?
    ");

    $recordIdsParams = $params;
    $recordIdsParams[] = $perPage;
    $recordIdsParams[] = $offset;
    $recordIdsStmt->execute($recordIdsParams);
    $recordIds = $recordIdsStmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($recordIds)) {
        jsonResponse(true, [
            'items' => [],
            'pagination' => [
                'page' => $page,
                'per_page' => $perPage,
                'total' => $total
            ]
        ]);
        return;
    }

    // Get all field values for these records
    $recordIdsList = array_map(function($r) { return $r['record_id']; }, $recordIds);
    $placeholders = str_repeat('?,', count($recordIdsList) - 1) . '?';

    $stmt = $pdo->prepare("
        SELECT r.id as record_id, r.created_at, r.slug,
               f.name as field_name, rv.value
        FROM category_records r
        LEFT JOIN category_record_values rv ON r.id = rv.record_id
        LEFT JOIN category_fields f ON rv.field_id = f.id
        WHERE r.id IN ({$placeholders})
        ORDER BY f.created_at, r.created_at DESC
    ");

    $stmt->execute($recordIdsList);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
      $records = [];
    foreach ($rows as $row) {
        if (!isset($records[$row['record_id']])) {
            $records[$row['record_id']] = [
                'id' => $row['record_id'],
                'slug' => $row['slug'],
                'category id' => $category, 
                'created_at' => $row['created_at'],
                'values' => []
            ];
        }
        if ($row['field_name']) {
            $value = $row['value'];
            
            // Convert comma-separated values to arrays for appropriate field types
            if ($value !== null && $value !== '') {
                $fieldName = $row['field_name'];
                $dataType = $fieldTypes[$fieldName] ?? '';

                // Handle link type first
                if ($dataType === 'link') {
                    // Handle link field type - decode JSON and validate structure
                    $decoded = json_decode($value, true);
                    if ($decoded !== null && is_array($decoded)) {
                        // Ensure the link has the required structure with media and text keys
                        $value = [
                            'media' => $decoded['media'] ?? '',
                            'text' => $decoded['text'] ?? ''
                        ];
                    } else {
                        // Fallback to default structure if not valid JSON
                        $value = [
                            'media' => '',
                            'text' => $value ?: ''
                        ];
                    }
                }
                // Convert to array for image, file, media fields and tags
                elseif (in_array($dataType, ['image', 'file', 'media']) || $fieldName === 'tags') {
                    
                    // Check if it's already a JSON string
                    if (is_string($value) && (
                        (substr($value, 0, 1) === '[' && substr($value, -1) === ']') ||
                        (substr($value, 0, 1) === '{' && substr($value, -1) === '}')
                    )) {
                        // Try to decode JSON first
                        $decoded = json_decode($value, true);
                        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                            $value = $decoded;
                        } else {
                            // If JSON decode failed, treat as comma-separated
                            $valueArray = array_map('trim', explode(',', trim($value, '[]"')));
                            $value = array_values(array_filter($valueArray));
                        }
                    } else {
                        // Split by commas and trim each value
                        $valueArray = array_map('trim', explode(',', $value));
                        // Filter out empty values and reindex to ensure sequential keys
                        $value = array_values(array_filter($valueArray));
                    }
                    
                    // Ensure we have an array
                    $value = !empty($value) ? $value : [];

                    // Process media URLs for image, file, media fields - return only relative paths
                    if (in_array($dataType, ['image', 'file', 'media'])) {
                        $value = cleanMediaUrls($value);
                    }
                }
            }
            
            // Add value to the indexed array (use field order as index)
            $fieldIndex = $fieldOrder[$row['field_name']] ?? count($records[$row['record_id']]['values']);
            $records[$row['record_id']]['values'][$fieldIndex] = $value;
        }
    }
    
    // Re-index arrays to ensure sequential numbering (0, 1, 2, ...)
    foreach ($records as &$record) {
        ksort($record['values']);
        $record['values'] = array_values($record['values']);
    }
    
    // Fix any remaining array field issues
    $records = fixArrayFields($records, $fieldTypes);
    
    jsonResponse(true, [
        'items' => array_values($records),
        'pagination' => [
            'page' => $page,
            'per_page' => $perPage,
            'total' => $total
        ]
    ]);
}

// Helper function to clean media URLs and return full URLs with base URL
function cleanMediaUrls($mediaArray) {
    if (empty($mediaArray) || !is_array($mediaArray)) {
        return $mediaArray;
    }
    
    // Define your base URL here
    $baseUrl = 'https://portfolio.maxmind.ma'; // Change this to your actual domain
    
    $cleanedUrls = [];
    foreach ($mediaArray as $url) {
        if (empty($url)) {
            continue;
        }
        
        // Remove localhost domain and get clean path
        $cleanUrl = $url;
        
        // Remove various localhost patterns - using different delimiters to avoid escaping
        $patterns = [
            '#^https?://localhost(:\d+)?/[^/]*/#',
            '#^https?://localhost(:\d+)?/#',
            '#^.*PORTFOLIO\.MAXMIND\.MA-main[\\\/]#',
            '#^.*PORTFOLI.*O\.MAXMIND\.MA-main[\\\/]#',
        ];
        
        foreach ($patterns as $pattern) {
            $cleanUrl = preg_replace($pattern, '', $cleanUrl);
        }
        
        // Ensure the path starts with /admin/uploads/ or just admin/uploads/
        if (!preg_match('#^/admin/uploads/#', $cleanUrl) && !preg_match('#^admin/uploads/#', $cleanUrl)) {
            // If it doesn't start with admin/uploads, try to find it in the path
            if (preg_match('#admin/uploads/.*#', $cleanUrl, $matches)) {
                $cleanUrl = '/' . $matches[0];
            }
        } elseif (preg_match('#^admin/uploads/#', $cleanUrl)) {
            // Add leading slash if missing
            $cleanUrl = '/' . $cleanUrl;
        }
        
        // Add base URL if the path starts with /admin
        if (preg_match('#^/admin/#', $cleanUrl)) {
            $cleanUrl = $baseUrl . $cleanUrl;
        }
        
        $cleanedUrls[] = $cleanUrl;
    }
    
    return $cleanedUrls;
}

// Get searchable fields for a category
function getSearchableFields($categoryId, $auth) {
    $pdo = getDbConnection();

    // Check if category exists
    $categoryStmt = $pdo->prepare("SELECT id, name FROM categories WHERE id = ?");
    $categoryStmt->execute([$categoryId]);
    $category = $categoryStmt->fetch(PDO::FETCH_ASSOC);

    if (!$category) {
        jsonResponse(false, "Category not found", 404);
    }

    // Get all fields for this category
    $fieldsStmt = $pdo->prepare("SELECT name, data_type FROM category_fields WHERE category_id = ? ORDER BY created_at");
    $fieldsStmt->execute([$categoryId]);
    $fields = $fieldsStmt->fetchAll(PDO::FETCH_ASSOC);

    // Add slug as a searchable field
    array_unshift($fields, ['name' => 'slug', 'data_type' => 'text']);

    jsonResponse(true, [
        'category' => $category,
        'searchable_fields' => $fields
    ]);
}

function createRecord($data, $auth) {
    validateRequired($data, ['category_id', 'values']);

    // Debug: Log the received data
    error_log("CreateRecord received data: " . json_encode($data));

    $pdo = getDbConnection();
    
    // Check if category exists
    $stmt = $pdo->prepare("SELECT id FROM categories WHERE id = ?");
    $stmt->execute([$data['category_id']]);
    if (!$stmt->fetch()) {
        jsonResponse(false, "Category not found", 404);
    }
    
    // Extract slug from values
    $slug = $data['values']['slug'] ?? '';
    unset($data['values']['slug']);

    // Validate slug format first
    validateSlugFormat($slug);

    // Validate slug uniqueness if provided
    if (!empty($slug)) {
        $slugCheckStmt = $pdo->prepare("SELECT id FROM category_records WHERE slug = ? AND category_id = ?");
        $slugCheckStmt->execute([$slug, $data['category_id']]);
        if ($slugCheckStmt->fetch()) {
            jsonResponse(false, "Slug must be unique. This slug already exists in this category.");
        }
    }

    // Get fields for this category with data types and required status
    $stmt = $pdo->prepare("SELECT id, name, data_type, is_required FROM category_fields WHERE category_id = ?");
    $stmt->execute([$data['category_id']]);
    $fields = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $fieldMap = [];
    $fieldTypes = [];
    $requiredFields = [];
    foreach ($fields as $field) {
        $fieldMap[$field['name']] = $field['id'];
        $fieldTypes[$field['name']] = $field['data_type'];
        if ((int)$field['is_required'] === 1) {
            $requiredFields[] = $field['name'];
        }
    }

    // Validate required fields
    $missingFields = [];
    foreach ($requiredFields as $requiredField) {
        if (!isset($data['values'][$requiredField]) ||
            (is_string($data['values'][$requiredField]) && trim($data['values'][$requiredField]) === '') ||
            (is_array($data['values'][$requiredField]) && empty($data['values'][$requiredField]))) {
            $missingFields[] = $requiredField;
        }
    }

    if (!empty($missingFields)) {
        jsonResponse(false, "Required fields are missing: " . implode(', ', $missingFields));
    }

    $pdo->beginTransaction();
    try {
        // Create record
        $stmt = $pdo->prepare("INSERT INTO category_records (category_id, slug) VALUES (?, ?)");
        $stmt->execute([$data['category_id'], $slug]);
        $recordId = $pdo->lastInsertId();

        // Save field values
        error_log("=== SAVING FIELD VALUES ===");
        error_log("Available fields in fieldMap: " . json_encode(array_keys($fieldMap)));
        error_log("Incoming field names: " . json_encode(array_keys($data['values'])));

        foreach ($data['values'] as $fieldName => $value) {
            error_log("Checking field: $fieldName");

            if (isset($fieldMap[$fieldName])) {
                $fieldType = $fieldTypes[$fieldName];

                // Debug: Log field processing
                error_log("Processing field: $fieldName, type: $fieldType, value: " . json_encode($value));

                // Handle different field types appropriately
                if ($fieldType === 'array' || $fieldType === 'json' || $fieldType === 'link') {
                    // For array, json, and link fields, store as JSON
                    $valueToStore = is_array($value) ? json_encode($value) : $value;
                } else {
                    // For other fields (image, file, media, etc), use comma-separated format
                    $valueToStore = is_array($value) ? implode(',', $value) : $value;
                }

                // Debug: Log what's being stored
                error_log("Storing field $fieldName with value: $valueToStore");

                try {
                    $stmt = $pdo->prepare("INSERT INTO category_record_values (record_id, field_id, value) VALUES (?, ?, ?)");
                    $success = $stmt->execute([$recordId, $fieldMap[$fieldName], $valueToStore]);
                    if ($success) {
                        error_log("Field $fieldName saved successfully. Row ID: " . $pdo->lastInsertId());
                    } else {
                        $errorInfo = $stmt->errorInfo();
                        error_log("FAILED to save field $fieldName. SQL Error: " . json_encode($errorInfo));
                    }
                } catch (Exception $insertError) {
                    error_log("EXCEPTION saving field $fieldName: " . $insertError->getMessage());
                    throw $insertError;
                }
            } else {
                error_log("SKIPPING field $fieldName - not found in fieldMap");
            }
        }
        
        $pdo->commit();
        jsonResponse(true, ['id' => $recordId, 'slug' => $slug]);
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Create record error: " . $e->getMessage());
        jsonResponse(false, "Failed to create record");
    }
}
function updateRecord($recordId, $data, $auth) {
    $pdo = getDbConnection();
    // Check if record exists and get current values
    $stmt = $pdo->prepare("SELECT id, category_id FROM category_records WHERE id = ?");
    $stmt->execute([$recordId]);
    $record = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$record) {
        jsonResponse(false, "Record not found", 404);
    }
    validateRequired($data, ['values']);
    
    $pdo->beginTransaction();
    try {
        // Extract and update slug
        if (isset($data['values']['slug'])) {
            $slug = $data['values']['slug'];
            unset($data['values']['slug']);

            // Validate slug format
            validateSlugFormat($slug);

            // Check slug uniqueness (exclude current record)
            if (!empty($slug)) {
                $slugCheckStmt = $pdo->prepare("SELECT id FROM category_records WHERE slug = ? AND category_id = ? AND id != ?");
                $slugCheckStmt->execute([$slug, $record['category_id'], $recordId]);
                if ($slugCheckStmt->fetch()) {
                    $pdo->rollback();
                    jsonResponse(false, "Slug must be unique. This slug already exists in this category.");
                }
            }

            $stmt = $pdo->prepare("UPDATE category_records SET slug = ? WHERE id = ?");
            $stmt->execute([$slug, $recordId]);
        }
        
        // Get fields for this category with data types
        $stmt = $pdo->prepare("SELECT id, name, data_type FROM category_fields WHERE category_id = ?");
        $stmt->execute([$record['category_id']]);
        $fields = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $fieldMap = [];
        $fieldTypes = [];
        foreach ($fields as $field) {
            $fieldMap[$field['name']] = $field['id'];
            $fieldTypes[$field['name']] = $field['data_type'];
        }

        // Update field values - Only update provided fields
        foreach ($data['values'] as $fieldName => $value) {
            // Skip if field doesn't exist in category
            if (!isset($fieldMap[$fieldName])) {
                continue;
            }
            
            $fieldId = $fieldMap[$fieldName];
            $fieldType = $fieldTypes[$fieldName];
            
            // Handle different field types appropriately
            if ($fieldType === 'array' || $fieldType === 'json' || $fieldType === 'link') {
                // For array, json, and link fields, store as JSON
                $valueToStore = is_array($value) ? json_encode($value) : $value;
            } else {
                // For other fields, use appropriate formatting
                $valueToStore = is_array($value) ? implode(',', $value) : $value;
            }
            
            // Check if value exists
            $stmt = $pdo->prepare("SELECT id FROM category_record_values WHERE record_id = ? AND field_id = ?");
            $stmt->execute([$recordId, $fieldId]);
            
            if ($stmt->fetch()) {
                // Update existing
                $stmt = $pdo->prepare("UPDATE category_record_values SET value = ? WHERE record_id = ? AND field_id = ?");
                $stmt->execute([$valueToStore, $recordId, $fieldId]);
            } else {
                // Insert new
                $stmt = $pdo->prepare("INSERT INTO category_record_values (record_id, field_id, value) VALUES (?, ?, ?)");
                $stmt->execute([$recordId, $fieldId, $valueToStore]);
            }
        }
        
        $pdo->commit();
        jsonResponse(true, "Record updated successfully");
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Update record error: " . $e->getMessage());
        jsonResponse(false, "Failed to update record");
    }
}

function deleteRecord($recordId, $auth) {
    $pdo = getDbConnection();
    
    // Check if record exists
    $stmt = $pdo->prepare("SELECT id FROM category_records WHERE id = ?");
    $stmt->execute([$recordId]);
    if (!$stmt->fetch()) {
        jsonResponse(false, "Record not found", 404);
    }
    
    $pdo->beginTransaction();
    try {
        // Delete record values
        $stmt = $pdo->prepare("DELETE FROM category_record_values WHERE record_id = ?");
        $stmt->execute([$recordId]);
        
        // Delete record
        $stmt = $pdo->prepare("DELETE FROM category_records WHERE id = ?");
        $stmt->execute([$recordId]);
        
        $pdo->commit();
        jsonResponse(true, "Record deleted successfully");
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Delete record error: " . $e->getMessage());
        jsonResponse(false, "Failed to delete record");
    }
}

function exportDatabase($categoryId, $auth) {
    $pdo = getDbConnection();

    // Check if category exists
    $stmt = $pdo->prepare("SELECT id, name FROM categories WHERE id = ?");
    $stmt->execute([$categoryId]);
    $category = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$category) {
        jsonResponse(false, "Category not found", 404);
    }

    try {
        $sql = "-- Database Export for Category: " . $category['name'] . "\n";
        $sql .= "-- Generated on: " . date('Y-m-d H:i:s') . "\n\n";

        // Export category structure
        $sql .= "-- Category: " . $category['name'] . "\n";
        $sql .= "INSERT INTO categories (id, name, user_id, created_at) VALUES (" .
                $category['id'] . ", '" . addslashes($category['name']) . "', " .
                ($auth['user_id'] ?? 'NULL') . ", '" . date('Y-m-d H:i:s') . "');\n\n";

        // Export fields for this category
        $fieldsStmt = $pdo->prepare("SELECT * FROM category_fields WHERE category_id = ? ORDER BY created_at");
        $fieldsStmt->execute([$categoryId]);
        $fields = $fieldsStmt->fetchAll(PDO::FETCH_ASSOC);

        if (!empty($fields)) {
            $sql .= "-- Category Fields\n";
            foreach ($fields as $field) {
                $sql .= "INSERT INTO category_fields (id, category_id, name, data_type, is_required, default_value, field_options, foreign_key_table, foreign_key_column, created_at) VALUES (";
                $sql .= $field['id'] . ", ";
                $sql .= $field['category_id'] . ", ";
                $sql .= "'" . addslashes($field['name']) . "', ";
                $sql .= "'" . addslashes($field['data_type']) . "', ";
                $sql .= $field['is_required'] . ", ";
                $sql .= ($field['default_value'] ? "'" . addslashes($field['default_value']) . "'" : 'NULL') . ", ";
                $sql .= ($field['field_options'] ? "'" . addslashes($field['field_options']) . "'" : 'NULL') . ", ";
                $sql .= ($field['foreign_key_table'] ? "'" . addslashes($field['foreign_key_table']) . "'" : 'NULL') . ", ";
                $sql .= ($field['foreign_key_column'] ? "'" . addslashes($field['foreign_key_column']) . "'" : 'NULL') . ", ";
                $sql .= "'" . date('Y-m-d H:i:s') . "'";
                $sql .= ");\n";
            }
            $sql .= "\n";
        }

        // Export records for this category
        $recordsStmt = $pdo->prepare("SELECT * FROM category_records WHERE category_id = ? ORDER BY created_at");
        $recordsStmt->execute([$categoryId]);
        $records = $recordsStmt->fetchAll(PDO::FETCH_ASSOC);

        if (!empty($records)) {
            $sql .= "-- Category Records\n";
            foreach ($records as $record) {
                $sql .= "INSERT INTO category_records (id, category_id, slug, created_at) VALUES (";
                $sql .= $record['id'] . ", ";
                $sql .= $record['category_id'] . ", ";
                $sql .= ($record['slug'] ? "'" . addslashes($record['slug']) . "'" : 'NULL') . ", ";
                $sql .= "'" . $record['created_at'] . "'";
                $sql .= ");\n";

                // Export record values
                $valuesStmt = $pdo->prepare("SELECT * FROM category_record_values WHERE record_id = ?");
                $valuesStmt->execute([$record['id']]);
                $values = $valuesStmt->fetchAll(PDO::FETCH_ASSOC);

                foreach ($values as $value) {
                    $sql .= "INSERT INTO category_record_values (id, record_id, field_id, value) VALUES (";
                    $sql .= $value['id'] . ", ";
                    $sql .= $value['record_id'] . ", ";
                    $sql .= $value['field_id'] . ", ";
                    $sql .= "'" . addslashes($value['value']) . "'";
                    $sql .= ");\n";
                }
                $sql .= "\n";
            }
        }

        // Return the SQL content with proper headers
        header('Content-Type: application/sql');
        header('Content-Disposition: attachment; filename="export_category_' . $categoryId . '_' . date('Y-m-d_H-i-s') . '.sql"');
        header('Content-Length: ' . strlen($sql));

        // Clear any output buffer
        if (ob_get_length()) ob_clean();

        echo $sql;
        exit;

    } catch (Exception $e) {
        error_log("Export database error: " . $e->getMessage());
        jsonResponse(false, "Failed to export database: " . $e->getMessage());
    }
}

// File Functions
function getUploadedFiles($auth) {
    $pdo = getDbConnection();
    $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
    $perPage = isset($_GET['per_page']) ? max(1, min(100, (int)$_GET['per_page'])) : 8;
    $offset = ($page - 1) * $perPage;
    
    // Filter by type, search, and tags if specified
    $typeFilter = isset($_GET['type']) ? $_GET['type'] : '';
    $searchFilter = isset($_GET['search']) ? trim($_GET['search']) : '';
    $searchType = isset($_GET['search_type']) ? $_GET['search_type'] : 'name';
    $tagFilter = isset($_GET['tag']) ? trim($_GET['tag']) : '';
    $whereConditions = [];
    $params = [];

    if ($typeFilter === 'image') {
        $whereConditions[] = "file_type LIKE 'image/%'";
    } elseif ($typeFilter === 'video') {
        $whereConditions[] = "file_type LIKE 'video/%'";
    } elseif ($typeFilter === 'audio') {
        $whereConditions[] = "file_type LIKE 'audio/%'";
    } elseif ($typeFilter === 'file') {
        $whereConditions[] = "file_type NOT LIKE 'image/%' AND file_type NOT LIKE 'video/%' AND file_type NOT LIKE 'audio/%'";
    }

    if (!empty($searchFilter)) {
        if ($searchType === 'tag') {
            // Search specifically in tags
            $whereConditions[] = "(tags LIKE ? OR tags LIKE ? OR tags LIKE ? OR tags = ?)";
            $params[] = "%,{$searchFilter},%";  // tag in middle
            $params[] = "{$searchFilter},%";    // tag at start
            $params[] = "%,{$searchFilter}";    // tag at end
            $params[] = $searchFilter;          // exact match
        } else {
            // Search by name (default behavior)
            $whereConditions[] = "(filename LIKE ? OR original_name LIKE ?)";
            $params[] = "%{$searchFilter}%";
            $params[] = "%{$searchFilter}%";
        }
    }

    if (!empty($tagFilter)) {
        $whereConditions[] = "(tags LIKE ? OR tags LIKE ? OR tags LIKE ? OR tags = ?)";
        $params[] = "%,{$tagFilter},%";  // tag in middle
        $params[] = "{$tagFilter},%";    // tag at start
        $params[] = "%,{$tagFilter}";    // tag at end
        $params[] = $tagFilter;          // exact match
    }

    $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
    
    // Get total count with filtering
    $countQuery = "SELECT COUNT(*) as total FROM uploaded_files " . $whereClause;
    $countStmt = $pdo->prepare($countQuery);
    $countStmt->execute($params);
    $total = (int)$countStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Get paginated results with filtering
    $filesQuery = "SELECT id, filename, original_name, file_type, file_size, upload_path, width, height, tags, slug, original_url, created_at FROM uploaded_files " . $whereClause . " ORDER BY created_at DESC LIMIT ? OFFSET ?";
    $stmt = $pdo->prepare($filesQuery);
    $stmt->execute(array_merge($params, [$perPage, $offset]));
    $files = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Add full URL for each file with dimensions and handle video URLs
    foreach ($files as &$file) {
        if ($file['file_type'] === 'video/url') {
            // For video URLs, the upload_path contains the embed URL
            $file['url'] = $file['upload_path'];
            $file['embed_url'] = $file['upload_path'];

            // Regenerate embed HTML based on the original URL (if available) or embed URL
            $urlToProcess = $file['original_url'] ?: $file['upload_path'];
            $videoInfo = processVideoUrl($urlToProcess);
            if ($videoInfo) {
                $file['embed_html'] = $videoInfo['embed_html'];
                $file['platform'] = $videoInfo['platform'];
                $file['thumbnail'] = $videoInfo['thumbnail'] ?? null;
            }
        } else {
            $file['url'] = constructMediaUrl($file['upload_path'], $file['width'], $file['height']);
        }
    }
    
    jsonResponse(true, [
        'items' => $files,
        'pagination' => [
            'page' => $page,
            'per_page' => $perPage,
            'total' => $total
        ]
    ]);
}

function uploadFile($file, $auth) {
    // Ensure we have a clean output buffer for JSON with safety counter
    $bufferClearCount = 0;
    $maxBufferClears = 10; // Prevent infinite loop
    while (ob_get_level() && $bufferClearCount < $maxBufferClears) {
        ob_end_clean();
        $bufferClearCount++;
    }
    ob_start();

    try {
        // Validate file array structure first
        $validationResult = validateUploadedFile($file);
        if (!$validationResult['valid']) {
            jsonResponse(false, $validationResult['error']);
        }

        // Create upload directory with proper permissions
        $uploadDir = createUploadDirectory();

        // Validate file type and get normalized file info
        $fileInfo = validateAndNormalizeFileType($file);

        // Validate file size based on type
        validateFileSize($file, $fileInfo);

        // Generate secure filename and upload path
        $filename = generateSecureFilename($fileInfo['extension']);
        $uploadPath = $uploadDir . $filename;

        // Move uploaded file securely
        if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
            jsonResponse(false, "Failed to move uploaded file to destination");
        }

        // Extract media dimensions
        $dimensions = extractMediaDimensions($uploadPath, $fileInfo);

        // Get and validate tags
        $tags = validateAndCleanTags($_POST['tags'] ?? '');

        // Get and validate slug
        $slug = trim($_POST['slug'] ?? '');

        // Store file info in database
        $fileId = storeFileInDatabase($filename, $file, $fileInfo, $uploadPath, $dimensions, $tags, $slug);

        $fileData = [
            'id' => $fileId,
            'filename' => $filename,
            'original_name' => $file['name'],
            'file_type' => $fileInfo['mime_type'],
            'file_size' => $file['size'],
            'width' => $dimensions['width'],
            'height' => $dimensions['height'],
            'tags' => $tags,
            'slug' => $slug ?: null,
            'url' => constructMediaUrl($uploadPath, $dimensions['width'], $dimensions['height'])
        ];

        jsonResponse(true, $fileData);

    } catch (Exception $e) {
        error_log("File upload error: " . $e->getMessage());
        jsonResponse(false, "Upload failed: " . $e->getMessage());
    }
}

function validateUploadedFile($file) {
    if (!is_array($file) || !isset($file['error']) || !isset($file['name']) || !isset($file['tmp_name'])) {
        return ['valid' => false, 'error' => 'Invalid file upload data'];
    }

    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errorMessages = [
            UPLOAD_ERR_INI_SIZE => 'File is too large (exceeds php.ini upload_max_filesize)',
            UPLOAD_ERR_FORM_SIZE => 'File is too large (exceeds form MAX_FILE_SIZE)',
            UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
            UPLOAD_ERR_NO_FILE => 'No file was uploaded',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
            UPLOAD_ERR_EXTENSION => 'File upload stopped by extension'
        ];
        $errorMsg = $errorMessages[$file['error']] ?? 'Unknown upload error';
        return ['valid' => false, 'error' => $errorMsg];
    }

    if (!is_uploaded_file($file['tmp_name'])) {
        return ['valid' => false, 'error' => 'Security error: file was not uploaded properly'];
    }

    return ['valid' => true];
}

function createUploadDirectory() {
    $uploadDir = 'uploads/';
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            throw new Exception("Failed to create upload directory");
        }
    }
    return $uploadDir;
}

function validateAndNormalizeFileType($file) {
    // Get real MIME type using finfo (more reliable than $_FILES['type'])
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    if (!$finfo) {
        throw new Exception("Unable to create fileinfo resource");
    }

    $realMimeType = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    if (!$realMimeType) {
        throw new Exception("Unable to determine file MIME type");
    }

    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

    // Define allowed file types with their corresponding extensions
    $allowedTypes = [
        // Images
        'image/jpeg' => ['jpg', 'jpeg'],
        'image/png' => ['png'],
        'image/gif' => ['gif'],
        'image/webp' => ['webp'],
        'image/svg+xml' => ['svg'],

        // Videos
        'video/mp4' => ['mp4'],
        'video/quicktime' => ['mov'],
        'video/x-msvideo' => ['avi'],
        'video/webm' => ['webm'],
        'video/x-matroska' => ['mkv'],

        // Audio
        'audio/mpeg' => ['mp3'],
        'audio/wav' => ['wav'],
        'audio/ogg' => ['ogg'],
        'audio/mp4' => ['m4a'],
        'audio/aac' => ['aac'],

        // Documents
        'application/pdf' => ['pdf'],
        'application/msword' => ['doc'],
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => ['docx']
    ];

    // Check if MIME type is allowed
    if (!array_key_exists($realMimeType, $allowedTypes)) {
        throw new Exception("File type not allowed: {$realMimeType}");
    }

    // Verify that the file extension matches the MIME type
    if (!in_array($fileExtension, $allowedTypes[$realMimeType])) {
        throw new Exception("File extension '{$fileExtension}' does not match MIME type '{$realMimeType}'");
    }

    return [
        'mime_type' => $realMimeType,
        'extension' => $fileExtension,
        'category' => getFileCategory($realMimeType)
    ];
}

function getFileCategory($mimeType) {
    if (strpos($mimeType, 'image/') === 0) return 'image';
    if (strpos($mimeType, 'video/') === 0) return 'video';
    if (strpos($mimeType, 'audio/') === 0) return 'audio';
    return 'document';
}

function validateFileSize($file, $fileInfo) {
    $maxSizes = [
        'image' => 50 * 1024 * 1024,    // 50MB for images
        'video' => 500 * 1024 * 1024,   // 500MB for videos
        'audio' => 50 * 1024 * 1024,    // 50MB for audio
        'document' => 50 * 1024 * 1024  // 50MB for documents
    ];

    $maxSize = $maxSizes[$fileInfo['category']] ?? 50 * 1024 * 1024;

    if ($file['size'] > $maxSize) {
        throw new Exception("File size too large (max " . formatBytes($maxSize) . "). Your file is " . formatBytes($file['size']));
    }
}

function generateSecureFilename($extension) {
    // Generate a secure filename using timestamp and random data
    return date('Y-m-d_H-i-s') . '_' . bin2hex(random_bytes(8)) . '.' . $extension;
}

function extractMediaDimensions($uploadPath, $fileInfo) {
    $width = null;
    $height = null;

    try {
        switch ($fileInfo['category']) {
            case 'image':
                $dimensions = extractImageDimensions($uploadPath);
                break;
            case 'video':
                $dimensions = extractVideoDimensions($uploadPath);
                break;
            default:
                $dimensions = ['width' => null, 'height' => null];
        }

        return $dimensions;
    } catch (Exception $e) {
        error_log("Dimension extraction failed: " . $e->getMessage());
        return ['width' => null, 'height' => null];
    }
}

function extractImageDimensions($imagePath) {
    $imageInfo = @getimagesize($imagePath);
    if ($imageInfo !== false && isset($imageInfo[0]) && isset($imageInfo[1])) {
        return ['width' => $imageInfo[0], 'height' => $imageInfo[1]];
    }
    return ['width' => null, 'height' => null];
}

function extractVideoDimensions($videoPath) {
    // Try getimagesize first (works for some video formats)
    $dimensions = extractImageDimensions($videoPath);
    if ($dimensions['width'] !== null) {
        return $dimensions;
    }

    // Try basic MP4 header reading as fallback
    if (pathinfo($videoPath, PATHINFO_EXTENSION) === 'mp4') {
        $mp4Dimensions = extractMP4Dimensions($videoPath);
        if ($mp4Dimensions) {
            return $mp4Dimensions;
        }
    }

    return ['width' => null, 'height' => null];
}

function validateAndCleanTags($tagsInput) {
    if (empty($tagsInput)) {
        return '';
    }

    // Split by comma, trim whitespace, remove empty values
    $tags = array_map('trim', explode(',', $tagsInput));
    $tags = array_filter($tags, function($tag) {
        return !empty($tag) && strlen($tag) <= 50; // Max 50 chars per tag
    });

    // Limit to 10 tags maximum
    $tags = array_slice($tags, 0, 10);

    return implode(',', $tags);
}

function storeFileInDatabase($filename, $file, $fileInfo, $uploadPath, $dimensions, $tags, $slug = null) {
    $pdo = getDbConnection();
    $stmt = $pdo->prepare("INSERT INTO uploaded_files (filename, original_name, file_type, file_size, upload_path, width, height, tags, slug) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([
        $filename,
        $file['name'],
        $fileInfo['mime_type'],
        $file['size'],
        $uploadPath,
        $dimensions['width'],
        $dimensions['height'],
        $tags,
        $slug ?: null  // Store null if slug is empty
    ]);

    return $pdo->lastInsertId();
}

function uploadVideoUrl($data, $auth) {
    validateRequired($data, ['url']);

    $url = trim($data['url']);
    $title = isset($data['title']) ? trim($data['title']) : '';
    $tags = isset($data['tags']) ? trim($data['tags']) : '';

    // Validate URL format
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        jsonResponse(false, "Invalid URL format");
    }

    // Extract video info and convert to embed URL
    $videoInfo = processVideoUrl($url);
    if (!$videoInfo) {
        jsonResponse(false, "Unsupported video URL or unable to process");
    }

    // Use provided title or extracted title
    $finalTitle = !empty($title) ? $title : $videoInfo['title'];

    try {
        // Store video URL info in database
        $pdo = getDbConnection();
        $stmt = $pdo->prepare("INSERT INTO uploaded_files (filename, original_name, file_type, file_size, upload_path, width, height, tags, original_url) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $videoInfo['video_id'], // Use video ID as filename
            $finalTitle, // Use title as original name
            'video/url', // Special type for video URLs
            0, // No file size for URLs
            $videoInfo['embed_url'], // Store embed URL as upload path
            $videoInfo['width'] ?? 560,
            $videoInfo['height'] ?? 315,
            $tags,
            $url // Store original URL for reference
        ]);

        $fileId = $pdo->lastInsertId();

        $fileData = [
            'id' => $fileId,
            'filename' => $videoInfo['video_id'],
            'original_name' => $finalTitle,
            'file_type' => 'video/url',
            'file_size' => 0,
            'width' => $videoInfo['width'] ?? 560,
            'height' => $videoInfo['height'] ?? 315,
            'tags' => $tags,
            'url' => $videoInfo['embed_url'],
            'original_url' => $url,
            'platform' => $videoInfo['platform'],
            'embed_html' => $videoInfo['embed_html']
        ];

        jsonResponse(true, $fileData);

    } catch (Exception $e) {
        error_log("Video URL upload error: " . $e->getMessage());
        jsonResponse(false, "Failed to save video URL: " . $e->getMessage());
    }
}

function uploadMediaUrl($data, $auth) {
    validateRequired($data, ['url', 'type']);

    $url = trim($data['url']);
    $type = trim($data['type']); // 'image' or 'video'
    $title = isset($data['title']) ? trim($data['title']) : '';
    $tags = isset($data['tags']) ? trim($data['tags']) : '';

    // Validate URL format
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        jsonResponse(false, "Invalid URL format");
    }

    // Validate and process based on type
    if ($type === 'video') {
        // Process video URL
        $mediaInfo = processVideoUrl($url);
        if (!$mediaInfo) {
            jsonResponse(false, "Unsupported video URL or unable to process");
        }
        $fileType = 'video/url';
        $normalizedFilename = $mediaInfo['video_id'] ?? md5($url);
        $normalizedUploadPath = $mediaInfo['embed_url'] ?? $url;
        $normalizedWidth = $mediaInfo['width'] ?? 560;
        $normalizedHeight = $mediaInfo['height'] ?? 315;
    } elseif ($type === 'image') {
        // Process image URL
        $mediaInfo = processImageUrl($url);
        if (!$mediaInfo) {
            jsonResponse(false, "Unable to process image URL");
        }
        $fileType = 'image/url';
        $normalizedFilename = $mediaInfo['media_id'] ?? md5($url);
        $normalizedUploadPath = $url;
        $normalizedWidth = $mediaInfo['width'] ?? null;
        $normalizedHeight = $mediaInfo['height'] ?? null;
    } else {
        jsonResponse(false, "Invalid media type. Must be 'image' or 'video'");
    }

    // Use provided title or extracted title
    $finalTitle = !empty($title) ? $title : $mediaInfo['title'];
    $finalTitle = !empty($finalTitle) ? $finalTitle : basename(parse_url($url, PHP_URL_PATH));

    try {
        // Store media URL info in database
        $pdo = getDbConnection();
        $stmt = $pdo->prepare("INSERT INTO uploaded_files (filename, original_name, file_type, file_size, upload_path, width, height, tags, original_url) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $normalizedFilename,
            $finalTitle,
            $fileType,
            0,
            $normalizedUploadPath,
            $normalizedWidth,
            $normalizedHeight,
            $tags,
            $url
        ]);

        $fileId = $pdo->lastInsertId();

        $fileData = [
            'id' => $fileId,
            'filename' => $normalizedFilename,
            'original_name' => $finalTitle,
            'file_type' => $fileType,
            'file_size' => 0,
            'width' => $normalizedWidth,
            'height' => $normalizedHeight,
            'tags' => $tags,
            'url' => $normalizedUploadPath,
            'original_url' => $url,
            'media_type' => $type
        ];

        // Add type-specific data
        if ($type === 'video' && isset($mediaInfo['embed_html'])) {
            $fileData['platform'] = $mediaInfo['platform'];
            $fileData['embed_html'] = $mediaInfo['embed_html'];
            $fileData['embed_url'] = $mediaInfo['embed_url'];
        }

        jsonResponse(true, $fileData);

    } catch (Exception $e) {
        error_log("Media URL upload error: " . $e->getMessage());
        jsonResponse(false, "Failed to save media URL: " . $e->getMessage());
    }
}

function processImageUrl($url) {
    // Validate that this is likely an image URL
    $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'bmp'];
    $urlPath = parse_url($url, PHP_URL_PATH);
    $extension = strtolower(pathinfo($urlPath, PATHINFO_EXTENSION));

    if (!in_array($extension, $imageExtensions)) {
        // Try to check if it's an image by attempting to get headers
        $headers = @get_headers($url, 1);
        if ($headers && isset($headers['Content-Type'])) {
            $contentType = is_array($headers['Content-Type']) ? end($headers['Content-Type']) : $headers['Content-Type'];
            if (strpos($contentType, 'image/') !== 0) {
                return false;
            }
        } else {
            return false;
        }
    }

    // Generate a unique ID for the image
    $imageId = md5($url);
    $title = basename($urlPath) ?: "External Image";

    // Try to get image dimensions (this might fail for external URLs due to CORS)
    $width = null;
    $height = null;

    try {
        $imageInfo = @getimagesize($url);
        if ($imageInfo !== false) {
            $width = $imageInfo[0];
            $height = $imageInfo[1];
        }
    } catch (Exception $e) {
        // Ignore dimension extraction errors
    }

    return [
        'platform' => 'external',
        'media_id' => $imageId,
        'title' => $title,
        'width' => $width,
        'height' => $height,
        'url' => $url
    ];
}

function processVideoUrl($url) {
    // YouTube URL patterns
    if (preg_match('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/', $url, $matches)) {
        $videoId = $matches[1];
        return [
            'platform' => 'youtube',
            'video_id' => $videoId,
            'embed_url' => "https://www.youtube.com/embed/{$videoId}",
            'thumbnail' => "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg",
            'title' => getYouTubeTitle($videoId),
            'width' => 560,
            'height' => 315,
            'embed_html' => "<iframe width=\"560\" height=\"315\" src=\"https://www.youtube.com/embed/{$videoId}\" frameborder=\"0\" allow=\"accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture\" allowfullscreen></iframe>"
        ];
    }

    // Vimeo URL patterns
    if (preg_match('/(?:vimeo\.com\/)(\d+)/', $url, $matches)) {
        $videoId = $matches[1];
        return [
            'platform' => 'vimeo',
            'video_id' => $videoId,
            'embed_url' => "https://player.vimeo.com/video/{$videoId}",
            'title' => "Vimeo Video {$videoId}",
            'width' => 560,
            'height' => 315,
            'embed_html' => "<iframe src=\"https://player.vimeo.com/video/{$videoId}\" width=\"560\" height=\"315\" frameborder=\"0\" allow=\"autoplay; fullscreen; picture-in-picture\" allowfullscreen></iframe>"
        ];
    }

    // Dailymotion URL patterns
    if (preg_match('/(?:dailymotion\.com\/video\/)([a-zA-Z0-9]+)/', $url, $matches)) {
        $videoId = $matches[1];
        return [
            'platform' => 'dailymotion',
            'video_id' => $videoId,
            'embed_url' => "https://www.dailymotion.com/embed/video/{$videoId}",
            'title' => "Dailymotion Video {$videoId}",
            'width' => 560,
            'height' => 315,
            'embed_html' => "<iframe frameborder=\"0\" width=\"560\" height=\"315\" src=\"https://www.dailymotion.com/embed/video/{$videoId}\" allowfullscreen allow=\"autoplay\"></iframe>"
        ];
    }

    // Instagram URL patterns (posts and reels)
    if (preg_match('/(?:instagram\.com\/(?:p|reel)\/([A-Za-z0-9_-]+))/', $url, $matches)) {
        $postId = $matches[1];
        // Use the appropriate embed URL based on whether it's a reel or post
        $isReel = strpos($url, '/reel/') !== false;
        $embedUrl = $isReel ? "https://www.instagram.com/reel/{$postId}/embed/" : "https://www.instagram.com/p/{$postId}/embed/";
        $title = $isReel ? "Instagram Reel {$postId}" : "Instagram Post {$postId}";
        return [
            'platform' => 'instagram',
            'video_id' => $postId,
            'embed_url' => $embedUrl,
            'title' => $title,
            'width' => 400,
            'height' => 480,
            'embed_html' => "<iframe src=\"{$embedUrl}\" width=\"400\" height=\"480\" frameborder=\"0\" scrolling=\"no\" allowtransparency=\"true\" allow=\"encrypted-media\"></iframe>"
        ];
    }

    // Instagram TV (IGTV) URL patterns
    if (preg_match('/(?:instagram\.com\/tv\/([A-Za-z0-9_-]+))/', $url, $matches)) {
        $postId = $matches[1];
        $embedUrl = "https://www.instagram.com/tv/{$postId}/embed/";
        return [
            'platform' => 'instagram',
            'video_id' => $postId,
            'embed_url' => $embedUrl,
            'title' => "Instagram TV {$postId}",
            'width' => 400,
            'height' => 480,
            'embed_html' => "<iframe src=\"{$embedUrl}\" width=\"400\" height=\"480\" frameborder=\"0\" scrolling=\"no\" allowtransparency=\"true\" allow=\"encrypted-media\"></iframe>"
        ];
    }

    // TikTok URL patterns
    if (preg_match('/(?:tiktok\.com\/@[^\/]+\/video\/(\d+)|vm\.tiktok\.com\/([A-Za-z0-9]+))/', $url, $matches)) {
        $videoId = $matches[1] ?? $matches[2];
        $embedUrl = "https://www.tiktok.com/embed/v2/{$videoId}";
        return [
            'platform' => 'tiktok',
            'video_id' => $videoId,
            'embed_url' => $embedUrl,
            'title' => "TikTok Video {$videoId}",
            'width' => 325,
            'height' => 578,
            'embed_html' => "<iframe src=\"{$embedUrl}\" width=\"325\" height=\"578\" frameborder=\"0\" scrolling=\"no\" allowtransparency=\"true\" allow=\"encrypted-media; autoplay\"></iframe>"
        ];
    }

    // TikTok alternative URL patterns (for share links)
    if (preg_match('/(?:tiktok\.com\/t\/([A-Za-z0-9]+))/', $url, $matches)) {
        $shareId = $matches[1];
        // For share links, we'll use the share ID directly
        $embedUrl = "https://www.tiktok.com/embed/v2/{$shareId}";
        return [
            'platform' => 'tiktok',
            'video_id' => $shareId,
            'embed_url' => $embedUrl,
            'title' => "TikTok Video {$shareId}",
            'width' => 325,
            'height' => 578,
            'embed_html' => "<iframe src=\"{$embedUrl}\" width=\"325\" height=\"578\" frameborder=\"0\" scrolling=\"no\" allowtransparency=\"true\" allow=\"encrypted-media; autoplay\"></iframe>"
        ];
    }

    // Facebook Video URL patterns
    if (preg_match('/(?:facebook\.com\/(?:watch\/\?v=|.*\/videos\/)(\d+))/', $url, $matches)) {
        $videoId = $matches[1];
        $embedUrl = "https://www.facebook.com/plugins/video.php?href=" . urlencode($url) . "&show_text=false&width=560";
        return [
            'platform' => 'facebook',
            'video_id' => $videoId,
            'embed_url' => $embedUrl,
            'title' => "Facebook Video {$videoId}",
            'width' => 560,
            'height' => 315,
            'embed_html' => "<iframe src=\"{$embedUrl}\" width=\"560\" height=\"315\" frameborder=\"0\" scrolling=\"no\" allowtransparency=\"true\" allowfullscreen=\"true\" allow=\"autoplay; clipboard-write; encrypted-media; picture-in-picture; web-share\"></iframe>"
        ];
    }

    // Twitter/X Video URL patterns
    if (preg_match('/(?:twitter\.com|x\.com)\/(?:#!\/)?(\w+)\/status(es)?\/(\d+)/', $url, $matches)) {
        $tweetId = $matches[3];
        $embedUrl = "https://platform.twitter.com/embed/Tweet.html?id={$tweetId}";
        return [
            'platform' => 'twitter',
            'video_id' => $tweetId,
            'embed_url' => $embedUrl,
            'title' => "Twitter/X Post {$tweetId}",
            'width' => 550,
            'height' => 400,
            'embed_html' => "<iframe src=\"{$embedUrl}\" width=\"550\" height=\"400\" frameborder=\"0\" scrolling=\"no\" allowtransparency=\"true\" allow=\"encrypted-media\"></iframe>"
        ];
    }

    // LinkedIn Video URL patterns
    if (preg_match('/(?:linkedin\.com\/(?:embed\/)?(?:feed\/update\/urn:li:share:|posts\/)([A-Za-z0-9-]+))/', $url, $matches)) {
        $postId = $matches[1];
        $embedUrl = "https://www.linkedin.com/embed/feed/update/urn:li:share:{$postId}";
        return [
            'platform' => 'linkedin',
            'video_id' => $postId,
            'embed_url' => $embedUrl,
            'title' => "LinkedIn Post {$postId}",
            'width' => 550,
            'height' => 400,
            'embed_html' => "<iframe src=\"{$embedUrl}\" width=\"550\" height=\"400\" frameborder=\"0\" allowtransparency=\"true\" allow=\"encrypted-media\"></iframe>"
        ];
    }

    // Generic video URL (mp4, webm, etc.)
    if (preg_match('/\.(mp4|webm|ogg|avi|mov)(\?.*)?$/i', $url)) {
        $videoId = md5($url);
        return [
            'platform' => 'direct',
            'video_id' => $videoId,
            'embed_url' => $url,
            'title' => basename(parse_url($url, PHP_URL_PATH)),
            'width' => 560,
            'height' => 315,
            'embed_html' => "<video width=\"560\" height=\"315\" controls><source src=\"{$url}\" type=\"video/mp4\">Your browser does not support the video tag.</video>"
        ];
    }

    return false;
}

function getYouTubeTitle($videoId) {
    // Try to get YouTube video title (optional, requires API key)
    // For now, return a default title
    return "YouTube Video {$videoId}";
}

function formatBytes($size) {
    $units = array('B', 'KB', 'MB', 'GB');
    for ($i = 0; $size > 1024 && $i < count($units) - 1; $i++) {
        $size /= 1024;
    }
    return round($size, 2) . ' ' . $units[$i];
}

function getUploadConfig($auth) {
    $config = [
        'upload_max_filesize' => ini_get('upload_max_filesize'),
        'post_max_size' => ini_get('post_max_size'),
        'max_execution_time' => ini_get('max_execution_time'),
        'max_input_time' => ini_get('max_input_time'),
        'memory_limit' => ini_get('memory_limit'),
        'upload_max_filesize_bytes' => parseSize(ini_get('upload_max_filesize')),
        'post_max_size_bytes' => parseSize(ini_get('post_max_size')),
        'recommended_video_limit' => '100MB',
        'recommended_other_limit' => '10MB'
    ];

    jsonResponse(true, $config);
}

function getAvailableTags($auth) {
    $pdo = getDbConnection();

    // Get all unique tags from uploaded files
    $stmt = $pdo->prepare("SELECT DISTINCT tags FROM uploaded_files WHERE tags IS NOT NULL AND tags != ''");
    $stmt->execute();
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $allTags = [];
    foreach ($rows as $row) {
        if (!empty($row['tags'])) {
            // Split comma-separated tags and trim whitespace
            $tags = array_map('trim', explode(',', $row['tags']));
            $allTags = array_merge($allTags, $tags);
        }
    }

    // Remove duplicates and empty values, then sort
    $uniqueTags = array_unique(array_filter($allTags));
    sort($uniqueTags);

    jsonResponse(true, $uniqueTags);
}

function parseSize($size) {
    $unit = preg_replace('/[^bkmgtpezy]/i', '', $size); // Remove the non-unit characters from the size.
    $size = preg_replace('/[^0-9\.]/', '', $size); // Remove the non-numeric characters from the size.
    if ($unit) {
        // Find the position of the unit in the ordered string which is the power of magnitude to multiply a kilobyte by.
        return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
    } else {
        return round($size);
    }
}

function getDatabaseSchema($auth) {
    error_log("=== getDatabaseSchema called ===");
    $pdo = getDbConnection();

    try {
        // Get all tables in the database
        $tablesStmt = $pdo->query("SELECT name FROM sqlite_master WHERE type='table' AND name NOT LIKE 'sqlite_%' ORDER BY name");
        $tableNames = $tablesStmt->fetchAll(PDO::FETCH_COLUMN);
        error_log("Found tables: " . implode(', ', $tableNames));

        $schema = [];

        foreach ($tableNames as $tableName) {
            // Get table structure
            $columnsStmt = $pdo->query("PRAGMA table_info($tableName)");
            $columns = $columnsStmt->fetchAll(PDO::FETCH_ASSOC);

            // Get foreign keys
            $fkStmt = $pdo->query("PRAGMA foreign_key_list($tableName)");
            $foreignKeys = $fkStmt->fetchAll(PDO::FETCH_ASSOC);

            $tableColumns = [];
            foreach ($columns as $col) {
                $column = [
                    'name' => $col['name'],
                    'type' => $col['type'],
                    'pk' => (bool)$col['pk'],
                    'notnull' => (bool)$col['notnull'],
                    'default' => $col['dflt_value']
                ];

                // Check if this column is a foreign key
                foreach ($foreignKeys as $fk) {
                    if ($fk['from'] === $col['name']) {
                        $column['fk'] = [
                            'table' => $fk['table'],
                            'column' => $fk['to']
                        ];
                        break;
                    }
                }

                $tableColumns[] = $column;
            }

            $schema[] = [
                'name' => $tableName,
                'columns' => $tableColumns
            ];
        }

        // Add dynamic category tables (these are virtual relationships via category_fields)
        $categoriesStmt = $pdo->query("SELECT id, name FROM categories ORDER BY name");
        $categories = $categoriesStmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($categories as $category) {
            // Get fields for this category
            $fieldsStmt = $pdo->prepare("
                SELECT name, data_type, is_required, foreign_key_table, foreign_key_column
                FROM category_fields
                WHERE category_id = ?
                ORDER BY created_at
            ");
            $fieldsStmt->execute([$category['id']]);
            $fields = $fieldsStmt->fetchAll(PDO::FETCH_ASSOC);

            $categoryColumns = [
                ['name' => 'id', 'type' => 'INTEGER', 'pk' => true],
                ['name' => 'slug', 'type' => 'TEXT', 'notnull' => true]
            ];

            foreach ($fields as $field) {
                $col = [
                    'name' => $field['name'],
                    'type' => strtoupper($field['data_type']),
                    'notnull' => (bool)$field['is_required']
                ];

                // Add foreign key if defined
                if ($field['foreign_key_table'] && $field['foreign_key_column']) {
                    $col['fk'] = [
                        'table' => $field['foreign_key_table'],
                        'column' => $field['foreign_key_column']
                    ];
                }

                $categoryColumns[] = $col;
            }

            $schema[] = [
                'name' => $category['name'] . ' (virtual)',
                'columns' => $categoryColumns,
                'virtual' => true,
                'category_id' => $category['id']
            ];
        }

        error_log("Schema generated successfully. Total tables: " . count($schema));
        jsonResponse(true, $schema);

    } catch (Exception $e) {
        error_log("Error getting database schema: " . $e->getMessage());
        error_log("Error trace: " . $e->getTraceAsString());
        jsonResponse(false, "Failed to get database schema: " . $e->getMessage());
    }
}

// Language Management Functions
function getLanguages($auth) {
    $pdo = getDbConnection();

    try {
        $stmt = $pdo->query("SELECT * FROM languages ORDER BY is_default DESC, name ASC");
        $languages = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Ensure boolean fields are integers
        foreach ($languages as &$lang) {
            $lang['is_active'] = (int)$lang['is_active'];
            $lang['is_default'] = (int)$lang['is_default'];
        }

        jsonResponse(true, $languages);
    } catch (Exception $e) {
        error_log("Error getting languages: " . $e->getMessage());
        jsonResponse(false, "Failed to get languages: " . $e->getMessage());
    }
}

function createLanguage($data, $auth) {
    validateRequired($data, ['code', 'name']);

    $pdo = getDbConnection();

    // Validate language code format (ISO 639-1 or custom)
    $code = strtolower(trim($data['code']));
    if (!preg_match('/^[a-z]{2,10}$/', $code)) {
        jsonResponse(false, "Language code must be 2-10 lowercase letters (e.g., 'en', 'fr', 'es')");
    }

    // Check if language code already exists
    $stmt = $pdo->prepare("SELECT id FROM languages WHERE code = ?");
    $stmt->execute([$code]);
    if ($stmt->fetch()) {
        jsonResponse(false, "Language code already exists. Please use a unique code.");
    }

    // Validate text direction
    $textDirection = isset($data['text_direction']) ? $data['text_direction'] : 'ltr';
    if (!in_array($textDirection, ['ltr', 'rtl'])) {
        $textDirection = 'ltr';
    }

    // If this is marked as default, unset other defaults
    $isDefault = isset($data['is_default']) && $data['is_default'] ? 1 : 0;
    $isActive = isset($data['is_active']) ? ($data['is_active'] ? 1 : 0) : 1; // Default to active

    if ($isDefault) {
        $pdo->exec("UPDATE languages SET is_default = 0");
    }

    try {
        $stmt = $pdo->prepare("INSERT INTO languages (code, name, text_direction, is_active, is_default) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$code, $data['name'], $textDirection, $isActive, $isDefault]);
        $languageId = $pdo->lastInsertId();

        $language = [
            'id' => $languageId,
            'code' => $code,
            'name' => $data['name'],
            'text_direction' => $textDirection,
            'is_active' => $isActive,
            'is_default' => $isDefault,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];

        jsonResponse(true, $language);
    } catch (Exception $e) {
        error_log("Error creating language: " . $e->getMessage());
        jsonResponse(false, "Failed to create language: " . $e->getMessage());
    }
}

function updateLanguage($languageId, $data, $auth) {
    validateRequired($data, ['name']);

    $pdo = getDbConnection();

    // Check if language exists
    $stmt = $pdo->prepare("SELECT * FROM languages WHERE id = ?");
    $stmt->execute([$languageId]);
    $language = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$language) {
        jsonResponse(false, "Language not found", 404);
    }

    // If code is being updated, validate it
    $code = isset($data['code']) ? strtolower(trim($data['code'])) : $language['code'];
    if ($code !== $language['code']) {
        if (!preg_match('/^[a-z]{2,10}$/', $code)) {
            jsonResponse(false, "Language code must be 2-10 lowercase letters");
        }

        // Check if new code already exists
        $stmt = $pdo->prepare("SELECT id FROM languages WHERE code = ? AND id != ?");
        $stmt->execute([$code, $languageId]);
        if ($stmt->fetch()) {
            jsonResponse(false, "Language code already exists");
        }
    }

    // Validate text direction with fallback
    $textDirection = isset($data['text_direction']) ? $data['text_direction'] : (isset($language['text_direction']) ? $language['text_direction'] : 'ltr');
    if (!in_array($textDirection, ['ltr', 'rtl'])) {
        $textDirection = 'ltr';
    }

    // If this is marked as default, unset other defaults
    $isDefault = isset($data['is_default']) && $data['is_default'] ? 1 : 0;
    $isActive = isset($data['is_active']) ? ($data['is_active'] ? 1 : 0) : (isset($language['is_active']) ? $language['is_active'] : 1);

    if ($isDefault) {
        $pdo->exec("UPDATE languages SET is_default = 0");
    }

    try {
        $stmt = $pdo->prepare("UPDATE languages SET code = ?, name = ?, text_direction = ?, is_active = ?, is_default = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?");
        $stmt->execute([$code, $data['name'], $textDirection, $isActive, $isDefault, $languageId]);

        $updatedLanguage = [
            'id' => $languageId,
            'code' => $code,
            'name' => $data['name'],
            'text_direction' => $textDirection,
            'is_active' => $isActive,
            'is_default' => $isDefault,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        jsonResponse(true, $updatedLanguage);
    } catch (Exception $e) {
        error_log("Error updating language: " . $e->getMessage());
        jsonResponse(false, "Failed to update language: " . $e->getMessage());
    }
}

function deleteLanguage($languageId, $auth) {
    $pdo = getDbConnection();

    // Check if language exists
    $stmt = $pdo->prepare("SELECT * FROM languages WHERE id = ?");
    $stmt->execute([$languageId]);
    $language = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$language) {
        jsonResponse(false, "Language not found", 404);
    }

    // Prevent deletion of default language
    if ($language['is_default']) {
        jsonResponse(false, "Cannot delete the default language. Please set another language as default first.", 400);
    }

    // Check if there are translations using this language
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM record_translations WHERE language_id = ?");
    $stmt->execute([$languageId]);
    $translationCount = $stmt->fetchColumn();

    try {
        if ($translationCount > 0) {
            // Delete all translations for this language
            $stmt = $pdo->prepare("DELETE FROM record_translations WHERE language_id = ?");
            $stmt->execute([$languageId]);
        }

        $stmt = $pdo->prepare("DELETE FROM languages WHERE id = ?");
        $stmt->execute([$languageId]);

        jsonResponse(true, ["message" => "Language deleted successfully", "deleted_translations" => $translationCount]);
    } catch (Exception $e) {
        error_log("Error deleting language: " . $e->getMessage());
        jsonResponse(false, "Failed to delete language: " . $e->getMessage());
    }
}

// Translation Management Functions
function getRecordTranslations($recordId, $auth) {
    $pdo = getDbConnection();

    try {
        // Get all translations for this record grouped by language
        $stmt = $pdo->prepare("
            SELECT
                rt.id,
                rt.language_id,
                l.code as language_code,
                l.name as language_name,
                rt.field_id,
                cf.name as field_name,
                rt.value
            FROM record_translations rt
            JOIN languages l ON rt.language_id = l.id
            JOIN category_fields cf ON rt.field_id = cf.id
            WHERE rt.record_id = ?
            ORDER BY l.name, cf.name
        ");
        $stmt->execute([$recordId]);
        $translations = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Group by language
        $grouped = [];
        foreach ($translations as $translation) {
            $langCode = $translation['language_code'];
            if (!isset($grouped[$langCode])) {
                $grouped[$langCode] = [
                    'language_id' => $translation['language_id'],
                    'language_code' => $langCode,
                    'language_name' => $translation['language_name'],
                    'fields' => []
                ];
            }
            $grouped[$langCode]['fields'][$translation['field_name']] = $translation['value'];
        }

        jsonResponse(true, $grouped);
    } catch (Exception $e) {
        error_log("Error getting record translations: " . $e->getMessage());
        jsonResponse(false, "Failed to get translations: " . $e->getMessage());
    }
}

function saveRecordTranslations($recordId, $data, $auth) {
    validateRequired($data, ['language_code', 'translations']);

    $pdo = getDbConnection();

    // Check if record exists
    $stmt = $pdo->prepare("SELECT category_id FROM category_records WHERE id = ?");
    $stmt->execute([$recordId]);
    $record = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$record) {
        jsonResponse(false, "Record not found", 404);
    }

    // Get or create language by code
    $languageCode = strtolower(trim($data['language_code']));

    // Validate language code format
    if (!preg_match('/^[a-z]{2,10}$/', $languageCode)) {
        jsonResponse(false, "Language code must be 2-10 lowercase letters");
    }

    $stmt = $pdo->prepare("SELECT id FROM languages WHERE code = ?");
    $stmt->execute([$languageCode]);
    $language = $stmt->fetch(PDO::FETCH_ASSOC);

    // Get is_default flag from request
    $isDefault = isset($data['is_default']) && $data['is_default'] ? 1 : 0;

    if (!$language) {
        // Auto-create the language with a generated name
        $languageName = strtoupper($languageCode); // e.g., "fr" -> "FR"

        // If this is set as default, unset other defaults
        if ($isDefault) {
            $pdo->exec("UPDATE languages SET is_default = 0");
        }

        $stmt = $pdo->prepare("INSERT INTO languages (code, name, is_default) VALUES (?, ?, ?)");
        $stmt->execute([$languageCode, $languageName, $isDefault]);
        $languageId = $pdo->lastInsertId();
        error_log("Auto-created language: $languageCode with ID: $languageId, is_default: $isDefault");
    } else {
        $languageId = $language['id'];

        // Update is_default status if changed
        if ($isDefault) {
            // Unset other defaults first
            $pdo->exec("UPDATE languages SET is_default = 0");
        }

        $stmt = $pdo->prepare("UPDATE languages SET is_default = ? WHERE id = ?");
        $stmt->execute([$isDefault, $languageId]);
        error_log("Updated language $languageCode is_default to: $isDefault");
    }

    // Get text-based fields for this category
    $stmt = $pdo->prepare("
        SELECT id, name, data_type
        FROM category_fields
        WHERE category_id = ?
        AND data_type IN ('text', 'textarea', 'rich_paragraph')
    ");
    $stmt->execute([$record['category_id']]);
    $fields = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $fieldMap = [];
    foreach ($fields as $field) {
        $fieldMap[$field['name']] = $field['id'];
    }

    $pdo->beginTransaction();

    try {
        $savedCount = 0;

        foreach ($data['translations'] as $fieldName => $value) {
            // Skip if field doesn't exist or isn't translatable
            if (!isset($fieldMap[$fieldName])) {
                continue;
            }

            $fieldId = $fieldMap[$fieldName];

            // Insert or update translation
            $stmt = $pdo->prepare("
                INSERT INTO record_translations (record_id, language_id, field_id, value, updated_at)
                VALUES (?, ?, ?, ?, CURRENT_TIMESTAMP)
                ON CONFLICT(record_id, language_id, field_id)
                DO UPDATE SET value = ?, updated_at = CURRENT_TIMESTAMP
            ");
            $stmt->execute([$recordId, $languageId, $fieldId, $value, $value]);
            $savedCount++;
        }

        $pdo->commit();

        jsonResponse(true, [
            "message" => "Translations saved successfully",
            "saved_count" => $savedCount,
            "language_code" => $data['language_code']
        ]);
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error saving translations: " . $e->getMessage());
        jsonResponse(false, "Failed to save translations: " . $e->getMessage());
    }
}

function deleteFile($fileId, $auth) {
    $pdo = getDbConnection();

    // Get file info first
    $stmt = $pdo->prepare("SELECT * FROM uploaded_files WHERE id = ?");
    $stmt->execute([$fileId]);
    $file = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$file) {
        jsonResponse(false, "File not found");
    }

    $pdo->beginTransaction();

    try {
        // Clean up file references from record values before deleting the file
        $cleanupResult = cleanupFileReferencesFromRecords($pdo, $file);

        // Delete physical file (only for uploaded files, not URL-based files)
        if (!empty($file['upload_path']) &&
            !in_array($file['file_type'], ['video/url', 'image/url']) &&
            file_exists($file['upload_path'])) {
            unlink($file['upload_path']);
        }

        // Delete from database
        $stmt = $pdo->prepare("DELETE FROM uploaded_files WHERE id = ?");
        $stmt->execute([$fileId]);

        $pdo->commit();

        $message = "File deleted successfully";
        if ($cleanupResult['cleaned_records'] > 0) {
            $message .= ". Removed file reference from {$cleanupResult['cleaned_records']} record(s)";
        }

        jsonResponse(true, $message);

    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error deleting file: " . $e->getMessage());
        jsonResponse(false, "Failed to delete file: " . $e->getMessage());
    }
}

function cleanupFileReferencesFromRecords($pdo, $file) {
    $cleanedRecords = 0;

    // Build possible file URL patterns to search for
    $searchPatterns = [];

    // Add the filename pattern
    $searchPatterns[] = $file['filename'];

    // Add the full upload path if it exists
    if (!empty($file['upload_path'])) {
        $searchPatterns[] = $file['upload_path'];
    }

    // Add constructed URL patterns
    $baseUrl = getBaseUrl();
    if (!empty($file['upload_path'])) {
        $searchPatterns[] = $baseUrl . '/' . $file['upload_path'];
        $searchPatterns[] = constructMediaUrl($file['upload_path'], $file['width'], $file['height']);
    }

    // Add original URL for URL-based files
    if (!empty($file['original_url'])) {
        $searchPatterns[] = $file['original_url'];
    }

    // Remove duplicates
    $searchPatterns = array_unique($searchPatterns);

    // Find and clean up records that reference this file
    foreach ($searchPatterns as $pattern) {
        // Find record values that contain this file reference
        $stmt = $pdo->prepare("
            SELECT rv.id, rv.record_id, rv.field_id, rv.value, f.name as field_name, f.data_type
            FROM category_record_values rv
            JOIN category_fields f ON rv.field_id = f.id
            WHERE rv.value LIKE ?
        ");
        $stmt->execute(["%{$pattern}%"]);
        $recordValues = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($recordValues as $recordValue) {
            $cleanedValue = cleanFileReferenceFromValue($recordValue['value'], $pattern, $recordValue['data_type']);

            if ($cleanedValue !== $recordValue['value']) {
                // Update the record value with the cleaned value
                if (empty($cleanedValue)) {
                    // If the value becomes empty, delete the record value
                    $updateStmt = $pdo->prepare("DELETE FROM category_record_values WHERE id = ?");
                    $updateStmt->execute([$recordValue['id']]);
                } else {
                    // Update with the cleaned value
                    $updateStmt = $pdo->prepare("UPDATE category_record_values SET value = ? WHERE id = ?");
                    $updateStmt->execute([$cleanedValue, $recordValue['id']]);
                }
                $cleanedRecords++;
            }
        }
    }

    return ['cleaned_records' => $cleanedRecords];
}

function cleanFileReferenceFromValue($value, $filePattern, $dataType) {
    // Handle different field types
    switch ($dataType) {
        case 'image':
        case 'file':
        case 'media':
            // These fields can contain comma-separated file references
            if (strpos($value, ',') !== false) {
                // Handle comma-separated values
                $values = array_map('trim', explode(',', $value));
                $cleanedValues = [];

                foreach ($values as $val) {
                    // Keep values that don't contain the file pattern
                    if (strpos($val, $filePattern) === false) {
                        $cleanedValues[] = $val;
                    }
                }

                return implode(',', $cleanedValues);
            } else {
                // Single value - remove if it contains the pattern
                if (strpos($value, $filePattern) !== false) {
                    return '';
                }
                return $value;
            }

        case 'json':
            // Try to decode JSON and clean file references
            $decoded = json_decode($value, true);
            if ($decoded !== null) {
                $cleaned = cleanFileReferencesFromArray($decoded, $filePattern);
                return json_encode($cleaned);
            }
            // Fall through to default if not valid JSON

        case 'array':
            // Handle array values stored as JSON
            $decoded = json_decode($value, true);
            if ($decoded !== null && is_array($decoded)) {
                $cleaned = cleanFileReferencesFromArray($decoded, $filePattern);
                return json_encode($cleaned);
            }
            // Fall through to default if not valid JSON array

        default:
            // For other field types, remove the entire value if it contains the pattern
            if (strpos($value, $filePattern) !== false) {
                return '';
            }
            return $value;
    }
}

function cleanFileReferencesFromArray($array, $filePattern) {
    $cleaned = [];

    foreach ($array as $key => $value) {
        if (is_array($value)) {
            $cleanedSubArray = cleanFileReferencesFromArray($value, $filePattern);
            if (!empty($cleanedSubArray)) {
                $cleaned[$key] = $cleanedSubArray;
            }
        } else if (is_string($value)) {
            // Keep values that don't contain the file pattern
            if (strpos($value, $filePattern) === false) {
                $cleaned[$key] = $value;
            }
        } else {
            // Keep non-string values as is
            $cleaned[$key] = $value;
        }
    }

    return $cleaned;
}

// Helper Functions
function getBaseUrl() {
    // Don't cache the base URL - always use current request context
    // Determine protocol more reliably
    $isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ||
               (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') ||
               (!empty($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] === 'on') ||
               (isset($_SERVER['SERVER_PORT']) && (int)$_SERVER['SERVER_PORT'] === 443);

    $protocol = $isHttps ? 'https' : 'http';

    // Use HTTP_HOST first as it reflects the actual requested domain
    // Fall back to SERVER_NAME, then to a more reliable default
    $host = '';
    if (!empty($_SERVER['HTTP_HOST'])) {
        $host = $_SERVER['HTTP_HOST'];
    } elseif (!empty($_SERVER['SERVER_NAME'])) {
        $host = $_SERVER['SERVER_NAME'];
    } else {
        // Try to construct from current URL context
        if (isset($_SERVER['REQUEST_URI']) && isset($_SERVER['HTTP_REFERER'])) {
            $parsed = parse_url($_SERVER['HTTP_REFERER']);
            if ($parsed && isset($parsed['host'])) {
                $host = $parsed['host'];
            }
        }

        // Final fallback - use localhost only if nothing else is available
        if (empty($host)) {
            $host = 'localhost';
        }
    }

    // Normalize path - get the directory of the current script
    $scriptPath = $_SERVER['SCRIPT_NAME'] ?? '';
    $path = dirname($scriptPath);

    // Remove trailing slash and normalize
    $path = rtrim($path, '/\\');
    if ($path === '.' || $path === '') {
        $path = '';
    }

    return $protocol . '://' . $host . $path;
}

function constructMediaUrl($uploadPath, $width = null, $height = null) {
    // Validate and normalize the upload path
    $uploadPath = normalizeUploadPath($uploadPath);

    if (!$uploadPath) {
        return '';
    }

    $baseUrl = getBaseUrl();

    // Ensure proper URL construction
    $url = $baseUrl . '/' . ltrim($uploadPath, '/');

    // Add dimensions as query parameters if available
    if ($width !== null && $height !== null && $width > 0 && $height > 0) {
        $url .= '?w=' . (int)$width . '&h=' . (int)$height;
    }

    return $url;
}

function normalizeUploadPath($path) {
    if (empty($path)) {
        return '';
    }

    // Convert backslashes to forward slashes
    $path = str_replace('\\', '/', $path);

    // Remove any directory traversal attempts
    $path = str_replace(['../', '../', './'], '', $path);

    // Ensure the path is within the uploads directory
    if (strpos($path, 'uploads/') !== 0) {
        // If it's just a filename, prepend uploads/
        if (basename($path) === $path) {
            $path = 'uploads/' . $path;
        }
    }

    return $path;
}

function enhanceMediaUrlsWithDimensions($value) {
    // Handle single URL
    if (is_string($value) && (strpos($value, '/uploads/') !== false || strpos($value, 'http') === 0)) {
        return enhanceSingleMediaUrl($value);
    }

    // Handle array of URLs
    if (is_array($value)) {
        return array_map(function($url) {
            if (is_string($url) && (strpos($url, '/uploads/') !== false || strpos($url, 'http') === 0)) {
                return enhanceSingleMediaUrl($url);
            }
            return $url;
        }, $value);
    }

    return $value;
}

function enhanceSingleMediaUrl($url) {
    static $fileCache = [];

    // Skip external URLs - only enhance local upload URLs
    if (!$url || strpos($url, 'http') === 0) {
        return $url;
    }

    // Extract filename from URL path
    $parsedUrl = parse_url($url);
    if (!$parsedUrl || !isset($parsedUrl['path'])) {
        return $url;
    }

    $filename = basename($parsedUrl['path']);

    // Skip if filename is empty or doesn't look like an uploaded file
    if (!$filename || strpos($filename, '.') === false) {
        return $url;
    }

    // Check cache first
    if (isset($fileCache[$filename])) {
        $fileData = $fileCache[$filename];
    } else {
        // Query database for file dimensions
        try {
            $pdo = getDbConnection();
            $stmt = $pdo->prepare("SELECT upload_path, width, height FROM uploaded_files WHERE filename = ? LIMIT 1");
            $stmt->execute([$filename]);
            $fileData = $stmt->fetch(PDO::FETCH_ASSOC);

            // Cache the result (even if null to avoid repeated queries)
            $fileCache[$filename] = $fileData ?: false;
        } catch (Exception $e) {
            error_log("Error enhancing media URL: " . $e->getMessage());
            return $url;
        }
    }

    // If we found the file and it has dimensions, construct proper URL
    if ($fileData && $fileData['upload_path']) {
        return constructMediaUrl(
            $fileData['upload_path'],
            $fileData['width'],
            $fileData['height']
        );
    }

    // Return original URL if no enhancement possible
    return $url;
}

function extractMP4Dimensions($filePath) {
    // Basic MP4 dimension extraction by reading file headers
    // This is a simplified implementation for MP4 files

    if (!file_exists($filePath) || !is_readable($filePath)) {
        return null;
    }

    try {
        $handle = fopen($filePath, 'rb');
        if (!$handle) {
            return null;
        }

        // Read first 8KB of the file to look for dimension metadata
        $data = fread($handle, 8192);
        fclose($handle);

        if (strlen($data) < 100) {
            return null;
        }

        // Look for 'moov' atom and video track header
        $moovPos = strpos($data, 'moov');
        if ($moovPos !== false) {
            // Look for track header atom 'tkhd' after moov
            $tkhdPos = strpos($data, 'tkhd', $moovPos);
            if ($tkhdPos !== false && $tkhdPos + 84 < strlen($data)) {
                // MP4 track header contains width and height at specific offsets
                // This is a simplified approach - full MP4 parsing would be more complex
                $width = unpack('N', substr($data, $tkhdPos + 76, 4))[1] >> 16;
                $height = unpack('N', substr($data, $tkhdPos + 80, 4))[1] >> 16;

                if ($width > 0 && $height > 0 && $width <= 8192 && $height <= 8192) {
                    return ['width' => $width, 'height' => $height];
                }
            }
        }

        // Alternative approach: look for STSD atom which contains video dimensions
        $stsdPos = strpos($data, 'stsd');
        if ($stsdPos !== false && $stsdPos + 32 < strlen($data)) {
            // Skip STSD header and look for video codec data
            $offset = $stsdPos + 16;
            if ($offset + 16 < strlen($data)) {
                $width = unpack('n', substr($data, $offset + 8, 2))[1];
                $height = unpack('n', substr($data, $offset + 10, 2))[1];

                if ($width > 0 && $height > 0 && $width <= 8192 && $height <= 8192) {
                    return ['width' => $width, 'height' => $height];
                }
            }
        }

    } catch (Exception $e) {
        error_log("MP4 dimension extraction error: " . $e->getMessage());
    }

    return null;
}

function jsonResponse($success, $data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    
    $response = [
        'success' => $success,
        'data' => $data
    ];
    
    // Use JSON_UNESCAPED_SLASHES to prevent escaping of forward slashes
    echo json_encode($response, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit;
}

function getJsonInput() {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        jsonResponse(false, "Invalid JSON input");
    }
    return $data;
}

function validateRequired($data, $required) {
    foreach ($required as $field) {
        if (!isset($data[$field]) || empty($data[$field])) {
            jsonResponse(false, "Field '$field' is required");
        }
    }
}

function validateSlugFormat($slug) {
    if (empty($slug)) {
        return true; // Empty slug is allowed
    }

    // Slug can contain letters (uppercase/lowercase), numbers, and hyphens only
    // Pattern: only letters, numbers, and hyphens allowed
    // Must not start or end with hyphen
    // No consecutive hyphens
    if (!preg_match('/^[a-zA-Z0-9]+(-[a-zA-Z0-9]+)*$/', $slug)) {
        jsonResponse(false, "Slug can contain letters, numbers, and hyphens only. Format: 'my-article-slug' or 'My-Article-Slug'. No spaces, no special characters except hyphens, cannot start or end with hyphen.");
    }

    // Additional check: no consecutive hyphens
    if (strpos($slug, '--') !== false) {
        jsonResponse(false, "Slug cannot contain consecutive hyphens. Format: 'my-article-slug' or 'My-Article-Slug'");
    }

    return true;
}

// Frontend Display
function displayFrontend() {
?><!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Single File CMS</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {
                    colors: {
                        primary: {
                            50: '#eff6ff',
                            100: '#dbeafe',
                            200: '#bfdbfe',
                            300: '#93c5fd',
                            400: '#60a5fa',
                            500: '#4945ff',
                            600: '#3b82f6',
                            700: '#1d4ed8',
                            800: '#1e40af',
                            900: '#1e3a8a',
                        }
                    }
                }
            }
        }
    </script>
    <style>
        /* Modern Color Palette - Solid Colors */
        :root {
            --primary: #6366f1;
            --primary-dark: #4f46e5;
            --primary-light: #818cf8;
            --success: #10b981;
            --success-dark: #059669;
            --danger: #ef4444;
            --danger-dark: #dc2626;
            --warning: #f59e0b;
            --warning-dark: #d97706;
            --info: #3b82f6;
            --info-dark: #2563eb;
            --gray-50: #f9fafb;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-300: #d1d5db;
            --gray-700: #374151;
            --gray-800: #1f2937;
            --gray-900: #111827;
        }

        /* Custom Scrollbar */
        ::-webkit-scrollbar {
            width: 10px;
            height: 10px;
        }

        ::-webkit-scrollbar-track {
            background: var(--gray-100);
            border-radius: 5px;
        }

        ::-webkit-scrollbar-thumb {
            background: var(--primary);
            border-radius: 5px;
        }

        ::-webkit-scrollbar-thumb:hover {
            background: var(--primary-dark);
        }

        /* Animations */
        @keyframes slideInRight {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }

        .animate-fade-in {
            animation: fadeIn 0.3s ease-out;
        }

        .animate-slide-up {
            animation: slideUp 0.4s ease-out;
        }

        /* Card Enhancements */
        .card-hover {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }

        .card-hover:hover {
            transform: translateY(-4px);
            box-shadow: 0 12px 24px rgba(0, 0, 0, 0.12);
        }

        /* Button Enhancements */
        .btn-primary {
            background: var(--primary);
            transition: all 0.2s ease;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
            box-shadow: 0 8px 16px rgba(99, 102, 241, 0.3);
        }

        .btn-primary:active {
            transform: translateY(0);
        }

        /* Table Enhancements */
        tbody tr {
            transition: all 0.2s ease;
        }

        tbody tr:hover {
            background: var(--gray-50);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
        }

        /* Checkbox Enhancements */
        input[type="checkbox"]:checked {
            background: var(--primary);
            border-color: var(--primary);
        }

        input[type="checkbox"]:hover {
            border-color: var(--primary-light);
        }

        /* Sidebar Enhancement */
        .sidebar-dark {
            background: var(--gray-900);
        }

        .nav-item {
            position: relative;
            overflow: hidden;
            color: #d1fae5;
        }

        .nav-item::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            height: 100%;
            width: 4px;
            background: var(--primary);
            transform: scaleY(0);
            transition: transform 0.3s ease;
        }

        .nav-item:hover {
            background: rgba(255, 227, 227, 0.08);
        }

        .nav-item.active {
            background: rgba(255, 255, 255, 0.12);
        }

        .nav-item.active::before,
        .nav-item:hover::before {
            transform: scaleY(1);
        }

        /* Badge Styles */
        .badge {
            font-size: 0.75rem;
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-weight: 600;
        }

        .badge-primary {
            background: #dbeafe;
            color: #1e40af;
        }

        .badge-success {
            background: #d1fae5;
            color: #065f46;
        }

        .badge-danger {
            background: #fee2e2;
            color: #991b1b;
        }

        .badge-warning {
            background: #fef3c7;
            color: #92400e;
        }

        .suggestion-active {
            background-color: var(--gray-100) !important;
        }

        .suggestion-item:hover .suggestion-active {
            background-color: var(--gray-200) !important;
        }

        mark {
            background-color: #fef3c7;
            padding: 1px 2px;
            border-radius: 2px;
        }

        .media-filter-btn {
            transition: all 0.2s ease;
        }

        .media-filter-btn.active {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
            box-shadow: 0 4px 12px rgba(99, 102, 241, 0.3);
        }

        .media-filter-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .media-filter-btn.active:hover {
            background: var(--primary-dark);
        }

        /* Carousel Styles */
        .carousel-container {
            position: relative;
            width: 100%;
            max-width: 1200px;
            overflow: hidden;
            margin: 0 auto;
        }

        .carousel-wrapper {
            overflow: hidden;
            width: 100%;
        }

        .carousel {
            display: flex;
            gap: 20px;
            transition: transform 0.5s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            padding: 20px 0;
        }

        .carousel-card {
            border-radius: 16px;
            width: 350px;
            height: 250px;
            position: relative;
            overflow: hidden;
            cursor: pointer;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            flex-shrink: 0;
        }

        .carousel-card img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            display: block;
        }

        .carousel-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0, 0, 0, 0.4);
        }

        .carousel-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(0, 0, 0, 0.1) 0%, rgba(0, 0, 0, 0.3) 100%);
            opacity: 0;
            transition: opacity 0.3s ease;
            z-index: 1;
        }

        .carousel-card:hover::before {
            opacity: 1;
        }

        .link-icon {
            position: absolute;
            top: 20px;
            right: 20px;
            width: 40px;
            height: 40px;
            background: rgba(255, 255, 255, 0.9);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
            z-index: 10;
            opacity: 0;
            transform: translateY(-10px);
        }

        .carousel-card:hover .link-icon {
            opacity: 1;
            transform: translateY(0);
        }

        .link-icon:hover {
            background: white;
            transform: scale(1.1);
        }

        .link-icon svg {
            width: 20px;
            height: 20px;
            stroke: #333;
            stroke-width: 2;
        }

        .nav-btn {
            background: rgba(255, 255, 255, 0.1);
            border: 2px solid rgba(255, 255, 255, 0.2);
            color: white;
            width: 50px;
            height: 50px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s ease;
            backdrop-filter: blur(10px);
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            z-index: 5;
        }

        .nav-btn:hover:not(:disabled) {
            background: rgba(255, 255, 255, 0.2);
            border-color: rgba(255, 255, 255, 0.4);
        }

        .nav-btn.prev {
            left: -70px;
        }

        .nav-btn.next {
            right: -70px;
        }

        .nav-btn:disabled {
            opacity: 0.3;
            cursor: not-allowed;
        }

        .dots {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 8px;
            margin-top: 30px;
        }

        .dot {
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.25);
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .dot.center {
            width: 10px;
            height: 10px;
            background: white;
        }

        .dot.adjacent {
            width: 8px;
            height: 8px;
            background: rgba(255, 255, 255, 0.4);
        }

        .dot.outer {
            width: 6px;
            height: 6px;
            background: rgba(255, 255, 255, 0.25);
        }

        .loading-placeholder {
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, #333 25%, #444 50%, #333 75%);
            background-size: 200% 100%;
            animation: loading 1.5s infinite;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 14px;
        }

        @keyframes loading {
            0% { background-position: 200% 0; }
            100% { background-position: -200% 0; }
        }

        @media (max-width: 768px) {
            .carousel {
                gap: 15px;
                padding: 20px;
            }

            .carousel-card {
                width: 300px;
            }

            .nav-btn.prev {
                left: 10px;
            }

            .nav-btn.next {
                right: 10px;
            }
        }

        /* Rich Text Editor Styles */
        .rte-wrapper {
            border: 1px solid #e5e7eb;
            border-radius: 0.75rem;
            overflow: hidden;
            background: white;
        }

        .rte-toolbar {
            display: flex;
            flex-wrap: wrap;
            gap: 0.5rem;
            padding: 0.75rem;
            background: #f9fafb;
            border-bottom: 1px solid #e5e7eb;
            align-items: center;
        }

        .rte-toolbar-group {
            display: flex;
            gap: 0.25rem;
        }

        .rte-separator {
            width: 1px;
            height: 24px;
            background: #e5e7eb;
            margin: 0 0.25rem;
        }

        .rte-toolbar-btn {
            padding: 0.375rem 0.625rem;
            background: white;
            border: 1px solid #e5e7eb;
            border-radius: 0.375rem;
            cursor: pointer;
            font-size: 0.875rem;
            font-weight: 500;
            transition: all 0.15s;
            color: #374151;
            min-width: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .rte-toolbar-btn:hover {
            background: #f3f4f6;
            border-color: #d1d5db;
        }

        .rte-toolbar-btn:active {
            background: #e5e7eb;
            transform: scale(0.95);
        }

        .rte-toolbar-btn:focus {
            outline: 2px solid #3b82f6;
            outline-offset: 2px;
        }

        .rte-editor {
            padding: 1rem;
            min-height: 400px;
            outline: none;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            font-size: 0.875rem;
            line-height: 1.6;
            color: #1f2937;
            overflow-y: auto;
        }

        .rte-editor:focus {
            background: #fefefe;
        }

        .rte-editor p {
            margin: 0.5rem 0;
        }

        .rte-editor h1 {
            font-size: 2em;
            font-weight: bold;
            margin: 0.67em 0;
        }

        .rte-editor h2 {
            font-size: 1.5em;
            font-weight: bold;
            margin: 0.75em 0;
        }

        .rte-editor h3 {
            font-size: 1.17em;
            font-weight: bold;
            margin: 0.83em 0;
        }

        .rte-editor ul, .rte-editor ol {
            margin: 0.5rem 0;
            padding-left: 2rem;
        }

        .rte-editor ul {
            list-style-type: disc;
        }

        .rte-editor ol {
            list-style-type: decimal;
        }

        .rte-editor li {
            margin: 0.25rem 0;
        }

        .rte-editor a {
            color: #3b82f6;
            text-decoration: underline;
        }

        .rte-editor img {
            max-width: 100%;
            height: auto;
            display: block;
            margin: 0.5rem 0;
        }

        .rte-editor blockquote {
            border-left: 4px solid #e5e7eb;
            padding-left: 1rem;
            margin: 0.5rem 0;
            color: #6b7280;
        }

        .rte-editor code {
            background: #f3f4f6;
            padding: 0.125rem 0.25rem;
            border-radius: 0.25rem;
            font-family: 'Courier New', monospace;
            font-size: 0.875em;
        }

        .rte-editor pre {
            background: #1f2937;
            color: #f9fafb;
            padding: 1rem;
            border-radius: 0.5rem;
            overflow-x: auto;
            margin: 0.5rem 0;
        }

        .rte-editor pre code {
            background: transparent;
            padding: 0;
            color: inherit;
        }

        .rte-editor table {
            border-collapse: collapse;
            width: 100%;
            margin: 0.5rem 0;
            border: 1px solid #e5e7eb;
        }

        .rte-editor th,
        .rte-editor td {
            border: 1px solid #e5e7eb;
            padding: 0.5rem;
            text-align: left;
        }

        .rte-editor th {
            background: #f9fafb;
            font-weight: 600;
        }

        .rte-editor sub {
            vertical-align: sub;
            font-size: smaller;
        }

        .rte-editor sup {
            vertical-align: super;
            font-size: smaller;
        }

        .rte-editor mark {
            background: #fef08a;
            padding: 0.125rem 0.25rem;
        }

        .rte-editor del,
        .rte-editor s {
            text-decoration: line-through;
        }

        .rte-editor ins {
            text-decoration: underline;
            background: #dcfce7;
        }

        .rte-editor small {
            font-size: 0.875em;
        }

        .rte-editor cite {
            font-style: italic;
        }

        .rte-editor abbr {
            text-decoration: underline dotted;
            cursor: help;
        }

        .rte-editor div,
        .rte-editor span {
            /* Allow custom styling from pasted content */
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    <!-- Loading Overlay -->
    <div id="loadingOverlay" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center">
        <div class="bg-white rounded-lg p-6 flex items-center space-x-3">
            <div class="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
            <span class="text-gray-700">Loading...</span>
        </div>
    </div>

    <!-- Authentication Section -->
    <div id="authSection" class="min-h-screen flex items-center justify-center p-4">
        <div class="max-w-md w-full bg-white rounded-lg shadow-lg p-8">
            <div class="text-center mb-8">
                <h1 class="text-3xl font-bold text-gray-900">Single File CMS</h1>
                <p class="text-gray-600 mt-2">Content Management System</p>
            </div>

            <!-- Auth Tabs -->
            <div class="flex mb-6">
                
            </div>

            <!-- Auth Form -->
            <form id="authForm" onsubmit="handleAuth(event)">

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                    <input type="email" id="authEmail" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" value="admin@test.com">
                </div>

                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Password</label>
                    <input type="password" id="authPassword" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" value="password">
                </div>

                <button type="submit" id="authSubmitBtn" class="w-full bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50">
                    <span id="authSpinner" class="hidden animate-spin rounded-full h-4 w-4 border-b-2 border-white inline-block mr-2"></span>
                    <span id="authSubmitText">Sign In</span>
                </button>
            </form>

            <div id="authError" class="hidden mt-4 p-3 bg-red-100 border border-red-300 rounded-md text-red-700 text-sm"></div>
        </div>
    </div>

    <!-- Main Application -->
    <div id="mainApp" class="hidden">
        <div class="flex h-screen bg-gray-50">
            <!-- Sidebar -->
            <div class="w-64 sidebar-dark shadow-2xl">
                <div class="p-6 border-b border-gray-700">
                    <h1 class="text-xl font-bold text-white">CMS Dashboard</h1>
                    <p id="userWelcome" class="text-gray-400 text-sm mt-1"></p>
                </div>
                <nav class="p-4">
                    <div class="space-y-1">
                        <button onclick="showTab('categories')" class="nav-item w-full text-left px-4 py-3 rounded-lg text-gray-300 transition-all">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 010 2.828l-7 7a2 2 0 01-2.828 0l-7-7A1.994 1.994 0 013 12V7a4 4 0 014-4z"></path>
                                </svg>
                                Categories
                            </span>
                        </button>
                        <button onclick="showTab('fields')" class="nav-item w-full text-left px-4 py-3 rounded-lg text-gray-300 transition-all">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                                </svg>
                                Fields
                            </span>
                        </button>
                        <button onclick="showTab('records')" class="nav-item w-full text-left px-4 py-3 rounded-lg text-gray-300 transition-all">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"></path>
                                </svg>
                                Records
                            </span>
                        </button>
                        <button onclick="showTab('media')" class="nav-item w-full text-left px-4 py-3 rounded-lg text-gray-300 transition-all">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
                                </svg>
                                Media
                            </span>
                        </button>
                        <button onclick="showTab('tester')" class="nav-item w-full text-left px-4 py-3 rounded-lg text-gray-300 transition-all">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 20l4-16m4 4l4 4-4 4M6 16l-4-4 4-4"></path>
                                </svg>
                                GET Tester
                            </span>
                        </button>
                        <button onclick="showTab('schema')" class="nav-item w-full text-left px-4 py-3 rounded-lg text-gray-300 transition-all">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 7v10c0 2 1 3 3 3h10c2 0 3-1 3-3V7c0-2-1-3-3-3H7C5 4 4 5 4 7z"></path>
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 4v16M15 4v16M4 9h16M4 15h16"></path>
                                </svg>
                                Database Schema
                            </span>
                        </button>
                        <button onclick="showTab('languages')" class="nav-item w-full text-left px-4 py-3 rounded-lg text-gray-300 transition-all">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 5h12M9 3v2m1.048 9.5A18.022 18.022 0 016.412 9m6.088 9h7M11 21l5-10 5 10M12.751 5C11.783 10.77 8.07 15.61 3 18.129"></path>
                                </svg>
                                Languages
                            </span>
                        </button>

                    </div>
                    <div class="mt-8 pt-4 border-t border-gray-700">
                        <button onclick="logout()" class="w-full text-left px-4 py-3 rounded-lg text-gray-300 hover:bg-red-600 hover:text-white transition-all">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"></path>
                                </svg>
                                Logout
                            </span>
                        </button>
                    </div>
                </nav>
            </div>

            <!-- Main Content -->
            <div class="flex-1 overflow-auto">
                <!-- Categories Tab -->
                <div id="categoriesTab" class="p-6">
                    <div class="flex justify-between items-center mb-6">
                        <h2 class="text-3xl font-bold text-gray-900">Categories</h2>
                        <button onclick="openModal('categoryModal')" class="btn-primary text-white px-6 py-3 rounded-lg font-semibold shadow-lg">
                            <span class="flex items-center">
                                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                                </svg>
                                Add Category
                            </span>
                        </button>
                    </div>

                    <!-- Search -->
                    <div class="mb-6 relative">
                        <input type="text" id="categoriesSearchInput" placeholder="Search categories..." class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" autocomplete="off">
                        <div id="categoriesSearchSuggestions" class="absolute top-full left-0 right-0 bg-white border border-gray-200 rounded-b-lg shadow-lg hidden z-10 max-h-60 overflow-y-auto"></div>
                    </div>

                    <!-- Bulk Actions Bar -->
                    <div id="categoryBulkActionsBar" class="hidden mb-4 bg-indigo-50 border-2 border-indigo-200 rounded-xl p-4 shadow-md animate-fade-in">
                        <div class="flex items-center justify-between">
                            <div class="flex items-center space-x-3">
                                <div class="w-10 h-10 rounded-full bg-indigo-600 flex items-center justify-center text-white font-bold shadow-md">
                                    <span id="categorySelectedCount">0</span>
                                </div>
                                <span class="text-sm font-semibold text-gray-700">items selected</span>
                            </div>
                            <div class="flex items-center space-x-2">
                                <button onclick="bulkDeleteCategories()" class="px-5 py-2.5 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-all font-semibold shadow-md hover:shadow-lg">
                                    <span class="flex items-center">
                                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                                        </svg>
                                        Delete Selected
                                    </span>
                                </button>
                                <button onclick="clearCategorySelection()" class="px-5 py-2.5 bg-white border-2 border-gray-300 rounded-lg hover:bg-gray-50 transition-all font-semibold shadow-sm hover:shadow">
                                    <span class="flex items-center">
                                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                        Clear
                                    </span>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Categories Table -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden border border-gray-200">
                        <table id="categoriesTable" class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-4 text-left">
                                        <input type="checkbox" id="categorySelectAll" onchange="toggleSelectAllCategories()" class="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500 w-5 h-5">
                                    </th>
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Name</th>
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase ">Fields</th>
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Records</th>
                                    <th class="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Created</th>
                                    <th class="px-6 py-4 text-right text-xs font-bold text-gray-700 uppercase tracking-wider">Actions</th>
                                </tr>
                            </thead>
                            <tbody id="categoriesTableBody" class="bg-white divide-y divide-gray-200">
                            </tbody>
                        </table>
                        <div id="categoriesEmpty" class="hidden p-8 text-center">
                            <p class="text-gray-500">No categories found</p>
                        </div>
                    </div>

                    <!-- Pagination -->
                    <div class="flex items-center justify-between mt-4">
                        <div class="text-sm text-gray-700">
                            <span id="categoryPageInfo">Page 1 of 1</span>
                        </div>
                        <div class="flex space-x-2">
                            <button id="categoryPrevBtn" class="px-3 py-2 border border-gray-300 rounded-md text-sm text-gray-500 hover:bg-gray-50 disabled:opacity-50">
                                Previous
                            </button>
                            <button id="categoryNextBtn" class="px-3 py-2 border border-gray-300 rounded-md text-sm text-gray-500 hover:bg-gray-50 disabled:opacity-50">
                                Next
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Fields Tab -->
                <div id="fieldsTab" class="hidden p-6">
                    <div class="flex justify-between items-center mb-6">
                        <h2 class="text-2xl font-bold text-gray-900">Fields</h2>
                        <button id="addFieldBtn" onclick="openModal('fieldModal')" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed" disabled>
                            Add Field
                        </button>
                    </div>

                    <!-- Category Selector -->
                    <div class="mb-6">
                        <label class="block text-sm font-medium text-gray-700 mb-2">Select Category</label>
                        <select id="fieldsCategorySelect" onchange="loadFieldsForCategory()" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="">Choose a category...</option>
                        </select>
                    </div>

                    <!-- Fields Table -->
                    <div class="bg-white rounded-lg shadow overflow-hidden">
                        <table id="fieldsTable" class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Name</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Required</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Default</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                </tr>
                            </thead>
                            <tbody id="fieldsTableBody" class="bg-white divide-y divide-gray-200">
                            </tbody>
                        </table>
                        <div id="fieldsEmpty" class="p-8 text-center">
                            <p class="text-gray-500">Select a category to manage its fields</p>
                        </div>
                    </div>
                </div>

                <!-- Records Tab -->
                <div id="recordsTab" class="hidden p-6">
                    <div class="flex justify-between items-center mb-6">
                        <h2 class="text-2xl font-bold text-gray-900">Records</h2>
                        <div class="flex gap-2">
                            <button onclick="exportDatabaseSQL()" id="exportButton" class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed" disabled>
                                Export Database
                            </button>
                            <button onclick="addNewRecord()" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700">
                                Add Record
                            </button>
                        </div>
                    </div>

                    <!-- Category and Language Selectors -->
                    <div class="mb-6 grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Select Category</label>
                            <select id="recordsCategorySelect" onchange="loadRecordsForCategory()" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="">Choose a category...</option>
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                <svg class="w-4 h-4 inline mr-1" fill="currentColor" viewBox="0 0 20 20">
                                    <path fill-rule="evenodd" d="M7 2a1 1 0 011 1v1h3a1 1 0 110 2H9.578a18.87 18.87 0 01-1.724 4.78c.29.354.596.696.914 1.026a1 1 0 11-1.44 1.389c-.188-.196-.373-.396-.554-.6a19.098 19.098 0 01-3.107 3.567 1 1 0 01-1.334-1.490 17.087 17.087 0 003.13-3.733 18.992 18.992 0 01-1.487-2.494 1 1 0 111.79-.89c.234.47.489.928.764 1.372.417-.934.752-1.913.997-2.927H3a1 1 0 110-2h3V3a1 1 0 011-1zm6 6a1 1 0 01.894.553l2.991 5.982a.869.869 0 01.02.037l.99 1.98a1 1 0 11-1.79.895L15.383 16h-4.764l-.724 1.447a1 1 0 11-1.788-.894l.99-1.98.019-.038 2.99-5.982A1 1 0 0113 8zm-1.382 6h2.764L13 11.236 11.618 14z" clip-rule="evenodd"/>
                                </svg>
                                Display Language
                            </label>
                            <select id="globalLanguageSelect" onchange="changeGlobalLanguage()" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 bg-gradient-to-r from-blue-50 to-purple-50">
                                <option value="">Default Language</option>
                            </select>
                        </div>
                    </div>

                    <!-- Bulk Actions Bar for Records -->
                    <div id="recordBulkActionsBar" class="hidden mb-4 bg-blue-50 border border-blue-200 rounded-lg p-4">
                        <div class="flex items-center justify-between">
                            <div class="flex items-center space-x-2">
                                <span class="text-sm font-medium text-blue-900"><span id="recordSelectedCount">0</span> selected</span>
                            </div>
                            <div class="flex items-center space-x-2">
                                <button onclick="bulkDeleteRecords()" class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors">
                                    Delete Selected
                                </button>
                                <button onclick="clearRecordSelection()" class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                                    Clear Selection
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Search Records -->
                    <div class="mb-6 hidden" id="recordsSearchContainer">
                        <div class="flex flex-col sm:flex-row gap-4">
                            <div class="flex-1">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Search Records</label>
                                <div class="relative">
                                    <input type="text" id="recordsSearchInput" placeholder="Search by slug or content..." class="w-full px-4 py-2 pr-10 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" autocomplete="off">
                                    <div class="absolute inset-y-0 right-0 flex items-center pr-3">
                                        <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                                        </svg>
                                    </div>
                                </div>
                            </div>
                            <div class="sm:w-48">
                                <label class="block text-sm font-medium text-gray-700 mb-2">Search Type</label>
                                <select id="recordsSearchType" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="all">All Fields</option>
                                    <option value="slug">Slug Only</option>
                                    <option value="content">Content Only</option>
                                </select>
                            </div>
                            <div class="flex items-end">
                                <button type="button" onclick="searchRecords()" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    Search
                                </button>
                                <button type="button" onclick="clearRecordsSearch()" class="ml-2 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    Clear
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- View Toggle -->
                    <div class="mb-6 hidden" id="recordsViewToggle">
                        <div class="flex justify-between items-center">
                            <span class="text-sm font-medium text-gray-700">View Mode</span>
                            <div class="flex bg-gray-200 rounded-lg p-1">
                                <button type="button" id="cardViewBtn" onclick="switchRecordsView('card')" class="px-3 py-1 text-sm font-medium rounded-md bg-white text-gray-700 shadow-sm">
                                    <svg class="w-4 h-4 inline-block mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14-7H5a2 2 0 00-2 2v12a2 2 0 002 2h14a2 2 0 002-2V6a2 2 0 00-2-2z"></path>
                                    </svg>
                                    Cards
                                </button>
                                <button type="button" id="tableViewBtn" onclick="switchRecordsView('table')" class="px-3 py-1 text-sm font-medium rounded-md text-gray-500 hover:text-gray-700">
                                    <svg class="w-4 h-4 inline-block mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M3 6h18M3 14h18M3 18h18"></path>
                                    </svg>
                                    Table
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Records Grid (Card View) -->
                    <div id="recordsGrid" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    </div>

                    <!-- Records Table (Table View) -->
                    <div id="recordsTable" class="hidden overflow-x-auto">
                        <table class="min-w-full bg-white border border-gray-200 rounded-lg">
                            <thead id="recordsTableHead" class="bg-gray-50">
                                <!-- Table headers will be generated dynamically -->
                            </thead>
                            <tbody id="recordsTableBody" class="divide-y divide-gray-200">
                                <!-- Table rows will be generated dynamically -->
                            </tbody>
                        </table>
                    </div>

                    <div id="recordsEmpty" class="p-8 text-center">
                        <p class="text-gray-500">Select a category to manage its records</p>
                    </div>

                    <!-- Pagination Controls -->
                    <div id="recordsPagination" class="hidden mt-6 flex items-center justify-between border-t border-gray-200 bg-white px-4 py-3 sm:px-6 rounded-lg shadow">
                        <div class="flex flex-1 justify-between sm:hidden">
                            <button onclick="previousRecordsPage()" id="recordsPrevMobile" class="relative inline-flex items-center rounded-md border border-gray-300 bg-white px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">
                                Previous
                            </button>
                            <button onclick="nextRecordsPage()" id="recordsNextMobile" class="relative ml-3 inline-flex items-center rounded-md border border-gray-300 bg-white px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">
                                Next
                            </button>
                        </div>
                        <div class="hidden sm:flex sm:flex-1 sm:items-center sm:justify-between">
                            <div>
                                <p class="text-sm text-gray-700">
                                    Showing <span class="font-medium" id="recordsShowingStart">1</span> to <span class="font-medium" id="recordsShowingEnd">50</span> of <span class="font-medium" id="recordsTotal">0</span> results
                                </p>
                            </div>
                            <div>
                                <nav class="isolate inline-flex -space-x-px rounded-md shadow-sm" aria-label="Pagination">
                                    <button onclick="previousRecordsPage()" id="recordsPrevDesktop" class="relative inline-flex items-center rounded-l-md px-2 py-2 text-gray-400 ring-1 ring-inset ring-gray-300 hover:bg-gray-50 focus:z-20 focus:outline-offset-0">
                                        <span class="sr-only">Previous</span>
                                        <svg class="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                            <path fill-rule="evenodd" d="M12.79 5.23a.75.75 0 01-.02 1.06L8.832 10l3.938 3.71a.75.75 0 11-1.04 1.08l-4.5-4.25a.75.75 0 010-1.08l4.5-4.25a.75.75 0 011.06.02z" clip-rule="evenodd" />
                                        </svg>
                                    </button>
                                    <span id="recordsPageNumbers" class="relative inline-flex items-center px-4 py-2 text-sm font-semibold text-gray-900 ring-1 ring-inset ring-gray-300">
                                        Page <span id="recordsCurrentPage">1</span> of <span id="recordsTotalPages">1</span>
                                    </span>
                                    <button onclick="nextRecordsPage()" id="recordsNextDesktop" class="relative inline-flex items-center rounded-r-md px-2 py-2 text-gray-400 ring-1 ring-inset ring-gray-300 hover:bg-gray-50 focus:z-20 focus:outline-offset-0">
                                        <span class="sr-only">Next</span>
                                        <svg class="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                            <path fill-rule="evenodd" d="M7.21 14.77a.75.75 0 01.02-1.06L11.168 10 7.23 6.29a.75.75 0 111.04-1.08l4.5 4.25a.75.75 0 010 1.08l-4.5 4.25a.75.75 0 01-1.06-.02z" clip-rule="evenodd" />
                                        </svg>
                                    </button>
                                </nav>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Media Tab -->
                <div id="mediaTab" class="hidden p-6">
                    <div class="flex justify-between items-center mb-6">
                        <h2 class="text-3xl font-bold text-gray-900">Media Gallery</h2>
                        <div class="flex space-x-2">
                            <button onclick="openModal('uploadModal')" class="btn-primary text-white px-6 py-2 rounded-lg font-semibold shadow-lg">
                                <span class="flex items-center">
                                    <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"></path>
                                    </svg>
                                    Upload File
                                </span>
                            </button>
                        </div>
                    </div>

                    <!-- Bulk Actions Bar for Media -->
                    <div id="mediaBulkActionsBar" class="hidden mb-4 bg-indigo-50 border-2 border-indigo-200 rounded-xl p-4 shadow-md animate-fade-in">
                        <div class="flex items-center justify-between">
                            <div class="flex items-center space-x-3">
                                <div class="w-10 h-10 rounded-full bg-indigo-600 flex items-center justify-center text-white font-bold shadow-md">
                                    <span id="mediaSelectedCount">0</span>
                                </div>
                                <span class="text-sm font-semibold text-gray-700">files selected</span>
                            </div>
                            <div class="flex items-center space-x-2">
                                <button onclick="bulkDeleteMedia()" class="px-5 py-2.5 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-all font-semibold shadow-md hover:shadow-lg">
                                    <span class="flex items-center">
                                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                                        </svg>
                                        Delete Selected
                                    </span>
                                </button>
                                <button onclick="clearMediaSelection()" class="px-5 py-2.5 bg-white border-2 border-gray-300 rounded-lg hover:bg-gray-50 transition-all font-semibold shadow-sm hover:shadow">
                                    <span class="flex items-center">
                                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                        Clear
                                    </span>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Media Search and Filters -->
                    <div class="mb-6 space-y-4">
                        <!-- Search Bar -->
                        <div class="flex gap-3">
                            <div class="flex-1 relative">
                                <input type="text" id="mediaSearchInput" placeholder="Search files..." class="w-full px-4 py-2 pr-10 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" autocomplete="off">
                                <div class="absolute inset-y-0 right-0 flex items-center pr-3">
                                    <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                                    </svg>
                                </div>
                            </div>
                            <div class="w-40">
                                <select id="mediaSearchType" onchange="updateSearchPlaceholder()" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="name">Search by Name</option>
                                    <option value="tag">Search by Tag</option>
                                </select>
                            </div>
                            <button type="button" onclick="performMediaSearch()" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                Search
                            </button>
                            <button type="button" onclick="clearMediaSearch()" class="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-blue-500">
                                Clear
                            </button>
                        </div>

                        <!-- Filters -->
                        <div class="flex flex-wrap gap-2 items-center">
                            <button onclick="filterMedia('')" class="media-filter-btn active px-4 py-2 rounded-lg border border-gray-300 hover:bg-gray-50 text-sm font-medium">All Files</button>
                            <button onclick="filterMedia('image')" class="media-filter-btn px-4 py-2 rounded-lg border border-gray-300 hover:bg-gray-50 text-sm font-medium">Images</button>
                            <button onclick="filterMedia('video')" class="media-filter-btn px-4 py-2 rounded-lg border border-gray-300 hover:bg-gray-50 text-sm font-medium">Videos</button>
                            <button onclick="filterMedia('audio')" class="media-filter-btn px-4 py-2 rounded-lg border border-gray-300 hover:bg-gray-50 text-sm font-medium">Audio</button>
                            <button onclick="filterMedia('file')" class="media-filter-btn px-4 py-2 rounded-lg border border-gray-300 hover:bg-gray-50 text-sm font-medium">Documents</button>

                            <div class="border-l pl-2 ml-2">
                                <select id="mediaTagFilter" onchange="filterMediaByTag()" class="px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="">Filter by tag...</option>
                                </select>
                            </div>

                            <button id="clearTagFilter" onclick="clearTagFilter()" class="hidden px-3 py-2 text-red-600 hover:text-red-800 text-sm">Clear tag filter</button>
                        </div>
                    </div>

                    <!-- Media Grid -->
                    <div id="mediaGrid" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                    </div>

                    <!-- Media Pagination -->
                    <div id="mediaPagination" class="hidden mt-6 flex items-center justify-between">
                        <div class="text-sm text-gray-700">
                            Showing <span id="mediaShowingStart">1</span> to <span id="mediaShowingEnd">12</span> of <span id="mediaTotal">0</span> files
                        </div>
                        <div class="flex items-center space-x-2">
                            <button id="mediaPrevBtn" onclick="loadMediaPage('prev')" class="px-3 py-1 border border-gray-300 rounded-md hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed" disabled>
                                Previous
                            </button>
                            <div id="mediaPageNumbers" class="flex items-center space-x-1">
                                <!-- Page numbers will be inserted here -->
                            </div>
                            <button id="mediaNextBtn" onclick="loadMediaPage('next')" class="px-3 py-1 border border-gray-300 rounded-md hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed" disabled>
                                Next
                            </button>
                        </div>
                    </div>

                    <div id="mediaEmpty" class="p-8 text-center">
                        <p class="text-gray-500">No media files found</p>
                    </div>
                </div>

                <!-- GET Request Tester Tab -->
                <div id="testerTab" class="hidden p-6">
                    <div class="mb-6">
                        <h2 class="text-2xl font-bold text-gray-900">GET Request Tester</h2>
                        <p class="text-gray-600 mt-2">Test GET requests and view responses</p>
                    </div>

                    <!-- Quick Links -->
                    <div class="mb-6 bg-white rounded-lg shadow p-4">
                        <h3 class="text-lg font-medium text-gray-900 mb-3">Quick Links</h3>
                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-2">
                            <button onclick="testGetRequest('/categories')" class="text-left px-4 py-2 border border-gray-300 rounded hover:bg-gray-50 text-sm">
                                <span class="text-blue-600 font-medium">GET</span> /categories
                            </button>
                            <button onclick="testGetRequest('/fields?category_id=1')" class="text-left px-4 py-2 border border-gray-300 rounded hover:bg-gray-50 text-sm">
                                <span class="text-blue-600 font-medium">GET</span> /fields?category_id=1
                            </button>
                            <button onclick="testGetRequest('/records?category_id=1')" class="text-left px-4 py-2 border border-gray-300 rounded hover:bg-gray-50 text-sm">
                                <span class="text-blue-600 font-medium">GET</span> /records?category_id=1
                            </button>
                            <button onclick="testGetRequest('/Records?category_id=1')" class="text-left px-4 py-2 border border-gray-300 rounded hover:bg-gray-50 text-sm">
                                <span class="text-blue-600 font-medium">GET</span> /Records?category_id=1
                            </button>
                            <button onclick="testGetRequest('/files')" class="text-left px-4 py-2 border border-gray-300 rounded hover:bg-gray-50 text-sm">
                                <span class="text-blue-600 font-medium">GET</span> /files
                            </button>
                           
                            <button onclick="testGetRequest('/searchable-fields?category_id=1')" class="text-left px-4 py-2 border border-gray-300 rounded hover:bg-gray-50 text-sm">
                                <span class="text-blue-600 font-medium">GET</span> /searchable-fields
                            </button>
                            <button onclick="testGetRequest('/tags')" class="text-left px-4 py-2 border border-gray-300 rounded hover:bg-gray-50 text-sm">
                                <span class="text-blue-600 font-medium">GET</span> /tags
                            </button>
                        </div>
                    </div>

                    <!-- Custom Request -->
                    <div class="mb-6 bg-white rounded-lg shadow p-4">
                        <h3 class="text-lg font-medium text-gray-900 mb-3">Custom Request</h3>
                        <div class="flex gap-2">
                            <input
                                type="text"
                                id="customGetPath"
                                placeholder="/endpoint?param=value"
                                class="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                            >
                            <button
                                onclick="testGetRequest(document.getElementById('customGetPath').value)"
                                class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                            >
                                Send Request
                            </button>
                        </div>
                    </div>

                    <!-- Request Info -->
                    <div id="requestInfo" class="hidden mb-4 bg-blue-50 border border-blue-200 rounded-lg p-4">
                        <div class="flex items-center justify-between">
                            <div>
                                <span class="text-sm font-medium text-blue-800">Request:</span>
                                <span id="requestUrl" class="text-sm text-blue-600 ml-2"></span>
                            </div>
                            <div class="flex items-center space-x-2">
                                <span id="requestStatus" class="text-sm font-medium"></span>
                                <span id="requestTime" class="text-xs text-gray-600"></span>
                            </div>
                        </div>
                    </div>

                    <!-- Response -->
                    <div class="bg-white rounded-lg shadow">
                        <div class="p-4 border-b border-gray-200">
                            <div class="flex items-center justify-between">
                                <h3 class="text-lg font-medium text-gray-900">Response</h3>
                                <button
                                    onclick="copyResponse()"
                                    class="px-3 py-1 text-sm text-blue-600 hover:text-blue-800"
                                    id="copyResponseBtn"
                                    style="display: none;"
                                >
                                    Copy JSON
                                </button>
                            </div>
                        </div>
                        <div class="p-4">
                            <div id="responseEmpty" class="text-center py-12 text-gray-500">
                                No request sent yet. Use quick links or enter a custom endpoint above.
                            </div>
                            <pre id="responseOutput" class="hidden bg-gray-50 p-4 rounded-lg overflow-x-auto text-sm" style="max-height: 600px;"></pre>
                        </div>
                    </div>
                </div>

                <!-- Database Schema Tab -->
                <div id="schemaTab" class="hidden p-6">
                    <div class="mb-6">
                        <h2 class="text-2xl font-bold text-gray-900">Database Schema</h2>
                        <p class="text-gray-600 mt-2">Visual representation of database tables and relationships</p>
                    </div>

                    <div class="bg-white ">
                        <div id="schemaCanvas" class="relative w-full bg-gray-50 rounded-lg" style="min-height: 600px; width: 180%;">
                            <!-- Schema will be rendered here -->
                        </div>
                    </div>
                </div>

                <!-- Languages Tab -->
                <div id="languagesTab" class="hidden p-6">
                    <div class="mb-6 flex justify-between items-center">
                        <div>
                            <h2 class="text-2xl font-bold text-gray-900">Languages</h2>
                            <p class="text-gray-600 mt-1">Manage multi-language support for your content</p>
                        </div>
                        <button onclick="openLanguageModal()" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg flex items-center">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                            </svg>
                            Add Language
                        </button>
                    </div>

                    <div class="bg-white rounded-lg shadow">
                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200">
                                <thead class="bg-gray-50">
                                    <tr>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Code</th>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Direction</th>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Default</th>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Created</th>
                                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="languagesList" class="bg-white divide-y divide-gray-200">
                                    <!-- Languages will be rendered here -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Language Modal -->
    <div id="languageModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-lg shadow-xl max-w-md w-full p-6">
            <h3 id="languageModalTitle" class="text-lg font-medium text-gray-900 mb-4">Add Language</h3>
            <form id="languageForm" onsubmit="handleLanguageSubmit(event)">
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Language Code</label>
                    <input type="text" id="languageCode" required maxlength="10" pattern="[a-z]{2,10}"
                           placeholder="e.g., en, fr, es, ar"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <p class="text-xs text-gray-500 mt-1">2-10 lowercase letters (ISO 639-1 recommended)</p>
                </div>
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Language Name</label>
                    <input type="text" id="languageName" required
                           placeholder="e.g., English, Français, Español"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Text Direction</label>
                    <select id="languageTextDirection" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option value="ltr">Left-to-Right (LTR)</option>
                        <option value="rtl">Right-to-Left (RTL)</option>
                    </select>
                    <p class="text-xs text-gray-500 mt-1">Select RTL for languages like Arabic, Hebrew, etc.</p>
                </div>
                <div class="mb-4 flex items-center space-x-6">
                    <label class="flex items-center">
                        <input type="checkbox" id="languageIsActive" checked class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                        <span class="ml-2 text-sm text-gray-700">Active</span>
                    </label>
                    <label class="flex items-center">
                        <input type="checkbox" id="languageIsDefault" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                        <span class="ml-2 text-sm text-gray-700">Set as default</span>
                    </label>
                </div>
                <div class="flex justify-end space-x-3 mt-6">
                    <button type="button" onclick="closeModal('languageModal')" class="px-4 py-2 text-gray-700 hover:bg-gray-100 rounded-lg">
                        Cancel
                    </button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg flex items-center">
                        <span id="languageSpinner" class="hidden mr-2">
                            <svg class="animate-spin h-4 w-4" fill="none" viewBox="0 0 24 24">
                                <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                            </svg>
                        </span>
                        <span id="languageButtonText">Save</span>
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Category Modal -->
    <div id="categoryModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-lg shadow-xl max-w-md w-full p-6">
            <h3 id="categoryModalTitle" class="text-lg font-medium text-gray-900 mb-4">Add Category</h3>
            <form id="categoryForm" onsubmit="handleCategorySubmit(event)">
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Category Name</label>
                    <input type="text" id="categoryName" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                
                <div id="firstFieldSection" class="mb-4">
                    <h4 class="text-md font-medium text-gray-700 mb-3">First Field</h4>
                    <div class="space-y-3">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Field Name</label>
                            <input type="text" id="firstFieldName" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">Field Type</label>
                            <select id="firstFieldType" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                               <option value="">Select type...</option>
                               <option value="text">Text</option>
                               <option value="textarea">Textarea</option>
                               <option value="number">Number</option>
                               <option value="email">Email</option>
                               <option value="url">URL</option>
                               <option value="date">Date</option>
                               <option value="boolean">Boolean</option>
                               <option value="checkbox">Checkbox</option>
                               <option value="image">Image</option>
                               <option value="file">File</option>
                               <option value="media">Media (Images & Videos)</option>
                               <option value="rich_paragraph">Rich Paragraph</option>
                               <option value="foreign_key">Foreign Key</option>
                               <option value="array">Array</option>
                               <option value="json">JSON (Key-Value)</option>
                               <option value="link">Link (Media + Text)</option>
                            </select>
                        </div>
                        <div class="flex items-center">
                            <input type="checkbox" id="firstFieldRequired" class="mr-2">
                            <label class="text-sm text-gray-700">Required field</label>
                        </div>
                    </div>
                </div>

                <div class="flex justify-end space-x-3">
                    <button type="button" onclick="closeModal('categoryModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="submit" id="categorySubmitBtn" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed">
                        <span id="categorySpinner" class="hidden animate-spin rounded-full h-4 w-4 border-b-2 border-white inline-block mr-2"></span>
                        Save
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Field Modal -->
    <div id="fieldModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-lg shadow-xl max-w-md w-full p-6">
            <h3 id="fieldModalTitle" class="text-lg font-medium text-gray-900 mb-4">Add Field</h3>
            <form id="fieldForm" onsubmit="handleFieldSubmit(event)">
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Field Name</label>
                    <input type="text" id="fieldName" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Field Type</label>
                    <select id="fieldType" required onchange="handleFieldTypeChange()" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option value="">Select type...</option>
                        <option value="text">Text</option>
                        <option value="textarea">Textarea</option>
                        <option value="number">Number</option>
                        <option value="email">Email</option>
                        <option value="url">URL</option>
                        <option value="date">Date</option>
                        <option value="boolean">Boolean</option>
                        <option value="checkbox">Checkbox</option>
                        <option value="image">Image</option>
                        <option value="file">File</option>
                        <option value="media">Media (Images & Videos)</option>
                        <option value="rich_paragraph">Rich Paragraph</option>
                        <option value="foreign_key">Foreign Key</option>
                        <option value="array">Array</option>
                        <option value="json">JSON (Key-Value)</option>
                        <option value="link">Link (Media + Text)</option>
                    </select>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Default Value</label>
                    <div id="defaultValueContainer">
                        <input type="text" id="fieldDefaultValue" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Enter default value (optional)">
                    </div>
                    <div id="defaultValueHelp" class="text-xs text-gray-500 mt-1"></div>
                </div>

                <!-- Foreign Key Configuration -->
                <div id="foreignKeyContainer" class="mb-4 hidden">
                    <div class="p-4 bg-blue-50 rounded-lg">
                        <h4 class="text-sm font-medium text-blue-900 mb-3">Foreign Key Configuration</h4>
                        <div class="grid grid-cols-2 gap-3">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Target Table</label>
                                <select id="foreignKeyTable" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="">Select table...</option>
                                </select>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Target Column</label>
                                <select id="foreignKeyColumn" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                    <option value="">Select column...</option>
                                </select>
                            </div>
                        </div>
                        <div class="text-xs text-gray-600 mt-2">
                            Select which table and column this field should reference.
                        </div>
                    </div>
                </div>

                <!-- Array Configuration -->
                <div id="arrayContainer" class="mb-4 hidden">
                    <div class="p-4 bg-green-50 rounded-lg">
                        <h4 class="text-sm font-medium text-green-900 mb-3">Array Configuration</h4>
                        <div class="mb-3">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Array Item Type</label>
                            <select id="arrayItemType" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="text">Text</option>
                                <option value="number">Number</option>
                                <option value="url">URL</option>
                                <option value="email">Email</option>
                            </select>
                        </div>
                        <div class="flex items-center mb-2">
                            <input type="checkbox" id="arrayAllowMultiple" class="mr-2" checked>
                            <label class="text-sm text-gray-700">Allow multiple values</label>
                        </div>
                        <div class="text-xs text-gray-600">
                            Array fields store multiple values as JSON. Users can add/remove items dynamically.
                        </div>
                    </div>
                </div>

                <!-- JSON Configuration -->
                <div id="jsonContainer" class="mb-4 hidden">
                    <div class="p-4 bg-purple-50 rounded-lg">
                        <h4 class="text-sm font-medium text-purple-900 mb-3">JSON Field Configuration</h4>
                        <div class="mb-3">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Value Type</label>
                            <select id="jsonValueType" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                                <option value="text">Text</option>
                                <option value="number">Number</option>
                                <option value="email">Email</option>
                                <option value="url">URL</option>
                                <option value="textarea">Long Text</option>
                                <option value="date">Date</option>
                                <option value="checkbox">Boolean (True/False)</option>
                            </select>
                        </div>
                        <div class="flex items-center mb-2">
                            <input type="checkbox" id="jsonAllowCustomKeys" class="mr-2" checked>
                            <label class="text-sm text-gray-700">Allow custom keys</label>
                        </div>
                        <div class="mb-3">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Predefined Keys (optional)</label>
                            <textarea id="jsonPredefinedKeys" placeholder="Enter keys separated by commas (e.g., name, description, price)" rows="2" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"></textarea>
                            <div class="text-xs text-gray-500 mt-1">If provided, users can select from these keys or add their own (if custom keys allowed)</div>
                        </div>
                        <div class="text-xs text-gray-600">
                            JSON fields store flexible key-value pairs. Perfect for dynamic data structures like product attributes, metadata, or configuration settings.
                        </div>
                    </div>
                </div>

                <!-- Rich Paragraph Configuration -->
                <div id="richParagraphContainer" class="mb-4 hidden">
                    <div class="p-4 bg-green-50 rounded-lg">
                        <h4 class="text-sm font-medium text-green-900 mb-3">Rich Paragraph Configuration</h4>
                        <div class="mb-3">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Maximum Characters</label>
                            <input type="number" id="richParagraphMaxChars" min="0" placeholder="Leave empty for no limit" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <div class="text-xs text-gray-500 mt-1">Set the maximum number of characters allowed (counting HTML tags). Leave empty for no limit.</div>
                        </div>
                        <div class="text-xs text-gray-600">
                            Character limit helps maintain consistent content length. The counter will show remaining characters as users type.
                        </div>
                    </div>
                </div>

                <div class="mb-4 flex items-center">
                    <input type="checkbox" id="fieldRequired" class="mr-2">
                    <label class="text-sm text-gray-700">Required field</label>
                </div>

                <div class="flex justify-end space-x-3">
                    <button type="button" onclick="closeModal('fieldModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                        Cancel
                    </button>
                    <!-- <button type="submit" id="fieldSubmitBtn" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"> -->
                    <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed">
                        <span id="fieldSpinner" class="hidden animate-spin rounded-full h-4 w-4 border-b-2 border-white inline-block mr-2"></span>
                        Save
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Record Modal -->
    <div id="recordModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-lg shadow-xl max-w-2xl w-full p-6 max-h-[90vh] overflow-y-auto">
            <h3 id="recordModalTitle" class="text-lg font-medium text-gray-900 mb-4">Add Record</h3>
            <form id="recordForm" onsubmit="handleRecordSubmit(event)">
                <div id="recordFields" class="space-y-4 mb-6">
                    <!-- Dynamic fields will be inserted here -->
                </div>

                <div class="flex justify-between items-center space-x-3">
                    <button type="button" id="manageTranslationsBtn" onclick="openTranslationsModal()" class="hidden px-4 py-2 border border-blue-600 text-blue-600 rounded-md hover:bg-blue-50 flex items-center">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 5h12M9 3v2m1.048 9.5A18.022 18.022 0 016.412 9m6.088 9h7M11 21l5-10 5 10M12.751 5C11.783 10.77 8.07 15.61 3 18.129"></path>
                        </svg>
                        Manage Translations
                    </button>
                    <div class="flex space-x-3">
                        <button type="button" onclick="closeModal('recordModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                            Cancel
                        </button>
                        <button type="submit" id="recordSubmitBtn" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed">
                            <span id="recordSpinner" class="hidden animate-spin rounded-full h-4 w-4 border-b-2 border-white inline-block mr-2"></span>
                            Save
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Upload Modal -->
    <div id="uploadModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-lg shadow-xl max-w-lg w-full p-6">
            <h3 class="text-lg font-medium text-gray-900 mb-4">Upload Media</h3>

            <!-- Upload Type Tabs -->
            <div class="flex mb-4 border-b">
                <button type="button" id="fileUploadTab" onclick="switchUploadTab('file')" class="flex-1 py-2 px-1 text-center font-medium text-xs sm:text-sm text-blue-600 border-b-2 border-blue-600 min-w-0">
                    <span class="hidden sm:inline">File Upload</span>
                    <span class="sm:hidden">File</span>
                </button>
                <button type="button" id="urlUploadTab" onclick="switchUploadTab('url')" class="flex-1 py-2 px-1 text-center font-medium text-xs sm:text-sm text-gray-500 border-b-2 border-transparent hover:text-gray-700 min-w-0">
                    <span class="hidden sm:inline">Video URL</span>
                    <span class="sm:hidden">Video</span>
                </button>
            </div>

            <!-- File Upload Form -->
            <form id="uploadForm" onsubmit="handleUploadSubmit(event)" class="upload-form">
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Select Files</label>
                    <input type="file" id="uploadFile" required multiple class="w-full px-3 py-2 border-2 border-dashed border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                           accept=".jpg,.jpeg,.png,.gif,.webp,.svg,.mp4,.avi,.mov,.wmv,.flv,.webm,.mkv,.mp3,.wav,.ogg,.m4a,.aac,.pdf,.doc,.docx">
                    <!-- <div class="mt-2 text-xs text-gray-500">
                        <p><strong>Supported formats:</strong></p>
                        <p>• Images: JPG, PNG, GIF, WebP, SVG (max 50MB each)</p>
                        <p>• Videos: MP4, AVI, MOV, WMV, WebM, MKV (no limit each)</p>
                        <p>• Audio: MP3, WAV, OGG, M4A, AAC (max 50MB each)</p>
                        <p>• Documents: PDF, DOC, DOCX (max 50MB each)</p>
                        <p class="text-blue-600 font-medium">• Multiple files supported - select multiple files at once</p>
                    </div> -->
                    <div id="selectedFilesPreview" class="mt-2 hidden">
                        <p class="text-sm font-medium text-gray-700 mb-2">Selected Files:</p>
                        <div id="filesList" class="space-y-1 max-h-32 overflow-y-auto"></div>
                    </div>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Tags (optional)</label>
                    <div class="relative">
                        <input type="text" id="uploadTags" placeholder="Enter tags separated by commas (e.g., photo, nature, landscape)"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                               autocomplete="off">
                        <div id="uploadTagsSuggestions" class="absolute top-full left-0 right-0 bg-white border border-gray-200 rounded-b-lg shadow-lg hidden z-10 max-h-48 overflow-y-auto"></div>
                    </div>
                    <div class="mt-2 text-xs text-gray-500">
                        Add tags to help organize and search your media files. Separate multiple tags with commas.
                    </div>
                </div>

                <!-- Quick links to URL uploads (in case tabs are not visible) -->
                <div class="mb-4 text-sm text-gray-600">
                    Or upload by URL:
                    <button type="button" onclick="switchUploadTab('url')" class="ml-2 text-blue-600 hover:underline">Video URL</button>
                </div>

                <div class="flex justify-end space-x-3">
                    <button type="button" onclick="closeModal('uploadModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="submit" id="uploadSubmitBtn" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed">
                        <span id="uploadSpinner" class="hidden animate-spin rounded-full h-4 w-4 border-b-2 border-white inline-block mr-2"></span>
                        Upload
                    </button>
                </div>
            </form>

            <!-- Video URL Upload Form -->
            <form id="urlUploadForm" onsubmit="handleUrlUploadSubmit(event)" class="upload-form hidden">
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Video URL</label>
                    <input type="url" id="videoUrl" required placeholder="https://www.youtube.com/watch?v=... or https://vimeo.com/..."
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <div class="mt-2 text-xs text-gray-500">
                        <p><strong>Supported platforms:</strong></p>
                        <p>• YouTube (youtube.com, youtu.be)</p>
                        <p>• Vimeo (vimeo.com)</p>
                        <p>• Dailymotion (dailymotion.com)</p>
                        <p>• Direct video URLs (mp4, webm, ogg)</p>
                    </div>
                </div>

                <!-- Live Preview -->
                <div class="mb-3">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Preview</label>
                    <div id="videoUrlPreview"
                         class="hidden w-full bg-gray-50 border border-gray-200 rounded-md p-2 flex items-center justify-center"></div>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Title (optional)</label>
                    <input type="text" id="videoTitle" placeholder="Custom video title"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <div class="mt-2 text-xs text-gray-500">
                        Leave empty to use auto-detected title
                    </div>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Tags (optional)</label>
                    <div class="relative">
                        <input type="text" id="urlUploadTags" placeholder="Enter tags separated by commas (e.g., video, tutorial, entertainment)"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                               autocomplete="off">
                        <div id="urlUploadTagsSuggestions" class="absolute top-full left-0 right-0 bg-white border border-gray-200 rounded-b-lg shadow-lg hidden z-10 max-h-48 overflow-y-auto"></div>
                    </div>
                    <div class="mt-2 text-xs text-gray-500">
                        Add tags to help organize and search your video links.
                    </div>
                </div>

                <div class="flex justify-end space-x-3">
                    <button type="button" onclick="closeModal('uploadModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="submit" id="urlUploadSubmitBtn" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed">
                        <span id="urlUploadSpinner" class="hidden animate-spin rounded-full h-4 w-4 border-b-2 border-white inline-block mr-2"></span>
                        Add Video
                    </button>
                </div>
            </form>

            <!-- Media URL Upload Form -->
            <form id="mediaUrlUploadForm" onsubmit="handleMediaUrlUploadSubmit(event)" class="upload-form hidden">
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Media Type</label>
                    <select id="mediaType" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option value="">Select Media Type</option>
                        <option value="image">Image</option>
                        <option value="video">Video</option>
                    </select>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Media URL</label>
                    <input type="url" id="mediaUrl" required placeholder="https://example.com/image.jpg or https://youtube.com/watch?v=..."
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>

                <!-- Live Preview -->
                <div class="mb-3">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Preview</label>
                    <div id="mediaUrlPreview"
                         class="hidden w-full bg-gray-50 border border-gray-200 rounded-md p-2 flex items-center justify-center"></div>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Title (optional)</label>
                    <input type="text" id="mediaUrlTitle" placeholder="Custom media title"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <div class="mt-2 text-xs text-gray-500">
                        Leave empty to use auto-detected title or filename
                    </div>
                </div>

                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Tags (optional)</label>
                    <div class="relative">
                        <input type="text" id="mediaUrlTags" placeholder="Enter tags separated by commas (e.g., image, photo, video)"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                               autocomplete="off">
                        <div id="mediaUrlTagsSuggestions" class="absolute top-full left-0 right-0 bg-white border border-gray-200 rounded-b-lg shadow-lg hidden z-10 max-h-48 overflow-y-auto"></div>
                    </div>
                    <div class="mt-2 text-xs text-gray-500">
                        Add tags to help organize and search your media.
                    </div>
                </div>

                <div class="flex justify-end space-x-3">
                    <button type="button" onclick="closeModal('uploadModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="submit" id="mediaUrlUploadSubmitBtn" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed">
                        <span id="mediaUrlUploadSpinner" class="hidden animate-spin rounded-full h-4 w-4 border-b-2 border-white inline-block mr-2"></span>
                        Add Media
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Iframe Video Modal -->
    <div id="iframeModal" class="hidden fixed inset-0 bg-black bg-opacity-90 z-50 flex items-center justify-center p-4">
        <div class="relative bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-full overflow-hidden">
            <!-- Modal Header -->
            <div class="flex items-center justify-between p-4 border-b">
                <h3 id="iframeModalTitle" class="text-lg font-medium text-gray-900">Video Player</h3>
                <button onclick="closeIframeModal()" class="text-gray-400 hover:text-gray-600 transition-colors">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            <!-- Modal Content -->
            <div class="p-4">
                <div id="iframeContainer" class="w-full flex justify-center">
                    <!-- Iframe will be inserted here -->
                </div>
            </div>
        </div>
    </div>

    <!-- Translation Modal -->
    <div id="translationModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
        <div class="bg-white rounded-lg shadow-xl max-w-6xl w-full p-6 max-h-[90vh] overflow-y-auto">
            <h3 class="text-lg font-medium text-gray-900 mb-4">Manage Translations</h3>

            <!-- Language Selector -->
            <div class="mb-6">
                <div class="flex items-end gap-4 mb-3">
                    <div class="flex-1">
                        <label class="block text-sm font-medium text-gray-700 mb-2">Select Language</label>
                        <select id="translationLanguageSelect"
                                class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                                onchange="handleLanguageSelection()">
                            <option value="">-- Select a language --</option>
                            <!-- Languages will be populated here -->
                        </select>
                        <p class="text-xs text-gray-500 mt-1">Select an existing language from the Languages menu</p>
                    </div>

                    <!-- Saved Languages Buttons -->
                    <div id="savedLanguagesContainer" class="hidden flex-1">
                        <label class="block text-sm font-medium text-gray-700 mb-2">Saved Translations</label>
                        <div id="savedLanguagesButtons" class="flex flex-wrap gap-2">
                            <!-- Language buttons will appear here -->
                        </div>
                    </div>
                </div>

                <!-- Default Language Checkbox -->
                <div id="defaultLanguageCheckbox" class="hidden">
                    <label class="flex items-center">
                        <input type="checkbox" id="translationIsDefault" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                        <span class="ml-2 text-sm text-gray-700">Set as default language</span>
                    </label>
                    <p class="text-xs text-gray-500 mt-1 ml-6">The default language will be used when no translation is available</p>
                </div>
            </div>

            <!-- Translation Fields Grid -->
            <div id="translationFieldsContainer" class="hidden">
                <div class="grid grid-cols-2 gap-6">
                    <!-- Original Content Column -->
                    <div>
                        <h4 class="font-semibold text-gray-900 mb-4 pb-2 border-b">Original Content</h4>
                        <div id="originalFields" class="space-y-4">
                            <!-- Original fields will be inserted here -->
                        </div>
                    </div>

                    <!-- Translation Column -->
                    <div>
                        <h4 class="font-semibold text-gray-900 mb-4 pb-2 border-b">
                            Translation (<span id="currentLanguageLabel">--</span>)
                        </h4>
                        <div id="translationFields" class="space-y-4">
                            <!-- Translation fields will be inserted here -->
                        </div>
                    </div>
                </div>
            </div>

            <div class="flex justify-between items-center mt-6 pt-4 border-t">
                <button type="button" onclick="deleteCurrentTranslation()" id="deleteTranslationBtn" class="hidden px-4 py-2 border border-red-600 text-red-600 rounded-md hover:bg-red-50">
                    Delete Translation
                </button>
                <div class="flex space-x-3 ml-auto">
                    <button type="button" onclick="closeModal('translationModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                        Close
                    </button>
                    <button type="button" id="saveTranslationsBtn" onclick="handleSaveTranslations()" class="hidden px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed">
                        <span id="translationSpinner" class="hidden animate-spin rounded-full h-4 w-4 border-b-2 border-white inline-block mr-2"></span>
                        Save Translation
                    </button>
                </div>
            </div>
        </div>
    </div>

<script>
// Global Variables
const API_BASE_URL = window.location.origin + window.location.pathname;
let currentUser = null;
let authToken = null;
let categories = [];
let currentTab = "categories";
let editingCategory = null;
let editingField = null;
let editingRecord = null;
let currentCategoryFields = [];
let currentMediaType = '';
let currentMediaPage = 1;
let currentMediaSearch = '';
let mediaPerPage = 12;
let recordsViewMode = 'card'; // 'card' or 'table'

// Utility Functions
function validateSlugFormat(slug) {
    if (!slug || slug.trim() === '') {
        return { valid: true, message: '' }; // Empty slug is allowed
    }

    slug = slug.trim();

    // Check if slug contains only letters (uppercase/lowercase), numbers, and hyphens
    // Must not start or end with hyphen, no consecutive hyphens
    const slugPattern = /^[a-zA-Z0-9]+(-[a-zA-Z0-9]+)*$/;

    if (!slugPattern.test(slug)) {
        return {
            valid: false,
            message: "Slug can contain letters, numbers, and hyphens only. Format: 'my-article-slug' or 'My-Article-Slug'. No spaces, no special characters except hyphens, cannot start or end with hyphen."
        };
    }

    // Additional check: no consecutive hyphens
    if (slug.includes('--')) {
        return {
            valid: false,
            message: "Slug cannot contain consecutive hyphens. Format: 'my-article-slug' or 'My-Article-Slug'"
        };
    }

    return { valid: true, message: 'Valid slug format ✓' };
}

// Upload Configuration
let uploadConfig = null;

async function loadUploadConfig() {
    if (uploadConfig) return uploadConfig;

    try {
        const response = await apiRequest('/upload-config');
        uploadConfig = response;
        return uploadConfig;
    } catch (error) {
        console.error('Failed to load upload config:', error);
        // Return unlimited defaults
        return {
            upload_max_filesize_bytes: Infinity,
            post_max_size_bytes: Infinity
        };
    }
}

function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

async function validateFileSize(file) {
    // No size limits - accept all files
    return { valid: true };
}

function validateSlugInput(inputElement) {
    const slug = inputElement.value;
    const validation = validateSlugFormat(slug);

    // Find or create validation message element
    let messageEl = inputElement.parentNode.querySelector('.slug-validation-message');
    if (!messageEl) {
        messageEl = document.createElement('p');
        messageEl.className = 'slug-validation-message text-xs mt-1';
        inputElement.parentNode.appendChild(messageEl);
    }

    // Update message and styling
    if (validation.valid) {
        messageEl.textContent = validation.message;
        messageEl.className = 'slug-validation-message text-xs mt-1 text-green-600';
        inputElement.classList.remove('border-red-300', 'focus:ring-red-500');
        inputElement.classList.add('border-green-300', 'focus:ring-green-500');
    } else {
        messageEl.textContent = validation.message;
        messageEl.className = 'slug-validation-message text-xs mt-1 text-red-600';
        inputElement.classList.remove('border-green-300', 'focus:ring-green-500');
        inputElement.classList.add('border-red-300', 'focus:ring-red-500');
    }

    return validation.valid;
}

function showToast(message, type = "success") {
    const toast = document.createElement("div");
    const bgColor = type === "success" ? "bg-green-500" :
        type === "error" ? "bg-red-500" :
        type === "warning" ? "bg-yellow-500" : "bg-blue-500";

    toast.className = `fixed top-4 left-4 px-6 py-4 rounded-lg shadow-lg z-50 ${bgColor} text-white font-medium transform transition-all duration-300 translate-x-full`;
    toast.textContent = message;

    document.body.appendChild(toast);

    setTimeout(() => toast.classList.remove("translate-x-full"), 100);
    setTimeout(() => {
        toast.classList.add("translate-x-full");
        setTimeout(() => toast.remove(), 300);
    }, 4000);
}

function showLoading() {
    const loadingOverlay = document.getElementById('loadingOverlay');
    if (loadingOverlay) {
        loadingOverlay.classList.remove('hidden');
    }
}

function hideLoading() {
    const loadingOverlay = document.getElementById('loadingOverlay');
    if (loadingOverlay) {
        loadingOverlay.classList.add('hidden');
    }
}

function setButtonLoading(buttonId, spinnerId, loading) {
    const button = document.getElementById(buttonId);
    const spinner = document.getElementById(spinnerId);

    if (button) {
        button.disabled = loading;
    }
    if (spinner) {
        if (loading) {
            spinner.classList.remove("hidden");
        } else {
            spinner.classList.add("hidden");
        }
    }
}

// API Request Function
async function apiRequest(endpoint, options = {}) {
    const url = `${API_BASE_URL}${endpoint}`;
    console.log("request :",url)
    const config = {
        headers: {
            'Content-Type': 'application/json',
            "X-API-TOKEN": "2250c47a4397a41fe5ff8f2c2a92b240ddf46aaeded8ffddb4f29f02ffef8720",
            ...options.headers
        },
        ...options
    };

    if (authToken && !config.headers['Authorization']) {
        config.headers['Authorization'] = `Bearer ${authToken}`;
    }

    try {
        const response = await fetch(url, config);
        
        let data;
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
            data = await response.json();
        } else {
            const text = await response.text();
            throw new Error(`Server returned non-JSON response: ${text}`);
        }

        if (!response.ok) {
            if (response.status === 401) {
                authToken = null;
                currentUser = null;
                localStorage.removeItem('authToken');
                localStorage.removeItem('currentUser');
                showToast("Please log in again", "warning");
                showAuthSection();
                return;
            }
            // Extract error message from response (can be in data.error or data.data or just data)
            const errorMessage = data.error || data.data || data.message || `HTTP ${response.status}: ${response.statusText}`;
            throw new Error(errorMessage);
        }

        if (!data.success) {
            // Extract error message from response (can be in data.error or data.data or just data)
            const errorMessage = data.error || data.data || data.message || 'API request failed';
            throw new Error(errorMessage);
        }

        return data.data;
    } catch (error) {
        console.error('API Error:', error);
        throw error;
    }
}

// Authentication Functions
async function login(email, password) {
    try {
        const result = await apiRequest('/login', {
            method: 'POST',
            body: JSON.stringify({ email, password })
        });

        if (!result.user || !result.token) {
            throw new Error("Invalid login response format");
        }

        currentUser = result.user;
        authToken = result.token;

        // Store in localStorage
        localStorage.setItem('authToken', authToken);
        localStorage.setItem('currentUser', JSON.stringify(currentUser));

        return { success: true, user: currentUser };
    } catch (error) {
        console.error("Login error:", error);
        throw error;
    }
}


function logout() {
    currentUser = null;
    authToken = null;
    localStorage.removeItem('authToken');
    localStorage.removeItem('currentUser');

    showAuthSection();
    showToast("Logged out successfully");
}

function showAuthSection() {
    document.getElementById("authSection").classList.remove("hidden");
    document.getElementById("mainApp").classList.add("hidden");
}

function showMainApp() {
    document.getElementById("authSection").classList.add("hidden");
    document.getElementById("mainApp").classList.remove("hidden");
    const userWelcome = document.getElementById("userWelcome");
    if (userWelcome) {
        userWelcome.textContent = `Welcome, ${currentUser.name}`;
    }

    loadData();
    showTab("categories");
}


// Tab Management
function showTab(tabName) {
    currentTab = tabName;
    
    // Hide all tabs
    document.querySelectorAll('[id$="Tab"]').forEach(tab => {
        tab.classList.add('hidden');
    });
    
    // Show selected tab
    document.getElementById(tabName + 'Tab').classList.remove('hidden');
    
    // Update navigation
    document.querySelectorAll('.nav-item').forEach(item => {
        item.classList.remove('bg-blue-500', 'text-white');
        // item.classList.add('text-gray-700', 'hover:bg-gray-100');
    });
    
    // Find and activate current nav item
    const navItems = document.querySelectorAll('.nav-item');
    navItems.forEach(item => {
        if (item.textContent.toLowerCase().trim() === tabName.toLowerCase() ||
            (tabName === 'tester' && item.textContent.toLowerCase().trim() === 'get tester') ||
            (tabName === 'schema' && item.textContent.toLowerCase().includes('database schema'))) {
            item.classList.add('bg-blue-500', 'text-white');
            item.classList.remove('text-gray-700', 'hover:bg-gray-100');
        }
    });

    // Load tab-specific data
    if (tabName === 'categories') {
        loadCategories();
    } else if (tabName === 'media') {
        loadMedia();
        // Initialize search functionality and tag filter for media tab
        setTimeout(() => {
            setupMediaSearch();
            populateTagFilter();
            updateSearchPlaceholder();
        }, 100);
    } else if (tabName === 'tester') {
        // No special initialization needed for tester tab
    } else if (tabName === 'schema') {
        loadDatabaseSchema();
    } else if (tabName === 'languages') {
        loadLanguages();
    }
}

// Data Loading Functions
async function loadData() {
    try {
        showLoading();
        await Promise.all([
            loadCategories(),
            populateCategorySelects()
        ]);
    } catch (error) {
        console.error('Error loading data:', error);
        showToast('Failed to load data: ' + error.message, 'error');
    } finally {
        hideLoading();
    }
}

// Category Functions
// Pagination state for categories
let categoriesCurrentPage = 1;
let categoriesPerPage = 10;
let categoriesTotalRecords = 0;
let categoriesLastSearchQuery = '';

async function loadCategories(searchQuery = '', page = 1) {
    try {
        categoriesCurrentPage = page;
        categoriesLastSearchQuery = searchQuery;

        const params = new URLSearchParams({
            page: page.toString(),
            per_page: categoriesPerPage.toString()
        });
        if (searchQuery) {
            params.append('q', searchQuery);
        }

        const res = await apiRequest(`/categories?${params.toString()}`);


        let total = 0;
        if (res && res.items && Array.isArray(res.items)) {
            categories = res.items;
            total = res.pagination && res.pagination.total ? res.pagination.total : res.items.length;
        } else if (Array.isArray(res)) {
            categories = res;
            total = res.length;
        } else {
            categories = [];
        }

        categoriesTotalRecords = total;
        renderCategories();
        updateCategoriesPagination();
        populateCategorySelects();
    } catch (error) {
        console.error('Error loading categories:', error);
        showToast('Failed to load categories: ' + error.message, 'error');
        categories = [];
        renderCategories();
    }
}

function updateCategoriesPagination() {
    const totalPages = Math.ceil(categoriesTotalRecords / categoriesPerPage);
    const pageInfo = document.getElementById('categoryPageInfo');
    const prevBtn = document.getElementById('categoryPrevBtn');
    const nextBtn = document.getElementById('categoryNextBtn');

    if (pageInfo) {
        pageInfo.textContent = `Page ${categoriesCurrentPage} of ${totalPages || 1}`;
    }

    if (prevBtn) {
        prevBtn.disabled = categoriesCurrentPage === 1;
        prevBtn.onclick = () => {
            if (categoriesCurrentPage > 1) {
                loadCategories(categoriesLastSearchQuery, categoriesCurrentPage - 1);
            }
        };
    }

    if (nextBtn) {
        nextBtn.disabled = categoriesCurrentPage >= totalPages;
        nextBtn.onclick = () => {
            if (categoriesCurrentPage < totalPages) {
                loadCategories(categoriesLastSearchQuery, categoriesCurrentPage + 1);
            }
        };
    }
}

function renderCategories() {
    const tbody = document.getElementById("categoriesTableBody");
    const emptyState = document.getElementById("categoriesEmpty");
    const table = document.getElementById("categoriesTable");

    if (!tbody) return;

    if (categories.length === 0) {
        if (table) table.classList.add("hidden");
        if (emptyState) emptyState.classList.remove("hidden");
        return;
    }

    if (table) table.classList.remove("hidden");
    if (emptyState) emptyState.classList.add("hidden");

    tbody.innerHTML = categories.map((category) => {
        return `
            <tr class="hover:bg-gray-50 transition-colors">
                <td class="px-6 py-4 whitespace-nowrap">
                    <input type="checkbox" class="category-checkbox rounded border-gray-300 text-blue-600 focus:ring-blue-500" data-id="${category.id}" onchange="updateCategorySelection()">
                </td>
                <td class="px-6 py-4 whitespace-nowrap">
                    <div class="text-sm font-medium text-gray-900">${escapeHtml(category.name)}</div>
                </td>
                <td class="px-6 py-4 whitespace-nowrap">
                    <span onclick="navigateToFields('${category.id}')" class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 cursor-pointer hover:bg-blue-200 transition-colors" title="Click to view fields">
                        ${category.field_count || 0} fields
                    </span>
                </td>
                <td class="px-6 py-4 whitespace-nowrap">
                    <span onclick="navigateToRecords('${category.id}')" class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800 cursor-pointer hover:bg-green-200 transition-colors" title="Click to view records">
                        ${category.record_count || 0} records
                    </span>
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    ${category.created_at ? new Date(category.created_at).toLocaleDateString() : new Date().toLocaleDateString()}
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                    <button onclick="editCategory('${category.id}')" class="text-blue-600 hover:text-blue-700 mr-3 transition-colors">Edit</button>
                    ${category.record_count > 0
                        ? `<span class="text-gray-400 cursor-not-allowed" title="Cannot delete category with ${category.record_count} record(s)">Delete</span>`
                        : `<button onclick="deleteCategory('${category.id}')" class="text-red-600 hover:text-red-800 transition-colors">Delete</button>`
                    }
                </td>
            </tr>
        `;
    }).join("");
}

async function populateCategorySelects() {
    const selects = ['fieldsCategorySelect', 'recordsCategorySelect'];

    selects.forEach(selectId => {
        const select = document.getElementById(selectId);
        if (select) {
            const currentValue = select.value;
            select.innerHTML = '<option value="">Choose a category...</option>';
            categories.forEach(category => {
                const option = document.createElement('option');
                option.value = category.id;
                option.textContent = category.name;
                if (currentValue == category.id) {
                    option.selected = true;
                }
                select.appendChild(option);
            });
        }
    });

    // Also populate global language selector
    await populateGlobalLanguageSelect();
}

// Global language state - Load from localStorage if available
let currentGlobalLanguage = localStorage.getItem('selectedLanguage') || '';

async function populateGlobalLanguageSelect() {
    const select = document.getElementById('globalLanguageSelect');
    if (!select) {
        console.log('Global language select element not found');
        return;
    }

    try {
        const languages = await apiRequest('/languages');
        console.log('Languages loaded:', languages);

        if (!languages || languages.length === 0) {
            select.innerHTML = '<option value="">Default Language (No languages configured)</option>';
            return;
        }

        // Filter to show only active languages
        const activeLanguages = languages.filter(lang => {
            const isActive = lang.is_active === 1 || lang.is_active === true || lang.is_active === '1';
            console.log(`Language ${lang.code}: is_active=${lang.is_active}, type=${typeof lang.is_active}, included=${isActive}`);
            return isActive;
        });

        if (activeLanguages.length === 0) {
            console.warn('No active languages found!');
            select.innerHTML = '<option value="">Default Language (No active languages)</option>';
            return;
        }

        console.log(`Populating select with ${activeLanguages.length} active languages`);
        select.innerHTML = '<option value="">Default Language</option>';
        activeLanguages.forEach(lang => {
            const option = document.createElement('option');
            option.value = lang.code;
            option.textContent = `${lang.name}${lang.is_default ? ' (Default)' : ''}`;
            select.appendChild(option);
            console.log(`Added option: ${lang.code} - ${lang.name}`);
        });

        // Restore saved language selection
        if (currentGlobalLanguage) {
            select.value = currentGlobalLanguage;
        }
    } catch (error) {
        console.error('Error loading languages:', error);
    }
}

function changeGlobalLanguage() {
    const select = document.getElementById('globalLanguageSelect');
    currentGlobalLanguage = select.value;

    // Save to localStorage for persistence
    if (currentGlobalLanguage) {
        localStorage.setItem('selectedLanguage', currentGlobalLanguage);
    } else {
        localStorage.removeItem('selectedLanguage');
    }

    // Reload records with new language
    const categoryId = document.getElementById('recordsCategorySelect').value;
    if (categoryId) {
        showToast(`Switching to ${select.options[select.selectedIndex].text}...`, 'info');
        loadRecordsForCategory();
    }
}

function editCategory(categoryId) {
    const category = categories.find((c) => c.id == categoryId);
    if (!category) return;

    editingCategory = categoryId;
    const categoryModalTitle = document.getElementById("categoryModalTitle");
    const categoryName = document.getElementById("categoryName");
    const firstFieldSection = document.getElementById("firstFieldSection");
    
    if (categoryModalTitle) categoryModalTitle.textContent = "Edit Category";
    if (categoryName) categoryName.value = category.name || '';
    
    // Hide first field section when editing
    if (firstFieldSection) firstFieldSection.style.display = 'none';

    openModal("categoryModal");
}

async function deleteCategory(categoryId) {
    if (!confirm("Are you sure you want to delete this category? This will also delete all associated fields and records.")) {
        return;
    }

    try {
        await apiRequest(`/categories/${categoryId}`, {
            method: 'DELETE'
        });

        showToast("Category deleted successfully");
        loadCategories();
    } catch (error) {
        console.error('Error deleting category:', error);
        showToast('Failed to delete category: ' + error.message, 'error');
    }
}

// Navigation functions - Navigate from Categories to Fields/Records
function navigateToFields(categoryId) {
    // Switch to Fields tab
    showTab('fields');

    // Wait for tab to be shown, then select the category
    setTimeout(() => {
        const fieldsCategorySelect = document.getElementById('fieldsCategorySelect');
        if (fieldsCategorySelect) {
            fieldsCategorySelect.value = categoryId;
            // Trigger the change event to load fields
            loadFieldsForCategory();
            showToast('Viewing fields for selected category', 'success');
        }
    }, 100);
}

function navigateToRecords(categoryId) {
    // Switch to Records tab
    showTab('records');

    // Wait for tab to be shown, then select the category
    setTimeout(() => {
        const recordsCategorySelect = document.getElementById('recordsCategorySelect');
        if (recordsCategorySelect) {
            recordsCategorySelect.value = categoryId;
            // Trigger the change event to load records
            loadRecordsForCategory();
            showToast('Viewing records for selected category', 'success');
        }
    }, 100);
}

// Bulk Category Operations
function toggleSelectAllCategories() {
    const selectAll = document.getElementById('categorySelectAll');
    const checkboxes = document.querySelectorAll('.category-checkbox');

    checkboxes.forEach(checkbox => {
        checkbox.checked = selectAll.checked;
    });

    updateCategorySelection();
}

function updateCategorySelection() {
    const checkboxes = document.querySelectorAll('.category-checkbox:checked');
    const count = checkboxes.length;
    const bulkActionsBar = document.getElementById('categoryBulkActionsBar');
    const selectedCount = document.getElementById('categorySelectedCount');

    if (count > 0) {
        bulkActionsBar.classList.remove('hidden');
        selectedCount.textContent = count;
    } else {
        bulkActionsBar.classList.add('hidden');
    }

    // Update select all checkbox state
    const selectAll = document.getElementById('categorySelectAll');
    const allCheckboxes = document.querySelectorAll('.category-checkbox');
    selectAll.checked = allCheckboxes.length > 0 && count === allCheckboxes.length;
}

async function bulkDeleteCategories() {
    const checkboxes = document.querySelectorAll('.category-checkbox:checked');
    const ids = Array.from(checkboxes).map(cb => cb.dataset.id);

    if (ids.length === 0) {
        showToast('No categories selected', 'warning');
        return;
    }

    if (!confirm(`Are you sure you want to delete ${ids.length} categor${ids.length > 1 ? 'ies' : 'y'}? This will also delete all associated fields and records.`)) {
        return;
    }

    try {
        showLoading();
        let successCount = 0;
        let failCount = 0;

        for (const id of ids) {
            try {
                await apiRequest(`/categories/${id}`, { method: 'DELETE' });
                successCount++;
            } catch (error) {
                console.error(`Failed to delete category ${id}:`, error);
                failCount++;
            }
        }

        hideLoading();

        if (successCount > 0) {
            showToast(`Successfully deleted ${successCount} categor${successCount > 1 ? 'ies' : 'y'}`, 'success');
        }
        if (failCount > 0) {
            showToast(`Failed to delete ${failCount} categor${failCount > 1 ? 'ies' : 'y'}`, 'error');
        }

        clearCategorySelection();
        loadCategories();
    } catch (error) {
        hideLoading();
        console.error('Bulk delete error:', error);
        showToast('Bulk delete failed: ' + error.message, 'error');
    }
}

function clearCategorySelection() {
    const checkboxes = document.querySelectorAll('.category-checkbox');
    const selectAll = document.getElementById('categorySelectAll');

    checkboxes.forEach(checkbox => {
        checkbox.checked = false;
    });

    if (selectAll) {
        selectAll.checked = false;
    }

    updateCategorySelection();
}

// Field Functions
async function loadFieldsForCategory() {
    const categoryId = document.getElementById('fieldsCategorySelect').value;
    const addFieldBtn = document.getElementById('addFieldBtn');
    
    if (!categoryId) {
        const fieldsTable = document.getElementById("fieldsTable");
        const fieldsEmpty = document.getElementById("fieldsEmpty");

        if (fieldsTable) fieldsTable.classList.add("hidden");
        if (fieldsEmpty) fieldsEmpty.classList.remove("hidden");
        
        // Disable Add Field button when no category is selected
        if (addFieldBtn) addFieldBtn.disabled = true;
        return;
    }
    
    // Enable Add Field button when category is selected
    if (addFieldBtn) addFieldBtn.disabled = false;

    try {
        const res = await apiRequest(`/fields?category_id=${categoryId}`);
        
        let fields = [];
        if (res && res.items && Array.isArray(res.items)) {
            fields = res.items;
        } else if (Array.isArray(res)) {
            fields = res;
        }

        renderFields(fields);
    } catch (error) {
        console.error('Error loading fields:', error);
        showToast('Failed to load fields: ' + error.message, 'error');
        renderFields([]);
    }
}

function formatFieldDefaultValue(dataType, defaultValue, field = null) {
    // Handle special field types
    if (dataType === 'foreign_key') {
        if (field && field.foreign_key_table && field.foreign_key_column) {
            return `<span class="text-blue-600 text-xs">→ ${field.foreign_key_table}.${field.foreign_key_column}</span>`;
        }
        return '<span class="text-blue-600 text-xs">Foreign Key</span>';
    }

    if (dataType === 'array') {
        if (field && field.field_options) {
            const options = typeof field.field_options === 'string' ? JSON.parse(field.field_options) : field.field_options;
            const itemType = options.item_type || 'text';
            const multiple = options.allow_multiple ? 'multiple' : 'single';
            return `<span class="text-green-600 text-xs">Array[${itemType}] (${multiple})</span>`;
        }
        return '<span class="text-green-600 text-xs">Array</span>';
    }

    if (dataType === 'json') {
        if (field && field.field_options) {
            const options = typeof field.field_options === 'string' ? JSON.parse(field.field_options) : field.field_options;
            const valueType = options.value_type || 'text';
            const customKeys = options.allow_custom_keys ? 'custom keys' : 'fixed keys';
            const predefinedKeys = options.predefined_keys && options.predefined_keys.length > 0
                ? `(${options.predefined_keys.slice(0, 3).join(', ')}${options.predefined_keys.length > 3 ? '...' : ''})`
                : '';
            return `<span class="text-purple-600 text-xs">JSON[${valueType}] ${customKeys} ${predefinedKeys}</span>`;
        }
        return '<span class="text-purple-600 text-xs">JSON Key-Value</span>';
    }

    if (dataType === 'rich_paragraph') {
        return '<span class="text-purple-600 text-xs">Rich Text Editor</span>';
    }

    if (!defaultValue) {
        return '<em class="text-gray-400">No default</em>';
    }

    switch(dataType) {
        case 'file':
            return '<em class="text-blue-600">File upload</em>';
        case 'media':
            return '<em class="text-purple-600">Media (Images & Videos)</em>';
        case 'boolean':
            return defaultValue === '1' || defaultValue === 'true' ?
                '<span class="text-green-600">✓ True</span>' :
                '<span class="text-red-600">✗ False</span>';
        case 'date':
        case 'datetime':
            return `<span class="text-purple-600">${escapeHtml(defaultValue)}</span>`;
        case 'number':
        case 'decimal':
            return `<span class="text-blue-600">${escapeHtml(defaultValue)}</span>`;
        default:
            return `<span class="text-gray-800">"${escapeHtml(defaultValue)}"</span>`;
    }
}

function renderFields(fields) {
    const tbody = document.getElementById("fieldsTableBody");
    const emptyState = document.getElementById("fieldsEmpty");
    const table = document.getElementById("fieldsTable");

    if (!tbody) return;

    if (fields.length === 0) {
        if (table) table.classList.add("hidden");
        if (emptyState) emptyState.classList.remove("hidden");
        return;
    }

    if (table) table.classList.remove("hidden");
    if (emptyState) emptyState.classList.add("hidden");

    tbody.innerHTML = fields.map(field => `
        <tr class="hover:bg-gray-50 transition-colors">
            <td class="px-6 py-4 whitespace-nowrap">
                <div class="text-sm font-medium text-gray-900">${escapeHtml(field.name)}</div>
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                    ${escapeHtml(field.data_type)}
                </span>
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                ${field.is_required ?
                    '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">Required</span>' :
                    '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">Optional</span>'
                }
            </td>
            <td class="px-6 py-4">
                <div class="text-sm text-gray-600">${formatFieldDefaultValue(field.data_type, field.default_value, field)}</div>
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                <button onclick="editField('${field.id}')" class="text-blue-600 hover:text-blue-700 mr-3 transition-colors">Edit</button>
                ${field.usage_count > 0
                    ? `<span class="text-gray-400 cursor-not-allowed" title="Cannot delete field used in ${field.usage_count} record(s)">Delete</span>`
                    : `<button onclick="deleteField('${field.id}')" class="text-red-600 hover:text-red-800 transition-colors">Delete</button>`
                }
            </td>
        </tr>
    `).join("");
}

function editField(fieldId) {
    apiRequest(`/fields/${fieldId}`)
        .then(field => {
            editingField = fieldId;
            const fieldModalTitle = document.getElementById("fieldModalTitle");
            const fieldName = document.getElementById("fieldName");
            const fieldType = document.getElementById("fieldType");
            const fieldRequired = document.getElementById("fieldRequired");
            const fieldDefaultValue = document.getElementById("fieldDefaultValue");

            if (fieldModalTitle) fieldModalTitle.textContent = "Edit Field";
            if (fieldName) fieldName.value = field.name;
            if (fieldType) {
                fieldType.value = field.data_type;
                // Trigger field type change to show appropriate UI
                handleFieldTypeChange();
            }
            if (fieldRequired) fieldRequired.checked = field.is_required;
            if (fieldDefaultValue) fieldDefaultValue.value = field.default_value || '';

            // Handle foreign key fields
            if (field.data_type === 'foreign_key' && field.foreign_key_table && field.foreign_key_column) {
                setTimeout(async () => {
                    // Wait for tables to load, then set values
                    const foreignKeyTable = document.getElementById("foreignKeyTable");
                    const foreignKeyColumn = document.getElementById("foreignKeyColumn");

                    if (foreignKeyTable) {
                        foreignKeyTable.value = field.foreign_key_table;
                        // Trigger column loading
                        await loadTableColumns();
                        if (foreignKeyColumn) {
                            foreignKeyColumn.value = field.foreign_key_column;
                        }
                    }
                }, 100);
            }

            // Handle array fields
            if (field.data_type === 'array' && field.field_options) {
                const arrayItemType = document.getElementById("arrayItemType");
                const arrayAllowMultiple = document.getElementById("arrayAllowMultiple");

                if (arrayItemType && field.field_options.item_type) {
                    arrayItemType.value = field.field_options.item_type;
                }
                if (arrayAllowMultiple && typeof field.field_options.allow_multiple !== 'undefined') {
                    arrayAllowMultiple.checked = field.field_options.allow_multiple;
                }
            }

            // Handle JSON fields
            if (field.data_type === 'json' && field.field_options) {
                const jsonValueType = document.getElementById("jsonValueType");
                const jsonAllowCustomKeys = document.getElementById("jsonAllowCustomKeys");
                const jsonPredefinedKeys = document.getElementById("jsonPredefinedKeys");

                if (jsonValueType && field.field_options.value_type) {
                    jsonValueType.value = field.field_options.value_type;
                }
                if (jsonAllowCustomKeys && typeof field.field_options.allow_custom_keys !== 'undefined') {
                    jsonAllowCustomKeys.checked = field.field_options.allow_custom_keys;
                }
                if (jsonPredefinedKeys && field.field_options.predefined_keys) {
                    jsonPredefinedKeys.value = field.field_options.predefined_keys.join(', ');
                }
            }

            // Handle rich paragraph fields
            if (field.data_type === 'rich_paragraph' && field.field_options) {
                const maxCharsInput = document.getElementById("richParagraphMaxChars");

                if (maxCharsInput && field.field_options.max_chars) {
                    maxCharsInput.value = field.field_options.max_chars;
                }
            }

            openModal("fieldModal");
        })
        .catch(error => {
            console.error('Error loading field:', error);
            showToast('Failed to load field details: ' + error.message, 'error');
        });
}

async function deleteField(fieldId) {
    if (!confirm("Are you sure you want to delete this field?")) {
        return;
    }

    try {
        await apiRequest(`/fields/${fieldId}`, {
            method: 'DELETE'
        });

        showToast("Field deleted successfully");
        loadFieldsForCategory();
    } catch (error) {
        console.error('Error deleting field:', error);
        showToast('Failed to delete field: ' + error.message, 'error');
    }
}

// Record Functions
// Pagination state
let recordsCurrentPage = 1;
let recordsPerPage = 10;
let recordsTotalRecords = 0;
let recordsLastSearchQuery = '';
let recordsLastSearchType = 'all';

async function loadRecordsForCategory(searchQuery = '', searchType = 'all', page = 1) {
    const categoryId = document.getElementById('recordsCategorySelect').value;
    const searchContainer = document.getElementById('recordsSearchContainer');
    const paginationContainer = document.getElementById('recordsPagination');

    if (!categoryId) {
        const recordsGrid = document.getElementById("recordsGrid");
        const recordsEmpty = document.getElementById("recordsEmpty");
        const exportButton = document.getElementById("exportButton");

        if (recordsGrid) recordsGrid.innerHTML = "";
        if (recordsEmpty) recordsEmpty.classList.remove("hidden");
        if (searchContainer) searchContainer.classList.add("hidden");
        if (paginationContainer) paginationContainer.classList.add("hidden");
        if (exportButton) exportButton.disabled = true;
        return;
    }

    // Show search container when category is selected
    if (searchContainer) searchContainer.classList.remove("hidden");

    // Enable export button when category is selected
    const exportButton = document.getElementById("exportButton");
    if (exportButton) exportButton.disabled = false;

    // Store search parameters
    recordsLastSearchQuery = searchQuery;
    recordsLastSearchType = searchType;
    recordsCurrentPage = page;

    try {
        // Build query parameters
        const params = new URLSearchParams({
            category_id: categoryId,
            per_page: recordsPerPage.toString(),
            page: page.toString()
        });

        // Add language parameter if selected
        if (currentGlobalLanguage) {
            params.append('lang', currentGlobalLanguage);
        }

        if (searchQuery) {
            params.append('q', searchQuery);
            params.append('search_type', searchType);
        }

        const res = await apiRequest(`/records?${params.toString()}`);

        let records = [];
        let total = 0;

        if (res && res.items && Array.isArray(res.items)) {
            records = res.items;
            // Get total from pagination object if available
            total = res.pagination && res.pagination.total ? res.pagination.total : res.items.length;
        } else if (Array.isArray(res)) {
            records = res;
            total = res.length;
        }

        recordsTotalRecords = total;
        renderRecords(records);
        updateRecordsPagination();
    } catch (error) {
        console.error('Error loading records:', error);
        showToast('Failed to load records: ' + error.message, 'error');
        renderRecords([]);
        if (paginationContainer) paginationContainer.classList.add("hidden");
    }
}

function updateRecordsPagination() {
    const paginationContainer = document.getElementById('recordsPagination');
    if (!paginationContainer) return;

    const totalPages = Math.ceil(recordsTotalRecords / recordsPerPage);

    // Show or hide pagination
    if (recordsTotalRecords > recordsPerPage) {
        paginationContainer.classList.remove('hidden');
    } else {
        paginationContainer.classList.add('hidden');
        return;
    }

    // Update pagination info
    const showingStart = (recordsCurrentPage - 1) * recordsPerPage + 1;
    const showingEnd = Math.min(recordsCurrentPage * recordsPerPage, recordsTotalRecords);

    document.getElementById('recordsShowingStart').textContent = showingStart;
    document.getElementById('recordsShowingEnd').textContent = showingEnd;
    document.getElementById('recordsTotal').textContent = recordsTotalRecords;
    document.getElementById('recordsCurrentPage').textContent = recordsCurrentPage;
    document.getElementById('recordsTotalPages').textContent = totalPages;

    // Enable/disable buttons
    const prevButtons = ['recordsPrevMobile', 'recordsPrevDesktop'];
    const nextButtons = ['recordsNextMobile', 'recordsNextDesktop'];

    prevButtons.forEach(id => {
        const btn = document.getElementById(id);
        if (btn) {
            btn.disabled = recordsCurrentPage === 1;
            btn.classList.toggle('opacity-50', recordsCurrentPage === 1);
            btn.classList.toggle('cursor-not-allowed', recordsCurrentPage === 1);
        }
    });

    nextButtons.forEach(id => {
        const btn = document.getElementById(id);
        if (btn) {
            btn.disabled = recordsCurrentPage >= totalPages;
            btn.classList.toggle('opacity-50', recordsCurrentPage >= totalPages);
            btn.classList.toggle('cursor-not-allowed', recordsCurrentPage >= totalPages);
        }
    });
}

function nextRecordsPage() {
    const totalPages = Math.ceil(recordsTotalRecords / recordsPerPage);
    if (recordsCurrentPage < totalPages) {
        loadRecordsForCategory(recordsLastSearchQuery, recordsLastSearchType, recordsCurrentPage + 1);
    }
}

function previousRecordsPage() {
    if (recordsCurrentPage > 1) {
        loadRecordsForCategory(recordsLastSearchQuery, recordsLastSearchType, recordsCurrentPage - 1);
    }
}

// New search function for records
function searchRecords() {
    const searchInput = document.getElementById('recordsSearchInput');
    const searchType = document.getElementById('recordsSearchType');

    if (!searchInput || !searchType) return;

    const query = searchInput.value.trim();
    const type = searchType.value;

    loadRecordsForCategory(query, type);
}

// Clear search function
function clearRecordsSearch() {
    const searchInput = document.getElementById('recordsSearchInput');
    const searchType = document.getElementById('recordsSearchType');

    if (searchInput) searchInput.value = '';
    if (searchType) searchType.value = 'all';

    loadRecordsForCategory();
}

// Export database function
async function exportDatabaseSQL() {
    const categoryId = document.getElementById('recordsCategorySelect').value;

    if (!categoryId) {
        showToast('Please select a category first', 'error');
        return;
    }

    try {
        const exportButton = document.getElementById('exportButton');
        if (exportButton) {
            exportButton.disabled = true;
            exportButton.textContent = 'Exporting...';
        }

        // Use a direct fetch for file download since apiRequest expects JSON responses
        const authToken = localStorage.getItem('auth_token');
        const headers = {
            'Authorization': authToken ? `Bearer ${authToken}` : ''
        };

        const response = await fetch(`${window.location.origin}${window.location.pathname}/records/export?category_id=${categoryId}`, {
            method: 'GET',
            headers: headers
        });

        if (!response.ok) {
            const errorText = await response.text();
            console.error('Export response error:', response.status, errorText);
            throw new Error(`Export failed: ${response.status} - ${errorText}`);
        }

        // Get the filename from the Content-Disposition header or create a default one
        const contentDisposition = response.headers.get('Content-Disposition');
        let filename = `database_export_category_${categoryId}.sql`;
        if (contentDisposition) {
            const filenameMatch = contentDisposition.match(/filename="?([^"]+)"?/);
            if (filenameMatch) {
                filename = filenameMatch[1];
            }
        }

        // Get the SQL content
        const sqlContent = await response.text();

        // Validate that we got SQL content
        if (!sqlContent || sqlContent.trim() === '') {
            throw new Error('Empty export file received');
        }

        // Create and trigger download
        const blob = new Blob([sqlContent], { type: 'application/sql' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);

        showToast('Database exported successfully', 'success');
    } catch (error) {
        console.error('Export error:', error);
        showToast('Failed to export database: ' + error.message, 'error');
    } finally {
        const exportButton = document.getElementById('exportButton');
        if (exportButton) {
            exportButton.disabled = false;
            exportButton.textContent = 'Export Database';
        }
    }
}

// Add event listener for Enter key on search input
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('recordsSearchInput');
    if (searchInput) {
        searchInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                searchRecords();
            }
        });
    }

    // File input change listener for multiple file preview
    const fileInput = document.getElementById('uploadFile');
    if (fileInput) {
        fileInput.addEventListener('change', function() {
            showSelectedFilesPreview();
        });
    }
});

function formatRecordFieldValue(key, value) {
    if (!value) return `<span class="text-sm text-gray-600">${escapeHtml(key)}: <em class="text-gray-400">Empty</em></span>`;
    
    const valueStr = String(value);
    
    // Check if this is a file path (contains protocol or starts with uploads/)
    const isFilePath = valueStr.match(/^https?:\/\//) || valueStr.includes('/uploads/') || valueStr.includes('uploads/');
    
    if (isFilePath) {
        // Handle multiple files (comma-separated)
        const urls = valueStr.split(',').filter(url => url.trim());
        
        if (urls.length === 1) {
            // Single file
            const url = urls[0].trim();
            const fileName = url.split('/').pop();
            
            // Check if it's an image
            const isImage = fileName.match(/\.(jpg|jpeg|png|gif|webp|svg)$/i);
            
            if (isImage) {
                return `<span class="text-sm text-gray-600">${escapeHtml(key)}: <br><img src="${escapeHtml(makeRelativeUrl(url))}" alt="${escapeHtml(fileName)}" class="mt-1 w-16 h-16 object-cover rounded border"></span>`;
            } else {
                return `<span class="text-sm text-gray-600">${escapeHtml(key)}: <span class="inline-flex items-center px-2 py-1 bg-gray-100 rounded text-xs">${escapeHtml(fileName)}</span></span>`;
            }
        } else {
            // Multiple files
            const fileCount = urls.length;
            const firstFile = urls[0].trim();
            const firstName = firstFile.split('/').pop();
            const isImage = firstName.match(/\.(jpg|jpeg|png|gif|webp|svg)$/i);
            
            if (isImage) {
                return `<span class="text-sm text-gray-600">${escapeHtml(key)}: <br><div class="flex items-center"><img src="${escapeHtml(firstFile)}" alt="${escapeHtml(firstName)}" class="mt-1 w-12 h-12 object-cover rounded border"><span class="ml-2 text-xs text-gray-500">+${fileCount-1} more</span></div></span>`;
            } else {
                return `<span class="text-sm text-gray-600">${escapeHtml(key)}: <span class="inline-flex items-center px-2 py-1 bg-gray-100 rounded text-xs">${escapeHtml(firstName)} +${fileCount-1} more</span></span>`;
            }
        }
    } else {
        // Regular field - truncate if too long
        const truncated = valueStr.length > 30 ? valueStr.substring(0, 30) + '...' : valueStr;
        return `<span class="text-sm text-gray-600">${escapeHtml(key)}: ${escapeHtml(truncated)}</span>`;
    }
}

function renderRecords(records) {
    const recordsGrid = document.getElementById("recordsGrid");
    const recordsTable = document.getElementById("recordsTable");
    const recordsTableHead = document.getElementById("recordsTableHead");
    const recordsTableBody = document.getElementById("recordsTableBody");
    const emptyState = document.getElementById("recordsEmpty");
    const recordsViewToggle = document.getElementById("recordsViewToggle");

    if (!recordsGrid || !recordsTable) return;

    if (records.length === 0) {
        recordsGrid.innerHTML = "";
        if (recordsTableBody) recordsTableBody.innerHTML = "";
        if (emptyState) emptyState.classList.remove("hidden");
        if (recordsViewToggle) recordsViewToggle.classList.add("hidden");
        return;
    }

    if (emptyState) emptyState.classList.add("hidden");
    if (recordsViewToggle) recordsViewToggle.classList.remove("hidden");

    if (recordsViewMode === 'card') {
        renderRecordsCard(records);
    } else {
        renderRecordsTable(records);
    }
}

function renderRecordsCard(records) {
    const recordsGrid = document.getElementById("recordsGrid");

    recordsGrid.innerHTML = records.map(record => {
        const displayValues = Object.entries(record.values || {}).slice(0, 3).map(([key, value]) => {
            return formatRecordFieldValue(key, value);
        }).join('<br>');

        return `
        <div class="bg-white rounded-lg shadow-lg border border-gray-200 p-6 hover:shadow-xl transition-all">
            <div class="space-y-3">
                <div class="flex items-start justify-between">
                    <div class="flex-1 space-y-2">${displayValues}</div>
                    <input type="checkbox" class="record-checkbox ml-2 rounded border-gray-300 text-blue-600 focus:ring-blue-500" data-id="${record.id}" onchange="updateRecordSelection()">
                </div>
                <div class="text-xs text-gray-500">
                    Created: ${record.created_at ? new Date(record.created_at).toLocaleDateString() : new Date().toLocaleDateString()}
                </div>
                <div class="flex justify-end space-x-2 pt-4 border-t border-gray-200">
                    <button onclick="editRecord('${record.id}')" class="text-blue-600 hover:text-blue-700 text-sm font-medium transition-colors">Edit</button>
                    <button onclick="deleteRecord('${record.id}')" class="text-red-600 hover:text-red-800 text-sm font-medium transition-colors">Delete</button>
                </div>
            </div>
        </div>
        `;
    }).join("");
}

function renderRecordsTable(records) {
    const recordsTableHead = document.getElementById("recordsTableHead");
    const recordsTableBody = document.getElementById("recordsTableBody");

    if (!recordsTableHead || !recordsTableBody || records.length === 0) return;

    // Get all unique field names from records
    const allFields = new Set();
    records.forEach(record => {
        Object.keys(record.values || {}).forEach(field => allFields.add(field));
    });

    const fieldNames = Array.from(allFields);

    // Generate table headers
    recordsTableHead.innerHTML = `
        <tr>
            <th class="px-6 py-3 text-left">
                <input type="checkbox" class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
            </th>
            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
            ${fieldNames.map(field => `
                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">${escapeHtml(field)}</th>
            `).join('')}
            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created</th>
            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
        </tr>
    `;

    // Generate table rows
    recordsTableBody.innerHTML = records.map(record => {
        const fieldCells = fieldNames.map(field => {
            const value = record.values[field];
            const isFallback = record._fallback_fields && record._fallback_fields.includes(field);
            const fallbackBadge = isFallback ? '<span class="ml-1 inline-flex items-center px-1.5 py-0.5 rounded text-xs font-medium bg-yellow-100 text-yellow-800" title="Using default language">Fallback</span>' : '';
            return `<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">${formatTableFieldValue(field, value)}${fallbackBadge}</td>`;
        }).join('');

        return `
        <tr class="hover:bg-gray-50">
            <td class="px-6 py-4 whitespace-nowrap">
                <input type="checkbox" class="record-checkbox rounded border-gray-300 text-blue-600 focus:ring-blue-500" data-id="${record.id}" onchange="updateRecordSelection()">
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">${record.id}</td>
            ${fieldCells}
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                ${record.created_at ? new Date(record.created_at).toLocaleDateString() : new Date().toLocaleDateString()}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                <button onclick="editRecord('${record.id}')" class="text-blue-600 hover:text-blue-700 transition-colors">Edit</button>
                <button onclick="deleteRecord('${record.id}')" class="text-red-600 hover:text-red-800 transition-colors">Delete</button>
            </td>
        </tr>
        `;
    }).join('');
}

function formatTableFieldValue(key, value) {
    if (!value) return '';

    const valueStr = String(value);

    // Check if this is a file path
    const isFilePath = valueStr.match(/^https?:\/\//) || valueStr.includes('/uploads/') || valueStr.includes('uploads/');

    if (isFilePath) {
        const urls = valueStr.split(',').filter(url => url.trim());
        if (urls.length === 1) {
            const url = urls[0].trim();
            const fileName = url.split('/').pop();
            const isImage = fileName.match(/\.(jpg|jpeg|png|gif|webp|svg)$/i);

            if (isImage) {
                return `<img src="${escapeHtml(makeRelativeUrl(url))}" alt="${escapeHtml(fileName)}" class="w-12 h-12 object-cover rounded border">`;
            } else {
                return `<span class="inline-flex items-center px-2 py-1 bg-gray-100 rounded text-xs">${escapeHtml(fileName)}</span>`;
            }
        } else {
            return `<span class="text-blue-600">${urls.length} files</span>`;
        }
    }

    // Truncate long text
    if (valueStr.length > 50) {
        return `<span title="${escapeHtml(valueStr)}">${escapeHtml(valueStr.substring(0, 50))}...</span>`;
    }

    return escapeHtml(valueStr);
}

// Switch between card and table view for records
function switchRecordsView(viewMode) {
    recordsViewMode = viewMode;

    const cardViewBtn = document.getElementById('cardViewBtn');
    const tableViewBtn = document.getElementById('tableViewBtn');
    const recordsGrid = document.getElementById('recordsGrid');
    const recordsTable = document.getElementById('recordsTable');

    if (viewMode === 'card') {
        // Activate card view
        cardViewBtn.classList.add('bg-white', 'text-gray-700', 'shadow-sm');
        cardViewBtn.classList.remove('text-gray-500');
        tableViewBtn.classList.add('text-gray-500');
        tableViewBtn.classList.remove('bg-white', 'text-gray-700', 'shadow-sm');

        recordsGrid.classList.remove('hidden');
        recordsTable.classList.add('hidden');
    } else {
        // Activate table view
        tableViewBtn.classList.add('bg-white', 'text-gray-700', 'shadow-sm');
        tableViewBtn.classList.remove('text-gray-500');
        cardViewBtn.classList.add('text-gray-500');
        cardViewBtn.classList.remove('bg-white', 'text-gray-700', 'shadow-sm');

        recordsGrid.classList.add('hidden');
        recordsTable.classList.remove('hidden');
    }

    // Re-render current records in the new view
    const categoryId = document.getElementById('recordsCategorySelect').value;
    if (categoryId) {
        loadRecordsForCategory();
    }
}

function addNewRecord() {
    const categoryId = document.getElementById('recordsCategorySelect').value;
    if (!categoryId) {
        showToast('Please select a category first', 'warning');
        return;
    }
    editingRecord = null;
    generateRecordForm(categoryId);
    openModal('recordModal');
}

function editRecord(recordId) {
    // Find record and generate form
    const categoryId = document.getElementById('recordsCategorySelect').value;
    if (categoryId) {
        editingRecord = recordId;
        generateRecordForm(categoryId, recordId);
        openModal("recordModal");
    }
}

async function deleteRecord(recordId) {
    if (!confirm("Are you sure you want to delete this record? This action cannot be undone.")) {
        return;
    }

    try {
        showLoading();

        await apiRequest(`/records/${recordId}`, {
            method: 'DELETE'
        });

        showToast("Record deleted successfully", "success");
        loadRecordsForCategory();
    } catch (error) {
        console.error('Error deleting record:', error);
        showToast('Failed to delete record: ' + error.message, 'error');
    } finally {
        hideLoading();
    }
}

// Bulk Record Operations
function updateRecordSelection() {
    const checkboxes = document.querySelectorAll('.record-checkbox:checked');
    const count = checkboxes.length;
    const bulkActionsBar = document.getElementById('recordBulkActionsBar');
    const selectedCount = document.getElementById('recordSelectedCount');

    if (count > 0) {
        bulkActionsBar.classList.remove('hidden');
        selectedCount.textContent = count;
    } else {
        bulkActionsBar.classList.add('hidden');
    }
}

async function bulkDeleteRecords() {
    const checkboxes = document.querySelectorAll('.record-checkbox:checked');
    const ids = Array.from(checkboxes).map(cb => cb.dataset.id);

    if (ids.length === 0) {
        showToast('No records selected', 'warning');
        return;
    }

    if (!confirm(`Are you sure you want to delete ${ids.length} record${ids.length > 1 ? 's' : ''}? This action cannot be undone.`)) {
        return;
    }

    try {
        showLoading();
        let successCount = 0;
        let failCount = 0;

        for (const id of ids) {
            try {
                await apiRequest(`/records/${id}`, { method: 'DELETE' });
                successCount++;
            } catch (error) {
                console.error(`Failed to delete record ${id}:`, error);
                failCount++;
            }
        }

        hideLoading();

        if (successCount > 0) {
            showToast(`Successfully deleted ${successCount} record${successCount > 1 ? 's' : ''}`, 'success');
        }
        if (failCount > 0) {
            showToast(`Failed to delete ${failCount} record${failCount > 1 ? 's' : ''}`, 'error');
        }

        clearRecordSelection();
        loadRecordsForCategory();
    } catch (error) {
        hideLoading();
        console.error('Bulk delete error:', error);
        showToast('Bulk delete failed: ' + error.message, 'error');
    }
}

function clearRecordSelection() {
    const checkboxes = document.querySelectorAll('.record-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.checked = false;
    });
    updateRecordSelection();
}

async function generateRecordForm(categoryId, recordId = null) {
    try {
        const fieldsRes = await apiRequest(`/fields?category_id=${categoryId}`);
        let fields = [];
        
        if (fieldsRes && fieldsRes.items && Array.isArray(fieldsRes.items)) {
            fields = fieldsRes.items;
        } else if (Array.isArray(fieldsRes)) {
            fields = fieldsRes;
        }

        currentCategoryFields = fields;

        const container = document.getElementById("recordFields");
        const modalTitle = document.getElementById("recordModalTitle");

        if (!container) return;

        if (modalTitle) {
            modalTitle.textContent = recordId ? "Edit Record" : "Add Record";
        }

        if (fields.length === 0) {
            container.innerHTML = `
                <div class="text-center py-8">
                    <p class="text-gray-500">No fields defined for this category yet.</p>
                    <p class="text-sm text-gray-400 mt-2">Add some fields to the category first.</p>
                </div>
            `;
            return;
        }

        // Get existing record data if editing
        let existingRecord = null;
        let isViewingTranslation = false;
        if (recordId) {
            // Load record with current language if selected
            let apiUrl = `/records?category_id=${categoryId}`;
            if (currentGlobalLanguage) {
                apiUrl += `&lang=${currentGlobalLanguage}`;
                isViewingTranslation = true;
            }

            const records = await apiRequest(apiUrl);
            const recordsList = records.items || records || [];
            existingRecord = recordsList.find(r => r.id == recordId);
        }

        // Update modal title to show language
        if (modalTitle && isViewingTranslation) {
            const langSelect = document.getElementById('globalLanguageSelect');
            const langName = langSelect ? langSelect.options[langSelect.selectedIndex].text : currentGlobalLanguage;
            modalTitle.textContent = `View Record (${langName}) - Read Only`;
        }

        const slugField = `
            <div class="space-y-2">
                <label class="block text-sm font-semibold text-gray-900">
                    Slug <span class="text-red-500">*</span>
                </label>
                <input type="text" id="record_slug" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors" value="${existingRecord ? escapeHtml(existingRecord.slug || '') : ''}" onchange="validateSlugInput(this)" oninput="validateSlugInput(this)" required>
                <p class="text-xs text-gray-500">Letters, numbers, and hyphens only. Example: my-article-slug or My-Article-Slug</p>
            </div>`;

        const tagsField = `
            <div class="space-y-2" style="display: none;" >
                <label class="block text-sm font-semibold text-gray-900">
                    Tags
                </label>
                <input type="text" id="record_tags" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors" value="${existingRecord && existingRecord.values && existingRecord.values.tags ? escapeHtml(existingRecord.values.tags) : ''}" placeholder="Enter comma-separated tags">
                <p class="text-xs text-gray-500">Separate multiple tags with commas. Example: web, design, portfolio</p>
            </div>`;

        // Determine if fields should be disabled
        const disabledAttr = isViewingTranslation ? 'disabled' : '';
        const disabledClass = isViewingTranslation ? 'bg-gray-100 cursor-not-allowed' : '';

        container.innerHTML = slugField + tagsField + fields.map(field => {
            const fieldId = `record_${field.name.replace(/\s+/g, '_')}`;

            // Get existing value - handle different data types properly
            let existingValue;
            if (existingRecord && field.name in existingRecord.values) {
                existingValue = existingRecord.values[field.name];
                // For arrays and objects, keep them as-is (don't default to empty string)
                if (existingValue === null || existingValue === undefined) {
                    existingValue = field.default_value || '';
                }
            } else {
                existingValue = field.default_value || '';
            }

            let inputHtml = '';

            switch (field.data_type) {
                case 'text':
                    inputHtml = `<input type="text" id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${disabledClass}" value="${escapeHtml(existingValue)}" ${field.is_required ? 'required' : ''} ${disabledAttr}>`;
                    break;
                case 'textarea':
                    inputHtml = `<textarea id="${fieldId}" rows="4" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors resize-none" ${field.is_required ? 'required' : ''}>${escapeHtml(existingValue)}</textarea>`;
                    break;
                case 'number':
                    inputHtml = `<input type="number" id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors" value="${escapeHtml(existingValue)}" ${field.is_required ? 'required' : ''}>`;
                    break;
                case 'email':
                    inputHtml = `<input type="email" id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors" value="${escapeHtml(existingValue)}" ${field.is_required ? 'required' : ''}>`;
                    break;
                case 'url':
                    inputHtml = `<input type="url" id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors" value="${escapeHtml(existingValue)}" ${field.is_required ? 'required' : ''}>`;
                    break;
                case 'date':
                    inputHtml = `<input type="date" id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors" value="${escapeHtml(existingValue)}" ${field.is_required ? 'required' : ''}>`;
                    break;
                case 'checkbox':
                    const isChecked = existingValue === 'true' || existingValue === true || existingValue === 1 || existingValue === '1';
                    inputHtml = `<div class="flex items-center">
                        <input type="checkbox" id="${fieldId}" class="w-5 h-5 text-blue-500 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 focus:ring-2" ${isChecked ? 'checked' : ''}>
                        <label for="${fieldId}" class="ml-3 text-sm text-gray-700">Enable ${escapeHtml(field.name)}</label>
                    </div>`;
                    break;
                case 'boolean':
                    const boolValue = existingValue === 'true' || existingValue === true || existingValue === 1 || existingValue === '1';
                    inputHtml = `<select id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors ${disabledClass}" ${field.is_required ? 'required' : ''} ${disabledAttr}>
                        <option value="false" ${!boolValue ? 'selected' : ''}>False</option>
                        <option value="true" ${boolValue ? 'selected' : ''}>True</option>
                    </select>`;
                    break;
                case 'image':
                    const existingImages = existingValue ? (Array.isArray(existingValue) ? existingValue : existingValue.split(',').filter(v => v.trim())) : [];
                    inputHtml = `<div class="space-y-4">
                        <button type="button" onclick="openMediaSelectorForRecord('${fieldId}', 'image')" class="w-full px-4 py-3 bg-blue-500 text-white border border-gray-200 rounded-xl hover:bg-blue-600 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors">
                            Select Images from Media
                        </button>
                        <input type="hidden" id="${fieldId}" value="${escapeHtml(Array.isArray(existingValue) ? existingValue.join(',') : existingValue)}" ${field.is_required ? 'required' : ''}>
                        <div id="${fieldId}_preview" class="mt-2 ${existingImages.length > 0 ? '' : 'hidden'}">
                            <div class="grid grid-cols-3 gap-2" id="${fieldId}_preview_grid">
                                ${existingImages.map((url, index) => `
                                    <div class="relative">
                                        <img src="${escapeHtml(makeRelativeUrl(url))}" alt="Selected image" class="w-20 h-20 object-cover rounded border">
                                        <button type="button" onclick="removeSelectedMedia('${fieldId}', ${index})" class="absolute top-1 bg-red-500 text-white rounded-full w-5 h-5 text-xs flex items-center justify-center hover:bg-red-600 shadow-lg">×</button>
                                    </div>
                                `).join('')}
                            </div>
                            <button type="button" onclick="clearRecordMedia('${fieldId}')" class="text-red-600 text-sm hover:text-red-800 mt-2">Clear All</button>
                        </div>
                    </div>`;
                    break;
                case 'file':
                    const existingFiles = existingValue ? (Array.isArray(existingValue) ? existingValue : existingValue.split(',').filter(v => v.trim())) : [];
                    inputHtml = `<div class="space-y-4">
                        <button type="button" onclick="openMediaSelectorForRecord('${fieldId}', 'file')" class="w-full px-4 py-3 bg-blue-500 text-white border border-gray-200 rounded-xl hover:bg-blue-600 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors">
                            Select Files from Media
                        </button>
                        <input type="hidden" id="${fieldId}" value="${escapeHtml(Array.isArray(existingValue) ? existingValue.join(',') : existingValue)}" ${field.is_required ? 'required' : ''}>
                        <div id="${fieldId}_preview" class="mt-2 ${existingFiles.length > 0 ? '' : 'hidden'}">
                            <div class="space-y-2" id="${fieldId}_preview_list">
                                ${existingFiles.map((url, index) => `
                                    <div class="relative flex items-center p-2 bg-gray-100 rounded">
                                        <div class="flex items-center space-x-2 pr-6">
                                            <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                                            </svg>
                                            <span>${url.split('/').pop()}</span>
                                        </div>
                                        <button type="button" onclick="removeSelectedMedia('${fieldId}', ${index})" class="absolute right-1 top-1/2 transform -translate-y-1/2 bg-red-500 text-white rounded-full w-5 h-5 text-xs flex items-center justify-center hover:bg-red-600">×</button>
                                    </div>
                                `).join('')}
                            </div>
                            <button type="button" onclick="clearRecordMedia('${fieldId}')" class="text-red-600 text-sm hover:text-red-800 mt-2">Clear All</button>
                        </div>
                    </div>`;
                    break;
                case 'media':
                    const existingMedia = existingValue ? (Array.isArray(existingValue) ? existingValue : existingValue.split(',').filter(v => v.trim())) : [];
                    inputHtml = `<div class="space-y-4">
                        <button type="button" onclick="openMediaSelectorForRecord('${fieldId}', 'media')" class="w-full px-4 py-3 bg-blue-500 text-white border border-gray-200 rounded-xl hover:bg-blue-600 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors">
                            Select Images & Videos from Media
                        </button>
                        <input type="hidden" id="${fieldId}" value="${escapeHtml(Array.isArray(existingValue) ? existingValue.join(',') : existingValue)}" ${field.is_required ? 'required' : ''}>
                        <div id="${fieldId}_preview" class="mt-2 ${existingMedia.length > 0 ? '' : 'hidden'}">
                            <div class="grid grid-cols-3 gap-2" id="${fieldId}_preview_grid">
                                ${existingMedia.map((url, index) => {
                                    const isVideo = url.match(/\.(mp4|webm|ogg|mov|avi|mkv)$/i);
                                    if (isVideo) {
                                        return `<div class="relative">
                                            <video src="${escapeHtml(makeRelativeUrl(url))}" class="w-20 h-20 object-cover rounded border" muted preload="metadata"></video>
                                            <button type="button" onclick="removeSelectedMedia('${fieldId}', ${index})" class="absolute top-1 bg-red-500 text-white rounded-full w-5 h-5 text-xs flex items-center justify-center hover:bg-red-600 shadow-lg">×</button>
                                        </div>`;
                                    } else {
                                        return `<div class="relative">
                                            <img src="${escapeHtml(makeRelativeUrl(url))}" alt="Selected media" class="w-20 h-20 object-cover rounded border">
                                            <button type="button" onclick="removeSelectedMedia('${fieldId}', ${index})" class="absolute top-1 bg-red-500 text-white rounded-full w-5 h-5 text-xs flex items-center justify-center hover:bg-red-600 shadow-lg">×</button>
                                        </div>`;
                                    }
                                }).join('')}
                            </div>
                            <button type="button" onclick="clearRecordMedia('${fieldId}')" class="text-red-600 text-sm hover:text-red-800 mt-2">Clear All</button>
                        </div>
                    </div>`;
                    break;
                case 'foreign_key':
                    // Generate dropdown with options from referenced table
                    inputHtml = `<div class="relative">
                        <select id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors appearance-none bg-white" ${field.is_required ? 'required' : ''}>
                            <option value="">Loading options...</option>
                        </select>
                        <div class="absolute inset-y-0 right-0 flex items-center px-3 pointer-events-none">
                            <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </div>
                    </div>`;
                    // Load foreign key options asynchronously
                    setTimeout(() => loadForeignKeyOptions(fieldId, field, existingValue), 100);
                    break;
                case 'array':
                    let arrayOptions = {};
                    try {
                        arrayOptions = field.field_options ? (typeof field.field_options === 'string' ? JSON.parse(field.field_options) : field.field_options) : {};
                    } catch (e) {
                        console.error('Error parsing field options:', e);
                        arrayOptions = {};
                    }
                    const itemType = arrayOptions.item_type || 'text';
                    const allowMultiple = arrayOptions.allow_multiple !== false;
                    let arrayValues = [];
                    console.log('Array field processing - existingValue:', existingValue, 'type:', typeof existingValue);
                    if (existingValue) {
                        if (Array.isArray(existingValue)) {
                            arrayValues = existingValue;
                        } else if (typeof existingValue === 'string') {
                            try {
                                // Try to parse as JSON first
                                const parsed = JSON.parse(existingValue);
                                if (Array.isArray(parsed)) {
                                    arrayValues = parsed;
                                } else {
                                    // If JSON parsed but not an array, convert to array
                                    arrayValues = [parsed];
                                }
                            } catch (e) {
                                // If JSON parsing fails, treat as comma-separated string
                                if (existingValue.trim()) {
                                    arrayValues = existingValue.split(',').map(v => v.trim()).filter(v => v);
                                } else {
                                    arrayValues = [];
                                }
                            }
                        } else {
                            // If existingValue is not a string or array, convert to array
                            arrayValues = [existingValue];
                        }
                    }

                    // Final safety check - ensure arrayValues is always an array
                    if (!Array.isArray(arrayValues)) {
                        console.warn('arrayValues is not an array after processing:', arrayValues);
                        arrayValues = [];
                    }
                    console.log('Array field final arrayValues:', arrayValues);

                    inputHtml = `<div class="space-y-3" data-array-type="${itemType}">
                        <div id="${fieldId}_container" class="space-y-2">
                            ${arrayValues.map((value, index) => `
                                <div class="flex items-center space-x-2 array-item">
                                    <input type="${itemType === 'number' ? 'number' : itemType === 'email' ? 'email' : itemType === 'url' ? 'url' : 'text'}"
                                           class="flex-1 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors array-item-input"
                                           value="${escapeHtml(value)}"
                                           placeholder="Enter ${itemType} value"
                                           onchange="updateArrayFieldValue('${fieldId}')"
                                           onkeyup="updateArrayFieldValue('${fieldId}')">
                                    <button type="button" onclick="removeArrayItem(this)" class="px-2 py-2 text-red-600 hover:text-red-800 hover:bg-red-50 rounded-lg transition-colors">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                    </button>
                                </div>
                            `).join('')}
                        </div>
                        <button type="button" onclick="addArrayItem('${fieldId}', '${itemType}')" class="w-full px-3 py-2 border-2 border-dashed border-gray-300 text-gray-600 rounded-lg hover:border-blue-400 hover:text-blue-600 transition-colors">
                            <svg class="w-4 h-4 inline mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                            </svg>
                            Add ${itemType} item
                        </button>
                        <input type="hidden" id="${fieldId}" value="${escapeHtml(JSON.stringify(arrayValues))}" ${field.is_required ? 'required' : ''}>
                    </div>`;
                    break;
                case 'json':
                    let jsonOptions = {};
                    try {
                        jsonOptions = field.field_options ? (typeof field.field_options === 'string' ? JSON.parse(field.field_options) : field.field_options) : {};
                    } catch (e) {
                        console.error('Error parsing JSON field options:', e);
                        jsonOptions = {};
                    }

                    const valueType = jsonOptions.value_type || 'text';
                    const allowCustomKeys = jsonOptions.allow_custom_keys !== false;
                    const predefinedKeys = jsonOptions.predefined_keys || [];

                    let jsonValues = {};
                    if (existingValue) {
                        try {
                            if (typeof existingValue === 'string') {
                                jsonValues = JSON.parse(existingValue);
                            } else if (typeof existingValue === 'object') {
                                jsonValues = existingValue;
                            }
                        } catch (e) {
                            console.error('Error parsing existing JSON value:', e);
                            jsonValues = {};
                        }
                    }

                    // Ensure jsonValues is always an object
                    if (typeof jsonValues !== 'object' || Array.isArray(jsonValues) || jsonValues === null) {
                        jsonValues = {};
                    }

                    inputHtml = `<div class="space-y-3" data-json-type="${valueType}">
                        <div id="${fieldId}_container" class="space-y-2">
                            ${Object.entries(jsonValues).map(([key, value]) => `
                                <div class="flex items-center space-x-2 json-item">
                                    ${allowCustomKeys ?
                                        `<input type="text" class="w-32 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors json-key-input" value="${escapeHtml(key)}" placeholder="Key">` :
                                        `<select class="w-32 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors json-key-select">
                                            ${predefinedKeys.map(preKey =>
                                                `<option value="${escapeHtml(preKey)}" ${preKey === key ? 'selected' : ''}>${escapeHtml(preKey)}</option>`
                                            ).join('')}
                                        </select>`
                                    }
                                    ${valueType === 'textarea' ?
                                        `<textarea class="flex-1 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors json-value-input" rows="2" placeholder="Enter value">${escapeHtml(value)}</textarea>` :
                                        valueType === 'checkbox' ?
                                        `<div class="flex-1 flex items-center"><input type="checkbox" class="w-4 h-4 text-blue-500 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 focus:ring-2 json-value-input" ${value === true || value === 'true' || value === '1' ? 'checked' : ''}><span class="ml-2 text-sm text-gray-700">True/False</span></div>` :
                                        `<input type="${valueType === 'number' ? 'number' : valueType === 'email' ? 'email' : valueType === 'url' ? 'url' : valueType === 'date' ? 'date' : 'text'}" class="flex-1 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors json-value-input" value="${escapeHtml(value)}" placeholder="Enter value">`
                                    }
                                    <button type="button" onclick="removeJsonItem(this)" class="px-2 py-2 text-red-600 hover:text-red-800 hover:bg-red-50 rounded-lg transition-colors">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                    </button>
                                </div>
                            `).join('')}
                        </div>
                        <button type="button" onclick="addJsonItem('${fieldId}', '${valueType}', ${allowCustomKeys}, '${predefinedKeys.join(',')}')" class="w-full px-3 py-2 border-2 border-dashed border-gray-300 text-gray-600 rounded-lg hover:border-purple-400 hover:text-purple-600 transition-colors">
                            <svg class="w-4 h-4 inline mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                            </svg>
                            Add key-value pair
                        </button>
                        <input type="hidden" id="${fieldId}" value="${escapeHtml(JSON.stringify(jsonValues))}" ${field.is_required ? 'required' : ''}>
                    </div>`;
                    break;
                case 'link':
                    // Handle link field type - Array of JSON objects with media and text keys
                    let linkValues = [];
                    console.log(`Generating link field ${field.name}, existingValue:`, existingValue, 'Type:', typeof existingValue);

                    if (existingValue) {
                        try {
                            if (typeof existingValue === 'string') {
                                const parsed = JSON.parse(existingValue);
                                if (Array.isArray(parsed)) {
                                    linkValues = parsed;
                                } else if (parsed && typeof parsed === 'object') {
                                    // Convert single object to array for backward compatibility
                                    linkValues = [parsed];
                                }
                            } else if (Array.isArray(existingValue)) {
                                linkValues = existingValue;
                            } else if (typeof existingValue === 'object' && existingValue !== null) {
                                linkValues = [existingValue];
                            }
                        } catch (e) {
                            console.error('Error parsing existing link value:', e, 'Value:', existingValue);
                            linkValues = [];
                        }
                    }

                    // Ensure linkValues is always an array
                    if (!Array.isArray(linkValues)) {
                        console.warn(`linkValues is not an array for field ${field.name}, forcing to array. Value:`, linkValues);
                        linkValues = [];
                    }

                    console.log(`Final linkValues for ${field.name}:`, linkValues, 'Count:', linkValues.length);

                    inputHtml = `<div class="space-y-4 p-4 border border-gray-200 rounded-xl bg-gray-50">
                        <div class="flex items-center justify-between mb-2">
                            <div class="text-sm font-medium text-gray-700">Link Entries</div>
                            <button type="button" onclick="addLinkEntry('${fieldId}')" class="px-3 py-1 bg-green-500 text-white text-xs rounded-md hover:bg-green-600 transition-colors">
                                + Add Link
                            </button>
                        </div>

                        <div id="${fieldId}_container" class="space-y-4">
                            ${linkValues.map((linkValue, index) => `
                                <div class="link-entry bg-white p-4 rounded-lg border border-gray-200" data-index="${index}">
                                    <div class="flex items-center justify-between mb-3">
                                        <div class="text-sm font-medium text-gray-600">Link #${index + 1}</div>
                                        <button type="button" onclick="removeLinkEntry('${fieldId}', ${index})" class="px-2 py-1 text-red-600 hover:text-red-800 hover:bg-red-50 rounded transition-colors">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                            </svg>
                                        </button>
                                    </div>

                                    <div class="space-y-3">
                                        <div>
                                            <label class="block text-sm font-medium text-gray-600 mb-1">Image</label>
                                            <button type="button" onclick="openMediaSelectorForLinkEntry('${fieldId}', ${index})" class="w-full px-3 py-2 bg-blue-500 text-white text-sm rounded-lg hover:bg-blue-600 transition-colors">
                                                Select Image
                                            </button>
                                            <input type="hidden" class="link-media-input" value="${escapeHtml(linkValue.media || '')}" data-index="${index}">
                                            <div class="link-media-preview mt-2 ${linkValue.media ? '' : 'hidden'}">
                                                <div class="relative inline-block">
                                                    <img src="${escapeHtml(linkValue.media || '')}" alt="Selected image" class="w-20 h-20 object-cover rounded border">
                                                    <button type="button" onclick="clearLinkEntryMedia('${fieldId}', ${index})" class="absolute top-1 right-1 bg-red-500 text-white rounded-full w-4 h-4 text-xs flex items-center justify-center hover:bg-red-600 shadow-lg">×</button>
                                                </div>
                                            </div>
                                        </div>

                                        <div>
                                            <label class="block text-sm font-medium text-gray-600 mb-1">Link URL</label>
                                            <input type="url" class="link-text-input w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                                                   value="${escapeHtml(linkValue.text || '')}"
                                                   placeholder="https://example.com"
                                                   data-index="${index}"
                                                   onchange="console.log('Link text changed for ${fieldId}'); updateLinkFieldFromEntry('${fieldId}')"
                                                   oninput="updateLinkFieldFromEntry('${fieldId}')">
                                        </div>
                                    </div>
                                </div>
                            `).join('')}
                        </div>

                        ${linkValues.length === 0 ? `
                            <div class="text-center py-8 text-gray-500">
                                <div class="text-sm">No links added yet</div>
                                <button type="button" onclick="addLinkEntry('${fieldId}')" class="mt-2 px-4 py-2 bg-blue-500 text-white text-sm rounded-lg hover:bg-blue-600 transition-colors">
                                    Add First Link
                                </button>
                            </div>
                        ` : ''}

                        <input type="hidden" id="${fieldId}" value="${escapeHtml(JSON.stringify(linkValues))}" ${field.is_required ? 'required' : ''}>
                    </div>`;
                    break;
                case 'rich_paragraph':
                    const maxChars = field.field_options && field.field_options.max_chars ? field.field_options.max_chars : null;
                    const charCounterId = `${fieldId}_char_count`;

                    inputHtml = `<div class="space-y-2">
                        <textarea id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors tinymce-editor" rows="10" ${field.is_required ? 'required' : ''} data-max-chars="${maxChars || ''}">${escapeHtml(existingValue)}</textarea>
                        ${maxChars ? `<div id="${charCounterId}" class="text-xs text-right font-medium text-gray-600"><span class="char-count">0</span> / ${maxChars} characters</div>` : ''}
                        <div class="text-xs text-gray-500">
                            Rich text editor with support for formatting, images, and videos${maxChars ? `. Maximum ${maxChars} characters.` : ''}
                        </div>
                    </div>`;
                    // Initialize TinyMCE for this field
                    setTimeout(() => {
                        const maxCharsNum = maxChars ? parseInt(maxChars, 10) : null;
                        console.log(`[RTE-INIT] Initializing RTE for ${fieldId} with maxChars: ${maxCharsNum}`);

                        // Initialize the editor first
                        initializeTinyMCE(fieldId);

                        // If character limit is set, initialize character counter
                        if (maxCharsNum && maxCharsNum > 0) {
                            console.log(`[RTE-INIT] Will set up character counter for ${fieldId} with limit ${maxCharsNum}`);
                            // Use longer delay to ensure editor is fully initialized
                            setTimeout(() => {
                                setupCharacterCounter(fieldId, maxCharsNum, charCounterId);
                            }, 1500);
                        }
                    }, 300);
                    break;
                default:
                    inputHtml = `<input type="text" id="${fieldId}" class="w-full px-4 py-3 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors" value="${escapeHtml(existingValue)}" ${field.is_required ? 'required' : ''}>`;
            }

            return `
                <div class="space-y-2">
                    <label for="${fieldId}" class="block text-sm font-semibold text-gray-900">
                        ${escapeHtml(field.name)}
                        ${field.is_required ? '<span class="text-red-500">*</span>' : ''}
                    </label>
                    ${inputHtml}
                </div>
            `;
        }).join('');

        // Setup validation for the dynamically generated record form
        setTimeout(() => {
            validateForm('recordForm', 'recordSubmitBtn');

            // Initialize array fields with their existing values
            fields.filter(field => field.data_type === 'array').forEach(field => {
                const fieldId = `record_${field.name.replace(/\s+/g, '_')}`;
                updateArrayFieldValue(fieldId);
            });

            // Initialize link fields to ensure hidden field values are synced
            fields.filter(field => field.data_type === 'link').forEach(field => {
                const fieldId = `record_${field.name.replace(/\s+/g, '_')}`;
                const hiddenField = document.getElementById(fieldId);
                const container = document.getElementById(fieldId + '_container');

                if (hiddenField && container) {
                    const initialValue = hiddenField.value;
                    console.log(`Initializing link field ${fieldId}`);
                    console.log(`  Initial hidden field value:`, initialValue);
                    console.log(`  Number of visible entries:`, container.querySelectorAll('.link-entry').length);

                    // The hidden field is already populated from the HTML generation,
                    // Sync it back from the DOM entries to ensure consistency
                    updateLinkFieldFromEntry(fieldId);

                    const finalValue = hiddenField.value;
                    console.log(`  Final hidden field value:`, finalValue);
                    console.log(`  Values match:`, initialValue === finalValue);
                }
            });
        }, 200);

        // Show/hide translations button based on whether editing existing record
        const translationsBtn = document.getElementById('manageTranslationsBtn');
        if (translationsBtn) {
            if (recordId) {
                translationsBtn.classList.remove('hidden');
            } else {
                translationsBtn.classList.add('hidden');
            }
        }

        // Add language selector event listener
        const languageSelect = document.getElementById('record_language');
        if (languageSelect && languages.length > 0) {
            languageSelect.addEventListener('change', async function() {
                const selectedLanguageCode = this.value;
                const selectedOption = this.options[this.selectedIndex];
                const languageNameSpan = document.getElementById('selectedLanguageName');

                if (languageNameSpan) {
                    languageNameSpan.textContent = selectedOption.text.replace(' (Default)', '');
                }

                // Switch to the selected language values
                await switchRecordLanguage(selectedLanguageCode, fields);
            });
        }

    } catch (error) {
        console.error('Error generating record form:', error);
        showToast('Failed to load form fields: ' + error.message, 'error');
    }
}

// Switch record form to different language
async function switchRecordLanguage(languageCode, fields) {
    if (!fields || !window.recordDefaultValues) return;

    const translations = window.recordTranslations[languageCode] || {};
    const isDefaultLanguage = languageCode === window.recordCurrentLanguage;

    showToast(`Switching to ${languageCode}...`, 'info');

    fields.forEach(field => {
        const fieldId = `record_${field.name.replace(/\s+/g, '_')}`;
        const fieldElement = document.getElementById(fieldId);

        if (!fieldElement) return;

        // Get the value: use translation if available, otherwise use default
        let value = translations[field.name] !== undefined
            ? translations[field.name]
            : window.recordDefaultValues[field.name] || '';

        // Update field based on type
        switch (field.data_type) {
            case 'text':
            case 'textarea':
            case 'number':
            case 'email':
            case 'url':
            case 'date':
                fieldElement.value = value || '';
                break;

            case 'checkbox':
                fieldElement.checked = value === 'true' || value === true || value === 1 || value === '1';
                break;

            case 'rich_paragraph':
                // Update TinyMCE editor if initialized
                if (typeof tinymce !== 'undefined') {
                    const editor = tinymce.get(fieldId);
                    if (editor) {
                        editor.setContent(value || '');
                    } else {
                        fieldElement.value = value || '';
                    }
                } else {
                    fieldElement.value = value || '';
                }
                break;

            case 'image':
            case 'file':
            case 'media':
                // Update hidden field
                fieldElement.value = value || '';
                // Trigger preview update if needed
                const previewId = fieldId + '_preview';
                const preview = document.getElementById(previewId);
                if (preview && value) {
                    preview.classList.remove('hidden');
                    // Update preview content based on type
                    if (field.data_type === 'image') {
                        const img = preview.querySelector('img');
                        if (img) img.src = value;
                    } else if (field.data_type === 'media') {
                        updateMediaPreview(fieldId, 'media', value.split(','));
                    }
                } else if (preview) {
                    preview.classList.add('hidden');
                }
                break;

            case 'array':
                // Update array field
                fieldElement.value = typeof value === 'string' ? value : JSON.stringify(value || []);
                // Regenerate array items UI if needed
                break;

            case 'json':
                // Update JSON field
                fieldElement.value = typeof value === 'string' ? value : JSON.stringify(value || {});
                // Regenerate JSON items UI if needed
                break;

            case 'link':
                // Update link field
                fieldElement.value = typeof value === 'string' ? value : JSON.stringify(value || []);
                // Regenerate link entries UI
                updateLinkFieldFromEntry(fieldId);
                break;

            default:
                fieldElement.value = value || '';
        }

        // Add visual indicator for translated vs default values
        if (translations[field.name] !== undefined && !isDefaultLanguage) {
            fieldElement.classList.add('border-green-400', 'bg-green-50');
            fieldElement.title = 'Translated content';
        } else if (!isDefaultLanguage) {
            fieldElement.classList.remove('border-green-400', 'bg-green-50');
            fieldElement.classList.add('border-yellow-300', 'bg-yellow-50');
            fieldElement.title = 'Using default language value (not yet translated)';
        } else {
            fieldElement.classList.remove('border-green-400', 'bg-green-50', 'border-yellow-300', 'bg-yellow-50');
            fieldElement.title = '';
        }
    });

    showToast(`Switched to ${languageCode} ${Object.keys(translations).length > 0 ? '(with translations)' : '(default values)'}`, 'success');
}

// Media Functions with enhanced search and navigation

async function loadMedia(typeFilter = '', page = 1, search = '') {
    try {
        currentMediaType = typeFilter;
        currentMediaPage = page;
        currentMediaSearch = search;

        const params = new URLSearchParams();
        if (typeFilter) {
            params.append('type', typeFilter);
        }
        if (search) {
            params.append('search', search);
        }
        params.append('page', page.toString());
        params.append('per_page', mediaPerPage.toString());

        const res = await apiRequest(`/files?${params.toString()}`);
        let files = [];
        let pagination = { page: 1, per_page: mediaPerPage, total: 0 };

        if (res && res.items && Array.isArray(res.items)) {
            files = res.items;
            pagination = res.pagination || pagination;
        } else if (Array.isArray(res)) {
            files = res;
        }

        renderMedia(files);
        updateMediaPagination(pagination);
    } catch (error) {
        console.error('Error loading media:', error);
        showToast('Failed to load media: ' + error.message, 'error');
        renderMedia([]);
        updateMediaPagination({ page: 1, per_page: mediaPerPage, total: 0 });
    }
}

function updateMediaPagination(pagination) {
    const paginationDiv = document.getElementById('mediaPagination');
    const showingStart = document.getElementById('mediaShowingStart');
    const showingEnd = document.getElementById('mediaShowingEnd');
    const mediaTotal = document.getElementById('mediaTotal');
    const prevBtn = document.getElementById('mediaPrevBtn');
    const nextBtn = document.getElementById('mediaNextBtn');
    const pageNumbers = document.getElementById('mediaPageNumbers');

    if (pagination.total > 0) {
        paginationDiv.classList.remove('hidden');

        const start = ((pagination.page - 1) * pagination.per_page) + 1;
        const end = Math.min(pagination.page * pagination.per_page, pagination.total);

        showingStart.textContent = start;
        showingEnd.textContent = end;
        mediaTotal.textContent = pagination.total;

        // Update navigation buttons
        prevBtn.disabled = pagination.page <= 1;
        nextBtn.disabled = pagination.page >= Math.ceil(pagination.total / pagination.per_page);

        // Generate page numbers
        const totalPages = Math.ceil(pagination.total / pagination.per_page);
        let pageNumbersHtml = '';

        for (let i = Math.max(1, pagination.page - 2); i <= Math.min(totalPages, pagination.page + 2); i++) {
            const isActive = i === pagination.page;
            pageNumbersHtml += `
                <button onclick="loadMediaPage(${i})"
                        class="px-3 py-1 border rounded-md text-sm ${isActive ? 'bg-blue-600 text-white border-blue-600' : 'border-gray-300 hover:bg-gray-50'}">
                    ${i}
                </button>
            `;
        }
        pageNumbers.innerHTML = pageNumbersHtml;
    } else {
        paginationDiv.classList.add('hidden');
    }
}

function loadMediaPage(action) {
    let newPage = currentMediaPage;

    if (action === 'prev') {
        newPage = Math.max(1, currentMediaPage - 1);
    } else if (action === 'next') {
        newPage = currentMediaPage + 1;
    } else if (typeof action === 'number') {
        newPage = action;
    }

    loadMedia(currentMediaType, newPage, currentMediaSearch);
}

// Media search functionality
let mediaSearchTimeout;
function setupMediaSearch() {
    const searchInput = document.getElementById('mediaSearchInput');
    if (searchInput) {
        // Add Enter key support for immediate search
        searchInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                performMediaSearch();
            }
        });

        // Keep the existing auto-search functionality but disable it when using new search types
        searchInput.addEventListener('input', function() {
            clearTimeout(mediaSearchTimeout);

            // Only auto-search for name searches to avoid confusion
            const searchType = document.getElementById('mediaSearchType');
            if (searchType && searchType.value === 'name') {
                mediaSearchTimeout = setTimeout(() => {
                    const searchTerm = this.value.trim();
                    if (searchTerm) {
                        performMediaSearch();
                    } else {
                        clearMediaSearch();
                    }
                }, 500); // Longer debounce for auto-search
            }
        });
    }
}

// Filter media in record selection modal
function filterRecordMediaList() {
    const searchInput = document.getElementById('recordMediaSearch');
    const mediaGrid = document.getElementById('recordMediaGrid');

    if (!searchInput || !mediaGrid) return;

    const searchTerm = searchInput.value.toLowerCase().trim();
    const mediaItems = mediaGrid.querySelectorAll('[data-file-name]');

    mediaItems.forEach(item => {
        const fileName = item.getAttribute('data-file-name').toLowerCase();
        const originalName = item.getAttribute('data-original-name')?.toLowerCase() || '';

        if (searchTerm === '' || fileName.includes(searchTerm) || originalName.includes(searchTerm)) {
            item.style.display = 'block';
        } else {
            item.style.display = 'none';
        }
    });
}

// Load media for default value selection with filtering
async function loadMediaForDefault(mediaType) {
    try {
        let res;
        if (mediaType === 'media') {
            // Load all files for media type
            res = await apiRequest(`/files`);
        } else {
            res = await apiRequest(`/files?type=${mediaType}`);
        }

        let files = [];

        if (res && res.items && Array.isArray(res.items)) {
            files = res.items;
        } else if (Array.isArray(res)) {
            files = res;
        }

        // Filter for media type if needed
        if (mediaType === 'media') {
            files = files.filter(file =>
                file.file_type.startsWith('image/') ||
                file.file_type.startsWith('video/')
                // Removed: file.file_type.startsWith('audio/')
            );
        }

        renderDefaultMediaSelector(files, mediaType);
    } catch (error) {
        console.error('Error loading default media:', error);
        showToast('Failed to load media: ' + error.message, 'error');
        renderDefaultMediaSelector([], mediaType);
    }
}

// Render media selector for field default values
function renderDefaultMediaSelector(files, mediaType) {
    // This function will be called from the media selector modal
    // Create or update a modal for selecting default media
    let modalHtml = `
        <div id="defaultMediaModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-4xl w-full p-6 max-h-[90vh] overflow-y-auto">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Select Default ${mediaType === 'image' ? 'Image' : mediaType === 'media' ? 'Media (Image/Video)' : 'File'}</h3>
                <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 mb-6" id="defaultMediaGrid">
                    ${files.map(file => {
                        const isImage = file.file_type.startsWith('image/');
                        const isVideo = file.file_type.startsWith('video/');
                        const isAudio = file.file_type.startsWith('audio/');
                        return `
                            <div class="border border-gray-200 rounded-lg p-3 cursor-pointer hover:bg-gray-50" onclick="selectDefaultMedia('${file.id}', '${file.url}', '${escapeJs(file.original_name)}', ${isImage}, ${isVideo})">
                                ${isImage ?
                                    `<img src="${file.url}" alt="${escapeHtml(file.original_name)}" class="w-full h-24 object-cover rounded mb-2">` :
                                isVideo ?
                                    `<video src="${file.url}" class="w-full h-24 object-cover rounded mb-2" muted preload="metadata" controls></video>` :
                                isAudio ?
                                    `<div class="w-full h-24 bg-purple-100 rounded mb-2 flex items-center justify-center">
                                        <svg class="w-8 h-8 text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3"></path>
                                        </svg>
                                    </div>` :
                                    `<div class="w-full h-24 bg-gray-100 rounded mb-2 flex items-center justify-center">
                                        <svg class="w-8 h-8 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                                        </svg>
                                    </div>`
                                }
                                <p class="text-xs text-gray-600 truncate">${escapeHtml(file.original_name)}</p>
                            </div>
                        `;
                    }).join('')}
                </div>
                <div class="flex justify-end space-x-3">
                    <button onclick="closeModal('defaultMediaModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing modal if present
    const existingModal = document.getElementById('defaultMediaModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Add modal to page
    document.body.insertAdjacentHTML('beforeend', modalHtml);
}

// Select media for default value
function selectDefaultMedia(fileId, fileUrl, fileName, isImage, isVideo) {
    const fieldDefaultValue = document.getElementById('fieldDefaultValue');

    if (window.defaultMediaSelectionType === 'image') {
        // Set value and show preview
        fieldDefaultValue.value = fileUrl;
        const preview = document.getElementById('defaultImagePreview');
        const container = document.getElementById('selectedDefaultImage');
        if (preview && container) {
            preview.src = fileUrl;
            preview.alt = fileName;
            container.classList.remove('hidden');
        }
    } else if (window.defaultMediaSelectionType === 'media') {
        // Set value and show preview for both images and videos
        fieldDefaultValue.value = fileUrl;
        const preview = document.getElementById('defaultMediaPreview');
        const container = document.getElementById('selectedDefaultMedia');
        if (preview && container) {
            if (isVideo) {
                preview.innerHTML = `<video src="${fileUrl}" class="w-20 h-20 object-cover rounded border" muted preload="metadata"></video>`;
            } else {
                preview.innerHTML = `<img src="${fileUrl}" alt="${fileName}" class="w-20 h-20 object-cover rounded border">`;
            }
            container.classList.remove('hidden');
        }
    } else {
        // Set value and show file name
        fieldDefaultValue.value = fileUrl;
        const fileName_element = document.getElementById('defaultFileName');
        const container = document.getElementById('selectedDefaultFile');
        if (fileName_element && container) {
            fileName_element.textContent = fileName;
            container.classList.remove('hidden');
        }
    }
    
    closeModal('defaultMediaModal');
}

function renderMedia(files) {
    const mediaGrid = document.getElementById("mediaGrid");
    const emptyState = document.getElementById("mediaEmpty");

    if (!mediaGrid) return;

    if (files.length === 0) {
        mediaGrid.innerHTML = "";
        if (emptyState) emptyState.classList.remove("hidden");
        return;
    }

    if (emptyState) emptyState.classList.add("hidden");

    mediaGrid.innerHTML = files.map(file => {
        const isImage = file.file_type.startsWith('image/');
        const isVideo = file.file_type.startsWith('video/');
        const isVideoUrl = file.file_type === 'video/url';
        const isImageUrl = file.file_type === 'image/url';
        const isAudio = file.file_type.startsWith('audio/');

        let filePreview;
        let mediaTypeIcon = '';

        if (isImage) {
            filePreview = `<img src="${file.url}" alt="${escapeHtml(file.original_name)}" class="w-full h-32 object-cover rounded-t-lg">`;
            // Different icons for local images vs image URLs
            if (isImageUrl) {
                mediaTypeIcon = '<span class="absolute top-2 right-2 bg-blue-500 text-white px-2 py-1 rounded text-xs">IMG URL</span>';
            } else {
                mediaTypeIcon = '<span class="absolute top-2 right-2 bg-green-500 text-white px-2 py-1 rounded text-xs">IMG</span>';
            }
        } else if (isVideoUrl) {
            // Show an actual iframe preview for video URLs
            const embedUrl = file.embed_url || file.url;
            filePreview = `
                <div class="relative w-full h-32 bg-black rounded-t-lg overflow-hidden flex items-center justify-center">
                    <iframe src="${embedUrl}"
                            class="w-full h-full"
                            frameborder="0"
                            allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share"
                            allowfullscreen>
                    </iframe>
                </div>
            `;
            mediaTypeIcon = '<span class="absolute top-2 right-2 bg-red-500 text-white px-2 py-1 rounded text-xs">VID URL</span>';
        } else if (isVideo) {
            filePreview = `
                <div class="relative w-full h-32 bg-black rounded-t-lg overflow-hidden">
                    <video class="w-full h-full object-cover" preload="metadata">
                        <source src="${file.url}#t=0.5" type="${file.file_type}">
                        Your browser does not support the video tag.
                    </video>
                    <div class="absolute inset-0 bg-black bg-opacity-30 flex items-center justify-center">
                        <svg class="w-12 h-12 text-white opacity-80" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M8 5v14l11-7z"/>
                        </svg>
                    </div>
                </div>
            `;
            mediaTypeIcon = '<span class="absolute top-2 right-2 bg-red-500 text-white px-2 py-1 rounded text-xs">VID</span>';
        } else if (isAudio) {
            filePreview = `
                <div class="w-full h-32 bg-purple-100 rounded-t-lg flex items-center justify-center">
                    <svg class="w-12 h-12 text-purple-500" fill="currentColor" viewBox="0 0 24 24">
                        <path d="M12 3v10.55c-.59-.34-1.27-.55-2-.55-2.21 0-4 1.79-4 4s1.79 4 4 4 4-1.79 4-4V7h4V3h-6z"/>
                    </svg>
                </div>
            `;
            mediaTypeIcon = '<span class="absolute top-2 right-2 bg-purple-500 text-white px-2 py-1 rounded text-xs">AUD</span>';
        } else {
            filePreview = `
                <div class="w-full h-32 bg-gray-100 rounded-t-lg flex items-center justify-center">
                    <svg class="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                    </svg>
                </div>
            `;
            mediaTypeIcon = '<span class="absolute top-2 right-2 bg-gray-500 text-white px-2 py-1 rounded text-xs">DOC</span>';
        }

        return `
        <div class="bg-white rounded-lg shadow-lg border border-gray-200 overflow-hidden hover:shadow-xl transition-all">
            <div class="relative">
                ${filePreview}
                <div class="absolute top-2 left-2 z-10">
                    <input type="checkbox" class="media-checkbox w-5 h-5 rounded border-gray-300 text-blue-600 focus:ring-blue-500 bg-white shadow-lg" data-id="${file.id}" onchange="updateMediaSelection()">
                </div>
            </div>
            <div class="p-4">
                <h3 class="text-sm font-medium text-gray-900 truncate" title="${escapeHtml(file.original_name)}">${escapeHtml(file.original_name)}</h3>
                <p class="text-xs text-gray-500 mt-1">${file.file_type} • ${formatFileSize(file.file_size)}</p>
                ${file.tags ? `
                    <div class="mt-2">
                        <div class="flex flex-wrap gap-1">
                            ${file.tags.split(',').map(tag => tag.trim()).filter(tag => tag).map(tag =>
                                `<button onclick="filterByClickedTag('${escapeHtml(tag)}')" class="inline-block bg-blue-100 hover:bg-blue-200 text-blue-800 text-xs px-2 py-1 rounded-full cursor-pointer transition-colors" title="Filter by this tag">${escapeHtml(tag)}</button>`
                            ).join('')}
                        </div>
                    </div>
                ` : ''}
                <div class="flex justify-between space-x-1 mt-3">
                    ${isVideoUrl ?
                        `<button onclick="openIframeModal('${(file.embed_url || file.url).replace(/'/g, "\'")}', '${escapeJs(file.original_name)}', '')" class="text-blue-600 hover:text-blue-700 text-xs font-medium transition-colors">Play</button>` :
                        isVideo ?
                        `<button onclick="openVideoModal('${file.url}', '${escapeJs(file.original_name)}')" class="text-blue-600 hover:text-blue-700 text-xs font-medium transition-colors">Play</button>` :
                        isAudio ?
                        `<button onclick="openAudioModal('${file.url}', '${escapeJs(file.original_name)}')" class="text-blue-600 hover:text-blue-700 text-xs font-medium transition-colors">Play</button>` :
                        `<a href="${file.url}" target="_blank" class="text-blue-600 hover:text-blue-700 text-xs font-medium transition-colors">View</a>`
                    }
                    <button onclick="selectMediaFromGallery('${file.id}', '${file.url}', '${escapeJs(file.original_name)}', ${isImage}, ${isVideo || isVideoUrl})" class="text-purple-600 hover:text-purple-700 text-xs font-medium transition-colors">Select</button>
                    <button onclick="copyToClipboard('${file.url}')" class="text-green-600 hover:text-green-700 text-xs transition-colors">Copy</button>
                    <button onclick="deleteMedia('${file.id}')" class="text-red-600 hover:text-red-800 text-xs font-medium transition-colors">Delete</button>
                </div>
            </div>
        </div>
        `;
    }).join("");
}

function formatFileSize(bytes) {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

async function deleteMedia(fileId) {
    if (!confirm("Are you sure you want to delete this file?")) {
        return;
    }

    try {
        await apiRequest(`/files/${fileId}`, {
            method: 'DELETE'
        });

        showToast("File deleted successfully");
        loadMedia();
    } catch (error) {
        console.error('Error deleting file:', error);
        showToast('Failed to delete file: ' + error.message, 'error');
    }
}

// Bulk Media Operations
function updateMediaSelection() {
    const checkboxes = document.querySelectorAll('.media-checkbox:checked');
    const count = checkboxes.length;
    const bulkActionsBar = document.getElementById('mediaBulkActionsBar');
    const selectedCount = document.getElementById('mediaSelectedCount');

    if (count > 0) {
        bulkActionsBar.classList.remove('hidden');
        selectedCount.textContent = count;
    } else {
        bulkActionsBar.classList.add('hidden');
    }
}

async function bulkDeleteMedia() {
    const checkboxes = document.querySelectorAll('.media-checkbox:checked');
    const ids = Array.from(checkboxes).map(cb => cb.dataset.id);

    if (ids.length === 0) {
        showToast('No files selected', 'warning');
        return;
    }

    if (!confirm(`Are you sure you want to delete ${ids.length} file${ids.length > 1 ? 's' : ''}?`)) {
        return;
    }

    try {
        showLoading();
        let successCount = 0;
        let failCount = 0;

        for (const id of ids) {
            try {
                await apiRequest(`/files/${id}`, { method: 'DELETE' });
                successCount++;
            } catch (error) {
                console.error(`Failed to delete file ${id}:`, error);
                failCount++;
            }
        }

        hideLoading();

        if (successCount > 0) {
            showToast(`Successfully deleted ${successCount} file${successCount > 1 ? 's' : ''}`, 'success');
        }
        if (failCount > 0) {
            showToast(`Failed to delete ${failCount} file${failCount > 1 ? 's' : ''}`, 'error');
        }

        clearMediaSelection();
        loadMedia();
    } catch (error) {
        hideLoading();
        console.error('Bulk delete error:', error);
        showToast('Bulk delete failed: ' + error.message, 'error');
    }
}

function clearMediaSelection() {
    const checkboxes = document.querySelectorAll('.media-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.checked = false;
    });
    updateMediaSelection();
}

// Video and Audio Modal Functions
function openVideoModal(videoUrl, title) {
    const modal = document.createElement('div');
    modal.className = 'fixed inset-0 bg-black bg-opacity-75 z-50 flex items-center justify-center p-4';
    modal.onclick = (e) => {
        if (e.target === modal) closeVideoModal(modal);
    };

    modal.innerHTML = `
        <div class="bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-[90vh] overflow-hidden">
            <div class="flex justify-between items-center p-4 border-b">
                <h3 class="text-lg font-medium text-gray-900">${escapeHtml(title)}</h3>
                <button onclick="closeVideoModal(this.closest('.fixed'))" class="text-gray-400 hover:text-gray-600">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            <div class="p-4">
                <video controls class="w-full max-h-[60vh]" preload="metadata">
                    <source src="${videoUrl}" type="video/mp4">
                    Your browser does not support the video tag.
                </video>
            </div>
        </div>
    `;

    document.body.appendChild(modal);
}

function openAudioModal(audioUrl, title) {
    const modal = document.createElement('div');
    modal.className = 'fixed inset-0 bg-black bg-opacity-75 z-50 flex items-center justify-center p-4';
    modal.onclick = (e) => {
        if (e.target === modal) closeAudioModal(modal);
    };

    modal.innerHTML = `
        <div class="bg-white rounded-lg shadow-xl max-w-md w-full">
            <div class="flex justify-between items-center p-4 border-b">
                <h3 class="text-lg font-medium text-gray-900">${escapeHtml(title)}</h3>
                <button onclick="closeAudioModal(this.closest('.fixed'))" class="text-gray-400 hover:text-gray-600">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
            <div class="p-6">
                <div class="flex items-center justify-center mb-4">
                    <svg class="w-16 h-16 text-purple-500" fill="currentColor" viewBox="0 0 24 24">
                        <path d="M12 3v10.55c-.59-.34-1.27-.55-2-.55-2.21 0-4 1.79-4 4s1.79 4 4 4 4-1.79 4-4V7h4V3h-6z"/>
                    </svg>
                </div>
                <audio controls class="w-full">
                    <source src="${audioUrl}" type="audio/mpeg">
                    Your browser does not support the audio element.
                </audio>
            </div>
        </div>
    `;

    document.body.appendChild(modal);
}

function closeVideoModal(modal) {
    const video = modal.querySelector('video');
    if (video) video.pause();
    modal.remove();
}

function closeAudioModal(modal) {
    const audio = modal.querySelector('audio');
    if (audio) audio.pause();
    modal.remove();
}

// Iframe modal functions for video URLs
function openIframeModal(videoUrl, title, embedHtml) {
    const modal = document.getElementById('iframeModal');
    const modalTitle = document.getElementById('iframeModalTitle');
    const iframeContainer = document.getElementById('iframeContainer');

    if (!modal || !modalTitle || !iframeContainer) return;

    modalTitle.textContent = title;

    // If embedHtml is provided, use it directly
    if (embedHtml && embedHtml.trim()) {
        iframeContainer.innerHTML = embedHtml;
    } else {
        // Fallback: create iframe from URL
        const iframe = document.createElement('iframe');
        iframe.src = videoUrl;
        iframe.width = '800';
        iframe.height = '450';
        iframe.frameBorder = '0';
        iframe.allowFullscreen = true;
        iframe.allow = 'accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture';
        iframe.className = 'w-full max-w-4xl';
        iframe.style.aspectRatio = '16/9';

        iframeContainer.innerHTML = '';
        iframeContainer.appendChild(iframe);
    }

    modal.classList.remove('hidden');

    // Close modal when clicking outside
    modal.onclick = (e) => {
        if (e.target === modal) closeIframeModal();
    };
}

function closeIframeModal() {
    const modal = document.getElementById('iframeModal');
    const iframeContainer = document.getElementById('iframeContainer');

    if (modal) {
        modal.classList.add('hidden');
    }

    // Clear iframe to stop video playback
    if (iframeContainer) {
        iframeContainer.innerHTML = '';
    }
}

// Close iframe modal on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        const iframeModal = document.getElementById('iframeModal');
        if (iframeModal && !iframeModal.classList.contains('hidden')) {
            closeIframeModal();
        }
    }
});

// Testing function - you can call this from browser console
function testUploadModal() {
    console.log('Testing upload modal...');

    // Open the modal
    openModal('uploadModal');

    // Wait a bit then test tab switching
    setTimeout(() => {
        console.log('Testing tab switching...');
        switchUploadTab('url');

        setTimeout(() => {
            switchUploadTab('file');
        }, 1000);
    }, 500);
}

// Copy to clipboard function
function copyToClipboard(text) {
    if (navigator.clipboard && navigator.clipboard.writeText) {
        navigator.clipboard.writeText(text).then(() => {
            showToast('URL copied to clipboard!', 'success');
        }).catch(() => {
            fallbackCopyToClipboard(text);
        });
    } else {
        fallbackCopyToClipboard(text);
    }
}

function fallbackCopyToClipboard(text) {
    const textArea = document.createElement('textarea');
    textArea.value = text;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    textArea.style.top = '-999999px';
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();

    try {
        document.execCommand('copy');
        showToast('URL copied to clipboard!', 'success');
    } catch (err) {
        showToast('Failed to copy URL', 'error');
    }

    document.body.removeChild(textArea);
}

// Media Filter Function
function filterMedia(type) {
    // Update active filter button
    document.querySelectorAll('.media-filter-btn').forEach(btn => {
        btn.classList.remove('active');
    });

    event.target.classList.add('active');

    // Reset search input when filtering
    const searchInput = document.getElementById('mediaSearchInput');
    if (searchInput) {
        searchInput.value = '';
    }

    // Reset tag filter when changing type filter
    const tagFilter = document.getElementById('mediaTagFilter');
    if (tagFilter) {
        tagFilter.value = '';
    }
    document.getElementById('clearTagFilter')?.classList.add('hidden');

    // Load media with filter, reset to page 1
    loadMedia(type, 1, '');
}

// Tag Filter Functions
function filterMediaByTag() {
    const tagFilter = document.getElementById('mediaTagFilter');
    const clearButton = document.getElementById('clearTagFilter');

    if (tagFilter && tagFilter.value) {
        clearButton?.classList.remove('hidden');

        // Get current type filter
        const activeTypeButton = document.querySelector('.media-filter-btn.active');
        const typeFilter = activeTypeButton ? activeTypeButton.textContent.toLowerCase() : '';
        const type = typeFilter === 'all files' ? '' : typeFilter;

        // Reset search input
        const searchInput = document.getElementById('mediaSearchInput');
        if (searchInput) {
            searchInput.value = '';
        }

        loadMediaWithTag(type, tagFilter.value, 1);
    } else {
        clearButton?.classList.add('hidden');
        // Reload without tag filter
        const activeTypeButton = document.querySelector('.media-filter-btn.active');
        const typeFilter = activeTypeButton ? activeTypeButton.textContent.toLowerCase() : '';
        const type = typeFilter === 'all files' ? '' : typeFilter;
        loadMedia(type, 1, '');
    }
}

function clearTagFilter() {
    const tagFilter = document.getElementById('mediaTagFilter');
    const clearButton = document.getElementById('clearTagFilter');

    if (tagFilter) {
        tagFilter.value = '';
    }
    clearButton?.classList.add('hidden');

    // Reload without tag filter
    const activeTypeButton = document.querySelector('.media-filter-btn.active');
    const typeFilter = activeTypeButton ? activeTypeButton.textContent.toLowerCase() : '';
    const type = typeFilter === 'all files' ? '' : typeFilter;
    loadMedia(type, 1, '');
}

async function loadMediaWithTag(type, tag, page = 1, search = '') {
    try {
        showLoading();

        let queryParams = new URLSearchParams({
            page: page,
            per_page: mediaPerPage || 8
        });

        if (search.trim()) {
            queryParams.append('search', search.trim());
        }

        if (type && type !== '') {
            queryParams.append('type', type);
        }

        if (tag) {
            queryParams.append('tag', tag);
        }

        const response = await apiRequest(`/files?${queryParams.toString()}`);

        let files = [];
        let pagination = null;

        if (response && response.items && Array.isArray(response.items)) {
            files = response.items;
            pagination = response.pagination;
        } else if (Array.isArray(response)) {
            files = response;
        }

        renderMedia(files, pagination);

    } catch (error) {
        console.error('Error loading media with tag:', error);
        showToast('Failed to load media: ' + error.message, 'error');
        document.getElementById('mediaGrid').innerHTML = '<div class="col-span-full text-center text-gray-500">Error loading media files</div>';
    } finally {
        hideLoading();
    }
}

async function populateTagFilter() {
    try {
        const tags = await apiRequest('/tags');
        const tagFilter = document.getElementById('mediaTagFilter');

        if (tagFilter && Array.isArray(tags)) {
            // Clear existing options except the first one
            tagFilter.innerHTML = '<option value="">Filter by tag...</option>';

            tags.forEach(tag => {
                const option = document.createElement('option');
                option.value = tag;
                option.textContent = tag;
                tagFilter.appendChild(option);
            });
        }
    } catch (error) {
        console.error('Error loading tags for filter:', error);
    }
}

function filterByClickedTag(tag) {
    const tagFilter = document.getElementById('mediaTagFilter');
    const clearButton = document.getElementById('clearTagFilter');

    if (tagFilter) {
        tagFilter.value = tag;
        clearButton?.classList.remove('hidden');

        // Get current type filter
        const activeTypeButton = document.querySelector('.media-filter-btn.active');
        const typeFilter = activeTypeButton ? activeTypeButton.textContent.toLowerCase() : '';
        const type = typeFilter === 'all files' ? '' : typeFilter;

        // Reset search input
        const searchInput = document.getElementById('mediaSearchInput');
        if (searchInput) {
            searchInput.value = '';
        }

        loadMediaWithTag(type, tag, 1);
    }
}

// Media Search Functions
function updateSearchPlaceholder() {
    const searchType = document.getElementById('mediaSearchType');
    const searchInput = document.getElementById('mediaSearchInput');

    if (searchType && searchInput) {
        if (searchType.value === 'tag') {
            searchInput.placeholder = 'Search by tag (e.g., nature, photo)...';
        } else {
            searchInput.placeholder = 'Search by filename...';
        }
    }
}

function performMediaSearch() {
    const searchInput = document.getElementById('mediaSearchInput');
    const searchType = document.getElementById('mediaSearchType');

    if (!searchInput || !searchType) return;

    const searchValue = searchInput.value.trim();
    const searchTypeValue = searchType.value;

    if (!searchValue) {
        showToast('Please enter a search term', 'warning');
        return;
    }

    // Get current type filter
    const activeTypeButton = document.querySelector('.media-filter-btn.active');
    const typeFilter = activeTypeButton ? activeTypeButton.textContent.toLowerCase() : '';
    const type = typeFilter === 'all files' ? '' : typeFilter;

    // Clear tag filter when performing search
    const tagFilter = document.getElementById('mediaTagFilter');
    if (tagFilter) {
        tagFilter.value = '';
    }
    document.getElementById('clearTagFilter')?.classList.add('hidden');

    // Perform search with type
    loadMediaWithSearch(type, searchValue, searchTypeValue, 1);
}

function clearMediaSearch() {
    const searchInput = document.getElementById('mediaSearchInput');
    const searchType = document.getElementById('mediaSearchType');
    const tagFilter = document.getElementById('mediaTagFilter');

    if (searchInput) searchInput.value = '';
    if (searchType) {
        searchType.value = 'name';
        updateSearchPlaceholder();
    }
    if (tagFilter) tagFilter.value = '';

    document.getElementById('clearTagFilter')?.classList.add('hidden');

    // Get current type filter and reload
    const activeTypeButton = document.querySelector('.media-filter-btn.active');
    const typeFilter = activeTypeButton ? activeTypeButton.textContent.toLowerCase() : '';
    const type = typeFilter === 'all files' ? '' : typeFilter;

    loadMedia(type, 1, '');
}

async function loadMediaWithSearch(type, search, searchType, page = 1) {
    try {
        showLoading();

        let queryParams = new URLSearchParams({
            page: page,
            per_page: mediaPerPage || 8
        });

        if (search.trim()) {
            queryParams.append('search', search.trim());
            queryParams.append('search_type', searchType);
        }

        if (type && type !== '') {
            queryParams.append('type', type);
        }

        const response = await apiRequest(`/files?${queryParams.toString()}`);

        let files = [];
        let pagination = null;

        if (response && response.items && Array.isArray(response.items)) {
            files = response.items;
            pagination = response.pagination;
        } else if (Array.isArray(response)) {
            files = response;
        }

        renderMedia(files, pagination);

    } catch (error) {
        console.error('Error loading media with search:', error);
        showToast('Failed to load media: ' + error.message, 'error');
        document.getElementById('mediaGrid').innerHTML = '<div class="col-span-full text-center text-gray-500">Error loading media files</div>';
    } finally {
        hideLoading();
    }
}

// Test upload function
async function testUpload() {
    try {
        // Create a small test file
        const testContent = 'This is a test file for debugging upload functionality.';
        const testFile = new File([testContent], 'test.txt', { type: 'text/plain' });

        const formData = new FormData();
        formData.append('file', testFile);

        console.log('Test Upload - Starting test...');

        const config = {
            method: 'POST',
            headers: {},
            body: formData
        };

        if (authToken) {
            config.headers['Authorization'] = `Bearer ${authToken}`;
        }

        const responseText = await response.text();

        console.log('Test Upload - Response status:', response.status);
        console.log('Test Upload - Raw response:', responseText);

        let data;
        try {
            data = JSON.parse(responseText);
            console.log('Test Upload - Parsed response:', data);

            if (data.success) {
                showToast('Test upload successful! Check console for details.', 'success');
            } else {
                showToast('Test upload failed: ' + (data.error || 'Unknown error'), 'error');
            }
        } catch (parseError) {
            console.error('Test Upload - JSON parse error:', parseError);
            showToast('Test upload returned invalid JSON. Check console.', 'error');
        }

    } catch (error) {
        console.error('Test Upload Error:', error);
        showToast('Test upload failed: ' + error.message, 'error');
    }
}

// Modal Functions
function openModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.remove('hidden');

        // Setup form validation when modal opens
        setTimeout(() => {
            setupFormValidation();
        }, 100);

        // Setup upload tags autocomplete when upload modal opens
        if (modalId === 'uploadModal') {
            console.log('Opening upload modal, initializing tabs...');
            setTimeout(() => {
                // Initialize upload modal to file upload tab
                switchUploadTab('file');

                loadAvailableTags().then(() => {
                    setupUploadTagsAutocomplete();
                    setupUrlUploadTagsAutocomplete();
                    console.log('Upload modal initialization complete');
                }).catch(error => {
                    console.error('Error loading available tags:', error);
                });
            }, 100);
        }
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.classList.add('hidden');
    }
    
    // Reset forms and editing states
    if (modalId === 'categoryModal') {
        resetCategoryModal();
    } else if (modalId === 'fieldModal') {
        resetFieldModal();
    } else if (modalId === 'recordModal') {
        resetRecordModal();
        cleanupTinyMCE(); // Clean up TinyMCE instances when closing record modal
    } else if (modalId === 'translationModal') {
        // Clean up translation Rich Paragraph editors
        if (translatableFields) {
            translatableFields.forEach(field => {
                if (field.data_type === 'rich_paragraph') {
                    const fieldId = `trans_${field.name.replace(/\s+/g, '_')}`;
                    const editor = RTE.get(fieldId);
                    if (editor) {
                        console.log(`[Translation-RTE] Cleanup: Removing editor for ${fieldId}`);
                        RTE.remove(fieldId);
                    }
                }
            });
        }
    } else if (modalId === 'uploadModal') {
        resetUploadModal();
    } else if (modalId === 'defaultMediaModal') {
        // Remove dynamically created modal
        const modal = document.getElementById('defaultMediaModal');
        if (modal) modal.remove();
    } else if (modalId === 'recordMediaModal') {
        // Clean up RTE mode if active
        if (window.rteMediaSelectionMode) {
            window.rteImageCallback = null;
            window.rteMediaSelectionMode = false;
        }
        // Remove dynamically created modal
        const modal = document.getElementById('recordMediaModal');
        if (modal) modal.remove();
    }
}

function resetCategoryModal() {
    editingCategory = null;
    const categoryModalTitle = document.getElementById("categoryModalTitle");
    const categoryName = document.getElementById("categoryName");
    const firstFieldName = document.getElementById("firstFieldName");
    const firstFieldType = document.getElementById("firstFieldType");
    const firstFieldRequired = document.getElementById("firstFieldRequired");
    const firstFieldSection = document.getElementById("firstFieldSection");
    
    if (categoryModalTitle) categoryModalTitle.textContent = "Add Category";
    if (categoryName) categoryName.value = '';
    if (firstFieldName) firstFieldName.value = '';
    if (firstFieldType) firstFieldType.value = '';
    if (firstFieldRequired) firstFieldRequired.checked = false;
    if (firstFieldSection) firstFieldSection.style.display = 'block';
}

function resetFieldModal() {
    editingField = null;
    const fieldModalTitle = document.getElementById("fieldModalTitle");
    const fieldName = document.getElementById("fieldName");
    const fieldType = document.getElementById("fieldType");
    const fieldRequired = document.getElementById("fieldRequired");
    const defaultValueContainer = document.getElementById("defaultValueContainer");
    const defaultValueHelp = document.getElementById("defaultValueHelp");
    const foreignKeyContainer = document.getElementById("foreignKeyContainer");
    const arrayContainer = document.getElementById("arrayContainer");

    if (fieldModalTitle) fieldModalTitle.textContent = "Add Field";
    if (fieldName) fieldName.value = '';
    if (fieldType) fieldType.value = '';
    if (fieldRequired) fieldRequired.checked = false;

    // Hide special containers
    if (foreignKeyContainer) foreignKeyContainer.classList.add('hidden');
    if (arrayContainer) arrayContainer.classList.add('hidden');

    // Reset foreign key fields
    const foreignKeyTable = document.getElementById("foreignKeyTable");
    const foreignKeyColumn = document.getElementById("foreignKeyColumn");
    if (foreignKeyTable) foreignKeyTable.value = '';
    if (foreignKeyColumn) foreignKeyColumn.innerHTML = '<option value="">Select column...</option>';

    // Reset array fields
    const arrayItemType = document.getElementById("arrayItemType");
    const arrayAllowMultiple = document.getElementById("arrayAllowMultiple");
    if (arrayItemType) arrayItemType.value = 'text';
    if (arrayAllowMultiple) arrayAllowMultiple.checked = true;

    // Reset JSON fields
    const jsonValueType = document.getElementById("jsonValueType");
    const jsonAllowCustomKeys = document.getElementById("jsonAllowCustomKeys");
    const jsonPredefinedKeys = document.getElementById("jsonPredefinedKeys");
    if (jsonValueType) jsonValueType.value = 'text';
    if (jsonAllowCustomKeys) jsonAllowCustomKeys.checked = true;
    if (jsonPredefinedKeys) jsonPredefinedKeys.value = '';

    // Reset default value container to basic input
    if (defaultValueContainer) {
        defaultValueContainer.innerHTML = '<input type="text" id="fieldDefaultValue" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Enter default value">';
    }
    if (defaultValueHelp) {
        defaultValueHelp.textContent = '';
    }
}

function resetRecordModal() {
    editingRecord = null;
    const recordModalTitle = document.getElementById("recordModalTitle");
    const recordFields = document.getElementById("recordFields");
    
    if (recordModalTitle) recordModalTitle.textContent = "Add Record";
    if (recordFields) recordFields.innerHTML = '';
    currentCategoryFields = [];
}

function resetUploadModal() {
    // Reset file upload form
    const uploadFile = document.getElementById("uploadFile");
    const uploadTags = document.getElementById("uploadTags");
    if (uploadFile) uploadFile.value = '';
    if (uploadTags) uploadTags.value = '';
    hideUploadTagsSuggestions();

    // Reset URL upload form
    const videoUrl = document.getElementById("videoUrl");
    const videoTitle = document.getElementById("videoTitle");
    const urlUploadTags = document.getElementById("urlUploadTags");
    if (videoUrl) videoUrl.value = '';
    if (videoTitle) videoTitle.value = '';
    if (urlUploadTags) urlUploadTags.value = '';
    hideUrlUploadTagsSuggestions();

    // Reset to file upload tab
    switchUploadTab('file');
}

// Form Submission Handlers
async function handleAuth(event) {
    event.preventDefault();

    const email = document.getElementById('authEmail').value;
    const password = document.getElementById('authPassword').value;

    setButtonLoading('authSubmitBtn', 'authSpinner', true);

    try {
        const result = await login(email, password);

        if (result.success) {
            showToast('Login successful');
            showMainApp();
        }
    } catch (error) {
        const authError = document.getElementById('authError');
        if (authError) {
            authError.textContent = error.message;
            authError.classList.remove('hidden');
        }
        showToast(error.message, 'error');
    } finally {
        setButtonLoading('authSubmitBtn', 'authSpinner', false);
    }
}

async function handleCategorySubmit(event) {
    event.preventDefault();
    
    const categoryName = document.getElementById('categoryName').value;
    const firstFieldName = document.getElementById('firstFieldName')?.value;
    const firstFieldType = document.getElementById('firstFieldType')?.value;
    const firstFieldRequired = document.getElementById('firstFieldRequired')?.checked;

    setButtonLoading('categoryForm', 'categorySpinner', true);

    try {
        const categoryData = { name: categoryName };
        
        // Add first field if creating new category
        if (!editingCategory && firstFieldName && firstFieldType) {
            categoryData.first_field = {
                name: firstFieldName,
                data_type: firstFieldType,
                is_required: firstFieldRequired || false
            };
        }

        if (editingCategory) {
            await apiRequest(`/categories/${editingCategory}`, {
                method: 'PUT',
                body: JSON.stringify(categoryData)
            });
            showToast('Category updated successfully');
        } else {
            await apiRequest('/categories', {
                method: 'POST',
                body: JSON.stringify(categoryData)
            });
            showToast('Category created successfully');
        }

        closeModal('categoryModal');
        loadCategories();
    } catch (error) {
        console.error('Error saving category:', error);
        showToast('Error saving category: ' + error.message, 'error');
    } finally {
        setButtonLoading('categoryForm', 'categorySpinner', false);
    }
}

// Field type change handler and validation
function handleFieldTypeChange() {
    const fieldType = document.getElementById('fieldType').value;
    const defaultValueContainer = document.getElementById('defaultValueContainer');
    const defaultValueHelp = document.getElementById('defaultValueHelp');
    const fieldDefaultValue = document.getElementById('fieldDefaultValue');
    const foreignKeyContainer = document.getElementById('foreignKeyContainer');
    const arrayContainer = document.getElementById('arrayContainer');
    const jsonContainer = document.getElementById('jsonContainer');
    const richParagraphContainer = document.getElementById('richParagraphContainer');

    // Hide special containers by default
    if (foreignKeyContainer) foreignKeyContainer.classList.add('hidden');
    if (arrayContainer) arrayContainer.classList.add('hidden');
    if (jsonContainer) jsonContainer.classList.add('hidden');
    if (richParagraphContainer) richParagraphContainer.classList.add('hidden');

    if (!fieldType) {
        fieldDefaultValue.type = 'text';
        fieldDefaultValue.placeholder = 'Enter default value';
        defaultValueHelp.textContent = '';
        return;
    }

    // Handle special field types
    if (fieldType === 'foreign_key') {
        if (foreignKeyContainer) {
            foreignKeyContainer.classList.remove('hidden');
            loadAvailableTables();
        }
        fieldDefaultValue.style.display = 'none';
        defaultValueHelp.textContent = 'Foreign key fields reference other tables.';
        return;
    }

    if (fieldType === 'array') {
        if (arrayContainer) arrayContainer.classList.remove('hidden');
        fieldDefaultValue.style.display = 'none';
        defaultValueHelp.textContent = 'Array fields store multiple values.';
        return;
    }

    if (fieldType === 'json') {
        if (jsonContainer) jsonContainer.classList.remove('hidden');
        fieldDefaultValue.style.display = 'none';
        defaultValueHelp.textContent = 'JSON fields store flexible key-value pairs.';
        return;
    }

    if (fieldType === 'link') {
        fieldDefaultValue.style.display = 'none';
        defaultValueHelp.textContent = 'Link fields store JSON with media and text components.';
        return;
    }

    // Show default value input for regular fields
    fieldDefaultValue.style.display = 'block';

    // Update input type and placeholder based on field type
    switch (fieldType) {
        case 'text':
            fieldDefaultValue.type = 'text';
            fieldDefaultValue.placeholder = 'Enter default text';
            defaultValueHelp.textContent = 'Text value (letters, numbers, symbols)';
            break;
        case 'textarea':
            // Replace input with textarea for better UX
            defaultValueContainer.innerHTML = `<textarea id="fieldDefaultValue" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Enter default text" rows="3"></textarea>`;
            defaultValueHelp.textContent = 'Long text content';
            break;
        case 'number':
            fieldDefaultValue.type = 'number';
            fieldDefaultValue.placeholder = '0';
            defaultValueHelp.textContent = 'Numeric value (integers or decimals)';
            break;
        case 'email':
            fieldDefaultValue.type = 'email';
            fieldDefaultValue.placeholder = 'user@example.com';
            defaultValueHelp.textContent = 'Valid email address';
            break;
        case 'url':
            fieldDefaultValue.type = 'url';
            fieldDefaultValue.placeholder = 'https://example.com';
            defaultValueHelp.textContent = 'Valid URL starting with http:// or https://';
            break;
        case 'date':
            fieldDefaultValue.type = 'date';
            fieldDefaultValue.placeholder = '';
            defaultValueHelp.textContent = 'Date in YYYY-MM-DD format';
            break;
        case 'checkbox':
            // Replace with select for boolean values
            defaultValueContainer.innerHTML = `<select id="fieldDefaultValue" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                <option value="false">False (unchecked)</option>
                <option value="true">True (checked)</option>
            </select>`;
            defaultValueHelp.textContent = 'Default checked state';
            break;
        case 'boolean':
            // Replace with select for true/false values
            defaultValueContainer.innerHTML = `<select id="fieldDefaultValue" required class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                <option value="false">False</option>
                <option value="true">True</option>
            </select>`;
            defaultValueHelp.textContent = 'Default boolean value (true or false)';
            break;
        case 'image':
            // Replace with media selector for images
            defaultValueContainer.innerHTML = `<div class="space-y-2">
                <button type="button" onclick="openMediaSelectorForDefault('image')" class="w-full px-3 py-2 bg-blue-500 text-white rounded-md hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-blue-500">
                    Select Default Image
                </button>
                <input type="hidden" id="fieldDefaultValue" required>
                <div id="selectedDefaultImage" class="hidden">
                    <img id="defaultImagePreview" src="" alt="Default image" class="w-20 h-20 object-cover rounded border">
                    <button type="button" onclick="clearDefaultImage()" class="text-red-600 text-sm hover:text-red-800">Remove</button>
                </div>
            </div>`;
            defaultValueHelp.textContent = 'Select a default image from media library';
            break;
        case 'file':
            // Replace with media selector for files
            defaultValueContainer.innerHTML = `<div class="space-y-2">
                <button type="button" onclick="openMediaSelectorForDefault('file')" class="w-full px-3 py-2 bg-blue-500 text-white rounded-md hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-blue-500">
                    Select Default File
                </button>
                <input type="hidden" id="fieldDefaultValue" required>
                <div id="selectedDefaultFile" class="hidden">
                    <div id="defaultFilePreview" class="flex items-center space-x-2 p-2 bg-gray-100 rounded">
                        <span id="defaultFileName"></span>
                    </div>
                    <button type="button" onclick="clearDefaultFile()" class="text-red-600 text-sm hover:text-red-800">Remove</button>
                </div>
            </div>`;
            defaultValueHelp.textContent = 'Select a default file from media library';
            break;
        case 'media':
            // Replace with media selector for media (images & videos)
            defaultValueContainer.innerHTML = `<div class="space-y-2">
                <button type="button" onclick="openMediaSelectorForDefault('media')" class="w-full px-3 py-2 bg-purple-500 text-white rounded-md hover:bg-purple-600 focus:outline-none focus:ring-2 focus:ring-purple-500">
                    Select Default Media
                </button>
                <input type="hidden" id="fieldDefaultValue" required>
                <div id="selectedDefaultMedia" class="hidden">
                    <div id="defaultMediaPreview" class="grid grid-cols-3 gap-2"></div>
                    <button type="button" onclick="clearDefaultMedia()" class="text-red-600 text-sm hover:text-red-800 mt-2">Clear All</button>
                </div>
            </div>`;
            defaultValueHelp.textContent = 'Select default media (images/videos) from media library';
            break;
        case 'rich_paragraph':
            defaultValueContainer.innerHTML = `<div class="space-y-2">
                <textarea id="fieldDefaultValue" rows="6" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" placeholder="Enter default rich content (HTML allowed)"></textarea>
                <div class="text-xs text-gray-500">You can include HTML tags for formatting</div>
            </div>`;
            defaultValueHelp.textContent = 'Rich text content with HTML support';

            // Show rich paragraph options container
            const richParagraphContainer = document.getElementById('richParagraphContainer');
            if (richParagraphContainer) {
                richParagraphContainer.classList.remove('hidden');
            }
            break;
        default:
            fieldDefaultValue.type = 'text';
            fieldDefaultValue.placeholder = 'Enter default value';
            defaultValueHelp.textContent = '';
    }
}

// Validation function for field default values
function validateFieldDefaultValue(fieldType, defaultValue) {
    // If no default value provided, it's always valid (field can be optional)
    if (!defaultValue || defaultValue.trim() === '') {
        return { valid: true };
    }

    switch (fieldType) {
        case 'number':
            if (isNaN(defaultValue)) {
                return { valid: false, message: 'Default value must be a number' };
            }
            break;
        case 'email':
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(defaultValue)) {
                return { valid: false, message: 'Default value must be a valid email address' };
            }
            break;
        case 'url':
            try {
                new URL(defaultValue);
            } catch {
                return { valid: false, message: 'Default value must be a valid URL' };
            }
            break;
        case 'date':
            if (!/^\d{4}-\d{2}-\d{2}$/.test(defaultValue)) {
                return { valid: false, message: 'Default value must be in YYYY-MM-DD format' };
            }
            break;
        case 'checkbox':
            if (defaultValue !== 'true' && defaultValue !== 'false') {
                return { valid: false, message: 'Default value must be true or false' };
            }
            break;
    }

    return { valid: true };
}

// Foreign key and array field helper functions
async function loadAvailableTables() {
    try {
        const response = await apiRequest('/categories');
        const categories = response.items || [];

        const foreignKeyTable = document.getElementById('foreignKeyTable');
        if (!foreignKeyTable) return;

        // Clear existing options
        foreignKeyTable.innerHTML = '<option value="">Select table...</option>';

        categories.forEach(category => {
            const option = document.createElement('option');
            option.value = category.name;
            option.textContent = category.name;
            foreignKeyTable.appendChild(option);
        });

        // Add event listener for table selection
        foreignKeyTable.addEventListener('change', loadTableColumns);
    } catch (error) {
        console.error('Error loading available tables:', error);
        showToast('Failed to load available tables', 'error');
    }
}

async function loadTableColumns() {
    const foreignKeyTable = document.getElementById('foreignKeyTable');
    const foreignKeyColumn = document.getElementById('foreignKeyColumn');

    if (!foreignKeyTable || !foreignKeyColumn) return;

    const selectedTable = foreignKeyTable.value;
    if (!selectedTable) {
        foreignKeyColumn.innerHTML = '<option value="">Select column...</option>';
        return;
    }

    try {
        // Find the category ID for the selected table
        const categoriesResponse = await apiRequest('/categories');
        const categories = categoriesResponse.items || [];
        const targetCategory = categories.find(cat => cat.name === selectedTable);

        if (!targetCategory) {
            foreignKeyColumn.innerHTML = '<option value="">Table not found...</option>';
            return;
        }

        // Load fields for the selected category
        const fieldsResponse = await apiRequest(`/fields?category_id=${targetCategory.id}`);
        const fields = fieldsResponse.items || [];

        // Clear and populate column dropdown
        foreignKeyColumn.innerHTML = '<option value="">Select column...</option>';

        // Add ID as a default option
        const idOption = document.createElement('option');
        idOption.value = 'id';
        idOption.textContent = 'id (Primary Key)';
        foreignKeyColumn.appendChild(idOption);

        // Add all other fields
        fields.forEach(field => {
            const option = document.createElement('option');
            option.value = field.name;
            option.textContent = `${field.name} (${field.data_type})`;
            foreignKeyColumn.appendChild(option);
        });

    } catch (error) {
        console.error('Error loading table columns:', error);
        foreignKeyColumn.innerHTML = '<option value="">Error loading columns...</option>';
    }
}

// Array field helper functions
function addArrayItem(fieldId, itemType) {
    const container = document.getElementById(fieldId + '_container');
    if (!container) return;

    const inputType = itemType === 'number' ? 'number' : itemType === 'email' ? 'email' : itemType === 'url' ? 'url' : 'text';

    const newItem = document.createElement('div');
    newItem.className = 'flex items-center space-x-2 array-item';
    newItem.innerHTML = `
        <input type="${inputType}"
               class="flex-1 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors array-item-input"
               value=""
               placeholder="Enter ${itemType} value"
               onchange="updateArrayFieldValue('${fieldId}')"
               onkeyup="updateArrayFieldValue('${fieldId}')">
        <button type="button" onclick="removeArrayItem(this)" class="px-2 py-2 text-red-600 hover:text-red-800 hover:bg-red-50 rounded-lg transition-colors">
            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
            </svg>
        </button>
    `;

    container.appendChild(newItem);
    updateArrayFieldValue(fieldId);

    // Focus the new input
    const newInput = newItem.querySelector('input');
    if (newInput) newInput.focus();
}

function removeArrayItem(button) {
    const item = button.closest('.array-item');
    if (!item) return;

    const container = item.parentElement;
    const fieldId = container.id.replace('_container', '');

    item.remove();
    updateArrayFieldValue(fieldId);
}

// JSON field helper functions
function addJsonItem(fieldId, valueType, allowCustomKeys, predefinedKeysStr) {
    const container = document.getElementById(fieldId + '_container');
    if (!container) return;

    const predefinedKeys = predefinedKeysStr ? predefinedKeysStr.split(',').filter(k => k.trim()) : [];

    const newItem = document.createElement('div');
    newItem.className = 'flex items-center space-x-2 json-item';

    const keyInput = allowCustomKeys ?
        `<input type="text" class="w-32 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors json-key-input" placeholder="Key">` :
        `<select class="w-32 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors json-key-select">
            <option value="">Select key...</option>
            ${predefinedKeys.map(key => `<option value="${escapeHtml(key)}">${escapeHtml(key)}</option>`).join('')}
        </select>`;

    const valueInput = valueType === 'textarea' ?
        `<textarea class="flex-1 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors json-value-input" rows="2" placeholder="Enter value"></textarea>` :
        valueType === 'checkbox' ?
        `<div class="flex-1 flex items-center"><input type="checkbox" class="w-4 h-4 text-blue-500 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 focus:ring-2 json-value-input"><span class="ml-2 text-sm text-gray-700">True/False</span></div>` :
        `<input type="${valueType === 'number' ? 'number' : valueType === 'email' ? 'email' : valueType === 'url' ? 'url' : valueType === 'date' ? 'date' : 'text'}" class="flex-1 px-3 py-2 border border-gray-200 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors json-value-input" placeholder="Enter value">`;

    newItem.innerHTML = `
        ${keyInput}
        ${valueInput}
        <button type="button" onclick="removeJsonItem(this)" class="px-2 py-2 text-red-600 hover:text-red-800 hover:bg-red-50 rounded-lg transition-colors">
            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
            </svg>
        </button>
    `;

    container.appendChild(newItem);

    // Add event listeners to update hidden field when values change
    const keyElement = newItem.querySelector('.json-key-input, .json-key-select');
    const valueElement = newItem.querySelector('.json-value-input');

    if (keyElement) {
        keyElement.addEventListener('input', () => updateJsonFieldValue(fieldId));
        keyElement.addEventListener('change', () => updateJsonFieldValue(fieldId));
    }
    if (valueElement) {
        valueElement.addEventListener('input', () => updateJsonFieldValue(fieldId));
        valueElement.addEventListener('change', () => updateJsonFieldValue(fieldId));
    }

    updateJsonFieldValue(fieldId);
}

function removeJsonItem(button) {
    const item = button.closest('.json-item');
    if (!item) return;

    const container = item.parentElement;
    const fieldId = container.id.replace('_container', '');

    item.remove();
    updateJsonFieldValue(fieldId);
}

function updateJsonFieldValue(fieldId) {
    const container = document.getElementById(fieldId + '_container');
    const hiddenField = document.getElementById(fieldId);
    if (!container || !hiddenField) return;

    const jsonItems = container.querySelectorAll('.json-item');
    const jsonObject = {};

    jsonItems.forEach(item => {
        const keyElement = item.querySelector('.json-key-input, .json-key-select');
        const valueElement = item.querySelector('.json-value-input');

        if (!keyElement || !valueElement) return;

        const key = keyElement.value.trim();
        if (!key) return; // Skip items without keys

        let value;
        if (valueElement.type === 'checkbox') {
            value = valueElement.checked;
        } else if (valueElement.type === 'number') {
            value = valueElement.value ? parseFloat(valueElement.value) : '';
        } else {
            value = valueElement.value;
        }

        jsonObject[key] = value;
    });

    hiddenField.value = JSON.stringify(jsonObject);
}

function updateArrayFieldValue(fieldId) {
    const container = document.getElementById(fieldId + '_container');
    const hiddenInput = document.getElementById(fieldId);

    if (!container || !hiddenInput) return;

    const inputs = container.querySelectorAll('.array-item-input');
    const values = Array.from(inputs).map(input => input.value).filter(value => value.trim() !== '');

    hiddenInput.value = JSON.stringify(values);
    console.log(`Updated array field ${fieldId}:`, values, 'JSON:', hiddenInput.value);
}

function addLinkEntry(fieldId) {
    const container = document.getElementById(fieldId + '_container');
    const hiddenField = document.getElementById(fieldId);

    if (!container || !hiddenField) return;

    // Get current values
    let linkValues = [];
    try {
        linkValues = JSON.parse(hiddenField.value || '[]');
    } catch (e) {
        linkValues = [];
    }

    // Add new empty link entry
    const newIndex = linkValues.length;
    linkValues.push({ media: '', text: '' });

    // Update hidden field
    hiddenField.value = JSON.stringify(linkValues);

    // Add new entry to DOM
    const newEntry = document.createElement('div');
    newEntry.className = 'link-entry bg-white p-4 rounded-lg border border-gray-200';
    newEntry.setAttribute('data-index', newIndex);
    newEntry.innerHTML = `
        <div class="flex items-center justify-between mb-3">
            <div class="text-sm font-medium text-gray-600">Link #${newIndex + 1}</div>
            <button type="button" onclick="removeLinkEntry('${fieldId}', ${newIndex})" class="px-2 py-1 text-red-600 hover:text-red-800 hover:bg-red-50 rounded transition-colors">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>

        <div class="space-y-3">
            <div>
                <label class="block text-sm font-medium text-gray-600 mb-1">Image</label>
                <button type="button" onclick="openMediaSelectorForLinkEntry('${fieldId}', ${newIndex})" class="w-full px-3 py-2 bg-blue-500 text-white text-sm rounded-lg hover:bg-blue-600 transition-colors">
                    Select Image
                </button>
                <input type="hidden" class="link-media-input" value="" data-index="${newIndex}">
                <div class="link-media-preview mt-2 hidden">
                    <div class="relative inline-block">
                        <img src="" alt="Selected image" class="w-20 h-20 object-cover rounded border">
                        <button type="button" onclick="clearLinkEntryMedia('${fieldId}', ${newIndex})" class="absolute top-1 right-1 bg-red-500 text-white rounded-full w-4 h-4 text-xs flex items-center justify-center hover:bg-red-600 shadow-lg">×</button>
                    </div>
                </div>
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-600 mb-1">Link URL</label>
                <input type="url" class="link-text-input w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                       value=""
                       placeholder="https://example.com"
                       data-index="${newIndex}"
                       onchange="console.log('New link text changed for ${fieldId}'); updateLinkFieldFromEntry('${fieldId}')"
                       oninput="updateLinkFieldFromEntry('${fieldId}')">
            </div>
        </div>
    `;

    container.appendChild(newEntry);

    // Update the main field value immediately
    console.log('Calling updateLinkFieldFromEntry after adding entry');
    updateLinkFieldFromEntry(fieldId);
}

function removeLinkEntry(fieldId, index) {
    const container = document.getElementById(fieldId + '_container');
    const hiddenField = document.getElementById(fieldId);

    if (!container || !hiddenField) return;

    // Get current values
    let linkValues = [];
    try {
        linkValues = JSON.parse(hiddenField.value || '[]');
    } catch (e) {
        linkValues = [];
    }

    // Remove entry from array
    linkValues.splice(index, 1);

    // Update hidden field
    hiddenField.value = JSON.stringify(linkValues);

    // Remove DOM element
    const entry = container.querySelector(`[data-index="${index}"]`);
    if (entry) {
        entry.remove();
    }

    // Reindex remaining entries
    const entries = container.querySelectorAll('.link-entry');
    entries.forEach((entry, newIndex) => {
        entry.setAttribute('data-index', newIndex);
        entry.querySelector('.text-gray-600').textContent = `Link #${newIndex + 1}`;

        // Update onclick handlers
        const removeBtn = entry.querySelector('[onclick*="removeLinkEntry"]');
        if (removeBtn) {
            removeBtn.setAttribute('onclick', `removeLinkEntry('${fieldId}', ${newIndex})`);
        }

        const selectBtn = entry.querySelector('[onclick*="openMediaSelectorForLinkEntry"]');
        if (selectBtn) {
            selectBtn.setAttribute('onclick', `openMediaSelectorForLinkEntry('${fieldId}', ${newIndex})`);
        }

        const clearBtn = entry.querySelector('[onclick*="clearLinkEntryMedia"]');
        if (clearBtn) {
            clearBtn.setAttribute('onclick', `clearLinkEntryMedia('${fieldId}', ${newIndex})`);
        }

        // Update data-index attributes
        const mediaInput = entry.querySelector('.link-media-input');
        const textInput = entry.querySelector('.link-text-input');
        if (mediaInput) mediaInput.setAttribute('data-index', newIndex);
        if (textInput) textInput.setAttribute('data-index', newIndex);
    });

    // Update the main field value after reindexing
    console.log('Calling updateLinkFieldFromEntry after removing entry');
    updateLinkFieldFromEntry(fieldId);
}

function updateLinkFieldFromEntry(fieldId) {
    const container = document.getElementById(fieldId + '_container');
    const hiddenField = document.getElementById(fieldId);

    console.log(`updateLinkFieldFromEntry called for ${fieldId}`);
    console.log('Container found:', !!container);
    console.log('Hidden field found:', !!hiddenField);

    if (!container || !hiddenField) {
        console.error(`Missing elements for link field ${fieldId}:`, {
            container: !!container,
            hiddenField: !!hiddenField
        });
        return;
    }

    const entries = container.querySelectorAll('.link-entry');
    const linkValues = [];

    console.log(`Found ${entries.length} link entries`);

    entries.forEach((entry, index) => {
        const mediaInput = entry.querySelector('.link-media-input');
        const textInput = entry.querySelector('.link-text-input');

        console.log(`Entry ${index}:`, {
            mediaInput: !!mediaInput,
            textInput: !!textInput,
            mediaValue: mediaInput ? mediaInput.value : 'N/A',
            textValue: textInput ? textInput.value : 'N/A'
        });

        if (mediaInput && textInput) {
            const linkEntry = {
                media: mediaInput.value.trim(),
                text: textInput.value.trim()
            };
            linkValues.push(linkEntry);
            console.log(`Added link entry ${index}:`, linkEntry);
        }
    });

    const jsonValue = JSON.stringify(linkValues);
    hiddenField.value = jsonValue;

    console.log(`Updated hidden field ${fieldId} with:`, jsonValue);
    console.log('Parsed back:', JSON.parse(jsonValue));
}

function openMediaSelectorForLinkEntry(fieldId, entryIndex) {
    // Store the current context for the media selector callback
    window.currentLinkFieldId = fieldId;
    window.currentLinkEntryIndex = entryIndex;
    window.isLinkEntry = true;

    // Use the existing record media selector for images only
    window.recordMediaSelectionFieldId = `${fieldId}_link_${entryIndex}`;
    window.recordMediaSelectionType = 'image';

    // Load and show media filtered by type
    loadMediaForRecord('image');
    openModal('recordMediaModal');
}

function clearLinkEntryMedia(fieldId, entryIndex) {
    const container = document.getElementById(fieldId + '_container');
    const entry = container.querySelector(`[data-index="${entryIndex}"]`);

    if (!entry) return;

    const mediaInput = entry.querySelector('.link-media-input');
    const mediaPreview = entry.querySelector('.link-media-preview');

    if (mediaInput) {
        mediaInput.value = '';
    }

    if (mediaPreview) {
        mediaPreview.classList.add('hidden');
    }

    updateLinkFieldFromEntry(fieldId);
}

// Add event listener for array item changes
document.addEventListener('input', function(event) {
    if (event.target.classList.contains('array-item-input')) {
        const container = event.target.closest('[id$="_container"]');
        if (container) {
            const fieldId = container.id.replace('_container', '');
            updateArrayFieldValue(fieldId);
        }
    }
});

async function loadForeignKeyOptions(fieldId, field, selectedValue = '') {
    console.log('loadForeignKeyOptions called with:', {
        fieldId,
        foreign_key_table: field.foreign_key_table,
        foreign_key_column: field.foreign_key_column,
        field
    });

    if (!field.foreign_key_table || !field.foreign_key_column) {
        console.error('Foreign key field missing table or column configuration');
        return;
    }

    const select = document.getElementById(fieldId);
    if (!select) return;

    try {
        // Use global categories variable
        const targetCategory = categories.find(cat => cat.name === field.foreign_key_table);
        console.log('Target category found:', targetCategory);

        if (!targetCategory) {
            select.innerHTML = '<option value="">Target table not found</option>';
            return;
        }

        // Get records from the target table with language filter if applicable
        let apiUrl = `/records?category_id=${targetCategory.id}`;
        if (currentGlobalLanguage) {
            apiUrl += `&lang=${currentGlobalLanguage}`;
        }
        const recordsResponse = await apiRequest(apiUrl);
        const records = recordsResponse.items || [];

        // Clear and populate options
        select.innerHTML = '<option value="">Select an option...</option>';

        records.forEach(record => {
            const option = document.createElement('option');

            if (field.foreign_key_column === 'id') {
                option.value = record.id;
                // For ID references, show a meaningful label if possible
                const firstTextValue = Object.values(record.values || {}).find(val => val && typeof val === 'string');
                option.textContent = firstTextValue ? `${record.id} - ${firstTextValue}` : `ID: ${record.id}`;
            } else {
                const columnValue = record.values[field.foreign_key_column] || '';
                option.value = columnValue;
                option.textContent = columnValue || `(empty)`;
            }

            if (option.value == selectedValue) {
                option.selected = true;
            }

            select.appendChild(option);
        });

    } catch (error) {
        console.error('Error loading foreign key options:', error);
        select.innerHTML = '<option value="">Error loading options</option>';
    }
}

// Custom Rich Text Editor Implementation
class RichTextEditor {
    constructor(selector, options = {}) {
        this.selector = selector;
        this.element = typeof selector === 'string' ? document.querySelector(selector) : selector;

        if (!this.element) {
            throw new Error(`Element not found: ${selector}`);
        }

        this.options = {
            height: options.height || 400,
            toolbar: options.toolbar || 'bold italic underline | p h1 h2 h3 | bullist numlist | link image | undo redo',
            onChange: options.onChange || null,
            onImageUpload: options.onImageUpload || null,
            onFilePicker: options.onFilePicker || null,
            maxChars: options.maxChars || null,
            ...options
        };

        this.commandHistory = [];
        this.historyIndex = -1;
        this.maxHistory = 100;
        this.lastValidContent = '';

        this.init();
    }

    init() {
        // Create editor wrapper
        this.wrapper = document.createElement('div');
        this.wrapper.className = 'rte-wrapper';
        this.element.parentNode.insertBefore(this.wrapper, this.element);

        // Create toolbar
        this.toolbar = document.createElement('div');
        this.toolbar.className = 'rte-toolbar';
        this.toolbar.setAttribute('role', 'toolbar');
        this.wrapper.appendChild(this.toolbar);

        // Create editor surface
        this.editor = document.createElement('div');
        this.editor.className = 'rte-editor';
        this.editor.contentEditable = true;
        this.editor.setAttribute('role', 'textbox');
        this.editor.setAttribute('aria-multiline', 'true');
        this.editor.style.minHeight = `${this.options.height}px`;
        this.wrapper.appendChild(this.editor);

        // Hide original element
        this.element.style.display = 'none';

        // Initialize content
        this.setContent(this.element.value || this.element.innerHTML || '');

        // Build toolbar
        this.buildToolbar();

        // Attach event listeners
        this.attachEvents();

        // Save initial state
        this.saveState();
    }

    buildToolbar() {
        const commands = this.options.toolbar.split('|').map(g => g.trim().split(/\s+/));

        commands.forEach((group, idx) => {
            if (idx > 0) {
                const separator = document.createElement('span');
                separator.className = 'rte-separator';
                this.toolbar.appendChild(separator);
            }

            const groupEl = document.createElement('div');
            groupEl.className = 'rte-toolbar-group';

            group.forEach(cmd => {
                const button = this.createToolbarButton(cmd);
                if (button) groupEl.appendChild(button);
            });

            this.toolbar.appendChild(groupEl);
        });
    }

    createToolbarButton(command) {
        const buttonConfig = {
            bold: { icon: 'B', title: 'Bold', style: 'font-weight: bold;' },
            italic: { icon: 'I', title: 'Italic', style: 'font-style: italic;' },
            underline: { icon: 'U', title: 'Underline', style: 'text-decoration: underline;' },
            strikethrough: { icon: 'S', title: 'Strikethrough', style: 'text-decoration: line-through;' },
            p: { icon: 'P', title: 'Paragraph' },
            h1: { icon: 'H1', title: 'Heading 1' },
            h2: { icon: 'H2', title: 'Heading 2' },
            h3: { icon: 'H3', title: 'Heading 3' },
            bullist: { icon: '\u2022 List', title: 'Bullet List' },
            numlist: { icon: '1. List', title: 'Numbered List' },
            link: { icon: '\uD83D\uDD17', title: 'Insert Link' },
            image: { icon: '\uD83D\uDDBC\uFE0F', title: 'Insert Image' },
            undo: { icon: '\u21B6', title: 'Undo' },
            redo: { icon: '\u21B7', title: 'Redo' },
            alignleft: { icon: '\u2261', title: 'Align Left' },
            aligncenter: { icon: '\u2263', title: 'Align Center' },
            alignright: { icon: '\u2261', title: 'Align Right' },
            removeformat: { icon: '\u2717', title: 'Clear Formatting' }
        };

        const config = buttonConfig[command];
        if (!config) return null;

        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'rte-toolbar-btn';
        button.innerHTML = `<span ${config.style ? `style="${config.style}"` : ''}>${config.icon}</span>`;
        button.title = config.title;
        button.setAttribute('aria-label', config.title);
        button.setAttribute('data-command', command);

        button.addEventListener('click', (e) => {
            e.preventDefault();
            this.exec(command);
        });

        return button;
    }

    attachEvents() {
        // Character limit enforcement - MUST BE FIRST
        if (this.options.maxChars && this.options.maxChars > 0) {
            console.log(`[RTE] Setting up character limit: ${this.options.maxChars} chars`);

            const getTextLength = (html) => {
                if (!html) return 0;
                const temp = document.createElement('div');
                temp.innerHTML = html;
                return (temp.textContent || temp.innerText || '').trim().length;
            };

            // Block typing at limit
            this.editor.addEventListener('keydown', (e) => {
                const currentLength = getTextLength(this.getContent());
                const hasSelection = window.getSelection().toString().length > 0;
                const allowedKeys = ['Backspace', 'Delete', 'ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown', 'Home', 'End', 'Tab'];

                if (e.ctrlKey || e.metaKey || allowedKeys.includes(e.key) || hasSelection) {
                    return; // Allow
                }

                if (currentLength >= this.options.maxChars) {
                    console.warn(`[RTE] 🚫 Character limit reached (${currentLength}/${this.options.maxChars})`);
                    e.preventDefault();
                    e.stopPropagation();
                    e.stopImmediatePropagation();
                    showToast(`Character limit: ${this.options.maxChars} maximum`, 'warning');
                    return false;
                }
            }, true);

            // Block beforeinput
            this.editor.addEventListener('beforeinput', (e) => {
                const currentLength = getTextLength(this.getContent());
                const hasSelection = window.getSelection().toString().length > 0;

                if (e.inputType && (e.inputType.includes('delete') || e.inputType.includes('format') || e.inputType.includes('history'))) {
                    return; // Allow
                }

                if (hasSelection) return; // Allow (replaces selection)

                if (currentLength >= this.options.maxChars) {
                    e.preventDefault();
                    e.stopPropagation();
                    e.stopImmediatePropagation();
                    return false;
                }
            }, true);

            // Store initial content
            this.lastValidContent = this.getContent();
        }

        // Track changes
        this.editor.addEventListener('input', () => {
            // Check character limit after input
            if (this.options.maxChars && this.options.maxChars > 0) {
                const getTextLength = (html) => {
                    if (!html) return 0;
                    const temp = document.createElement('div');
                    temp.innerHTML = html;
                    return (temp.textContent || temp.innerText || '').trim().length;
                };

                const currentLength = getTextLength(this.getContent());
                if (currentLength > this.options.maxChars) {
                    console.warn(`[RTE] Reverting - exceeded limit (${currentLength}/${this.options.maxChars})`);
                    this.setContent(this.lastValidContent);
                    showToast(`Character limit exceeded. Changes reverted.`, 'warning');
                } else {
                    this.lastValidContent = this.getContent();
                }
            }

            this.syncContent();
            if (this.options.onChange) {
                this.options.onChange(this.getContent());
            }
        });

        // Paste handling - preserve original formatting
        this.editor.addEventListener('paste', (e) => {
            e.preventDefault();
            const html = e.clipboardData.getData('text/html');
            const text = e.clipboardData.getData('text/plain');

            if (html) {
                // Preserve original HTML with minimal sanitization
                const sanitized = this.sanitizeHTML(html, true);
                document.execCommand('insertHTML', false, sanitized);
            } else if (text) {
                // Convert plain text line breaks to HTML
                const formattedText = text.replace(/\n\n/g, '</p><p>').replace(/\n/g, '<br>');
                const wrapped = `<p>${formattedText}</p>`;
                document.execCommand('insertHTML', false, wrapped);
            }

            this.saveState();
        });

        // Keyboard shortcuts
        this.editor.addEventListener('keydown', (e) => {
            const ctrl = e.ctrlKey || e.metaKey;

            if (ctrl && e.key === 'b') {
                e.preventDefault();
                this.exec('bold');
            } else if (ctrl && e.key === 'i') {
                e.preventDefault();
                this.exec('italic');
            } else if (ctrl && e.key === 'u') {
                e.preventDefault();
                this.exec('underline');
            } else if (ctrl && e.key === 'z' && !e.shiftKey) {
                e.preventDefault();
                this.undo();
            } else if (ctrl && (e.key === 'y' || (e.key === 'z' && e.shiftKey))) {
                e.preventDefault();
                this.redo();
            } else if (e.key === 'Enter' && !e.shiftKey) {
                setTimeout(() => this.saveState(), 10);
            }
        });

        this.editor.addEventListener('blur', () => {
            this.saveState();
        });

        // Image drag and drop
        this.editor.addEventListener('drop', (e) => {
            e.preventDefault();

            const files = Array.from(e.dataTransfer.files);
            const images = files.filter(f => f.type.startsWith('image/'));

            if (images.length > 0 && this.options.onImageUpload) {
                images.forEach(file => {
                    this.uploadImage(file);
                });
            }
        });

        this.editor.addEventListener('dragover', (e) => {
            e.preventDefault();
        });
    }

    exec(command, value = null) {
        this.editor.focus();

        const handlers = {
            bold: () => document.execCommand('bold'),
            italic: () => document.execCommand('italic'),
            underline: () => document.execCommand('underline'),
            strikethrough: () => document.execCommand('strikethrough'),
            p: () => this.formatBlock('p'),
            h1: () => this.formatBlock('h1'),
            h2: () => this.formatBlock('h2'),
            h3: () => this.formatBlock('h3'),
            bullist: () => document.execCommand('insertUnorderedList'),
            numlist: () => document.execCommand('insertOrderedList'),
            link: () => this.insertLink(),
            image: () => this.insertImage(),
            undo: () => this.undo(),
            redo: () => this.redo(),
            alignleft: () => document.execCommand('justifyLeft'),
            aligncenter: () => document.execCommand('justifyCenter'),
            alignright: () => document.execCommand('justifyRight'),
            removeformat: () => document.execCommand('removeFormat')
        };

        if (handlers[command]) {
            handlers[command](value);
            this.saveState();
            this.syncContent();
        }
    }

    formatBlock(tag) {
        const selection = window.getSelection();
        if (!selection.rangeCount) return;

        const range = selection.getRangeAt(0);
        const element = range.commonAncestorContainer.nodeType === 1
            ? range.commonAncestorContainer
            : range.commonAncestorContainer.parentElement;

        let currentBlock = element;
        while (currentBlock && currentBlock !== this.editor) {
            if (currentBlock.tagName && currentBlock.tagName.toLowerCase() === tag) {
                document.execCommand('formatBlock', false, 'p');
                return;
            }
            currentBlock = currentBlock.parentElement;
        }

        document.execCommand('formatBlock', false, tag);
    }

    insertLink() {
        const url = prompt('Enter URL:');
        if (url) {
            const sanitizedUrl = this.sanitizeURL(url);
            if (sanitizedUrl) {
                document.execCommand('createLink', false, sanitizedUrl);
            }
        }
    }

    insertImage() {
        // Store reference to this editor instance
        const editorInstance = this;

        // Use the media gallery modal if available
        if (typeof openMediaSelectorForRTE === 'function') {
            openMediaSelectorForRTE((url) => {
                console.log('RTE: Image URL received:', url);
                if (url) {
                    // Focus the editor first to ensure proper cursor position
                    editorInstance.editor.focus();

                    // Don't sanitize if it's an image data URL (base64 encoded images)
                    let finalUrl = url;
                    if (!url.startsWith('data:image/')) {
                        finalUrl = editorInstance.sanitizeURL(url);
                        console.log('RTE: Sanitized URL:', finalUrl);
                    } else {
                        console.log('RTE: Using data URL directly');
                    }

                    if (finalUrl) {
                        // Create img element
                        const img = document.createElement('img');
                        img.src = finalUrl;
                        img.alt = '';
                        img.style.maxWidth = '100%';
                        img.style.height = 'auto';
                        img.style.display = 'block';
                        img.style.margin = '0.5rem 0';

                        console.log('RTE: Inserting image element:', img);

                        // Try execCommand first
                        const inserted = document.execCommand('insertHTML', false, img.outerHTML);
                        console.log('RTE: execCommand result:', inserted);

                        // If execCommand failed, try direct insertion
                        if (!inserted) {
                            console.log('RTE: execCommand failed, trying direct insertion');
                            const selection = window.getSelection();
                            if (selection.rangeCount > 0) {
                                const range = selection.getRangeAt(0);
                                range.deleteContents();
                                range.insertNode(img);
                                range.collapse(false);
                            } else {
                                // Fallback: append to editor
                                editorInstance.editor.appendChild(img);
                            }
                        }

                        editorInstance.saveState();
                        editorInstance.syncContent();

                        console.log('RTE: Editor content after insert:', editorInstance.editor.innerHTML.substring(0, 200));
                    } else {
                        console.error('RTE: URL was sanitized to empty string');
                    }
                }
            });
            return;
        }

        // Fallback to original behavior
        if (this.options.onFilePicker) {
            this.options.onFilePicker((url) => {
                if (url) {
                    const img = `<img src="${this.sanitizeURL(url)}" alt="" style="max-width: 100%;">`;
                    document.execCommand('insertHTML', false, img);
                    this.saveState();
                    this.syncContent();
                }
            });
        } else {
            const url = prompt('Enter image URL:');
            if (url) {
                const sanitizedUrl = this.sanitizeURL(url);
                if (sanitizedUrl) {
                    const img = `<img src="${sanitizedUrl}" alt="" style="max-width: 100%;">`;
                    document.execCommand('insertHTML', false, img);
                }
            }
        }
    }

    async uploadImage(file) {
        if (!this.options.onImageUpload) return;

        try {
            const url = await this.options.onImageUpload(file);
            if (url) {
                const img = `<img src="${this.sanitizeURL(url)}" alt="${file.name}" style="max-width: 100%;">`;
                document.execCommand('insertHTML', false, img);
                this.saveState();
                this.syncContent();
            }
        } catch (error) {
            console.error('Image upload failed:', error);
        }
    }

    sanitizeHTML(html, preserveFormatting = false) {
        // Extended allowed tags to preserve more formatting
        const allowedTags = ['b', 'i', 'u', 'strong', 'em', 'p', 'br', 'ul', 'ol', 'li',
                            'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'a', 'img', 'blockquote', 'code', 'pre',
                            'div', 'span', 'table', 'thead', 'tbody', 'tr', 'td', 'th',
                            'sub', 'sup', 'mark', 'del', 'ins', 's', 'small', 'cite', 'abbr'];

        // Extended allowed attributes to preserve styling
        const allowedAttrs = {
            a: ['href', 'title', 'target', 'rel'],
            img: ['src', 'alt', 'title', 'width', 'height', 'style'],
            // Allow style attribute on common elements when preserving formatting
            p: preserveFormatting ? ['style', 'class'] : ['class'],
            div: preserveFormatting ? ['style', 'class'] : ['class'],
            span: preserveFormatting ? ['style', 'class'] : ['class'],
            h1: preserveFormatting ? ['style', 'class'] : ['class'],
            h2: preserveFormatting ? ['style', 'class'] : ['class'],
            h3: preserveFormatting ? ['style', 'class'] : ['class'],
            h4: preserveFormatting ? ['style', 'class'] : ['class'],
            h5: preserveFormatting ? ['style', 'class'] : ['class'],
            h6: preserveFormatting ? ['style', 'class'] : ['class'],
            ul: preserveFormatting ? ['style', 'class'] : ['class'],
            ol: preserveFormatting ? ['style', 'class'] : ['class'],
            li: preserveFormatting ? ['style', 'class'] : ['class'],
            table: preserveFormatting ? ['style', 'class', 'border', 'cellpadding', 'cellspacing'] : ['class'],
            td: preserveFormatting ? ['style', 'class', 'colspan', 'rowspan'] : ['class'],
            th: preserveFormatting ? ['style', 'class', 'colspan', 'rowspan'] : ['class']
        };

        const temp = document.createElement('div');
        temp.innerHTML = html;

        const clean = (node) => {
            const nodeName = node.nodeName.toLowerCase();

            // Always remove script and style tags
            if (nodeName === 'script' || nodeName === 'style') {
                node.remove();
                return;
            }

            if (node.attributes) {
                Array.from(node.attributes).forEach(attr => {
                    const attrName = attr.name.toLowerCase();

                    // Always remove event handlers
                    if (attrName.startsWith('on')) {
                        node.removeAttribute(attr.name);
                        return;
                    }

                    const allowed = allowedAttrs[nodeName];
                    if (allowed) {
                        if (!allowed.includes(attrName)) {
                            node.removeAttribute(attr.name);
                        }
                    } else if (!['class', 'id'].includes(attrName)) {
                        // Allow class and id on all elements by default
                        if (preserveFormatting && attrName === 'style') {
                            // Keep style attribute when preserving formatting
                        } else {
                            node.removeAttribute(attr.name);
                        }
                    }

                    // Sanitize dangerous URLs
                    if (attrName === 'href' || attrName === 'src') {
                        const value = attr.value;
                        if (value && (value.startsWith('javascript:') || value.startsWith('data:text/html'))) {
                            node.removeAttribute(attr.name);
                        }
                    }

                    // Sanitize style attribute for dangerous values
                    if (attrName === 'style' && preserveFormatting) {
                        const styleValue = attr.value;
                        // Remove potentially dangerous style properties
                        if (styleValue && (
                            styleValue.includes('expression') ||
                            styleValue.includes('javascript:') ||
                            styleValue.includes('import') ||
                            styleValue.includes('behavior')
                        )) {
                            node.removeAttribute(attr.name);
                        }
                    }
                });
            }

            Array.from(node.childNodes).forEach(child => clean(child));

            // For preserving formatting, convert unknown tags to spans instead of removing
            if (node.nodeType === 1 && !allowedTags.includes(nodeName)) {
                if (preserveFormatting) {
                    // Convert unknown tags to span to preserve content and styling
                    const span = document.createElement('span');
                    if (node.hasAttribute('style')) {
                        span.setAttribute('style', node.getAttribute('style'));
                    }
                    if (node.hasAttribute('class')) {
                        span.setAttribute('class', node.getAttribute('class'));
                    }
                    while (node.firstChild) {
                        span.appendChild(node.firstChild);
                    }
                    node.parentNode.replaceChild(span, node);
                } else {
                    // Remove tag but keep content
                    const parent = node.parentNode;
                    while (node.firstChild) {
                        parent.insertBefore(node.firstChild, node);
                    }
                    node.remove();
                }
            }
        };

        clean(temp);
        return temp.innerHTML;
    }

    sanitizeURL(url) {
        if (!url) return '';
        if (url.match(/^(javascript|data):/i)) {
            return '';
        }
        return url;
    }

    saveState() {
        const content = this.editor.innerHTML;

        if (this.commandHistory[this.historyIndex] === content) {
            return;
        }

        this.commandHistory = this.commandHistory.slice(0, this.historyIndex + 1);
        this.commandHistory.push(content);

        if (this.commandHistory.length > this.maxHistory) {
            this.commandHistory.shift();
        } else {
            this.historyIndex++;
        }
    }

    undo() {
        if (this.historyIndex > 0) {
            this.historyIndex--;
            this.editor.innerHTML = this.commandHistory[this.historyIndex];
            this.syncContent();
        }
    }

    redo() {
        if (this.historyIndex < this.commandHistory.length - 1) {
            this.historyIndex++;
            this.editor.innerHTML = this.commandHistory[this.historyIndex];
            this.syncContent();
        }
    }

    getContent(format = 'html') {
        if (format === 'html') {
            return this.sanitizeHTML(this.editor.innerHTML);
        } else if (format === 'text') {
            return this.editor.innerText;
        }
    }

    setContent(content) {
        this.editor.innerHTML = this.sanitizeHTML(content);
        this.syncContent();
        this.saveState();
    }

    syncContent() {
        const content = this.getContent();
        if (this.element.tagName === 'TEXTAREA' || this.element.tagName === 'INPUT') {
            this.element.value = content;
        } else {
            this.element.innerHTML = content;
        }

        const event = new Event('change', { bubbles: true });
        this.element.dispatchEvent(event);
    }

    save() {
        this.syncContent();
    }

    destroy() {
        this.syncContent();
        this.element.style.display = '';

        if (this.wrapper && this.wrapper.parentNode) {
            this.wrapper.parentNode.removeChild(this.wrapper);
        }
    }
}

// Global RTE registry
window.RTE = {
    editors: new Map(),

    init(selector, options = {}) {
        const elements = typeof selector === 'string'
            ? document.querySelectorAll(selector)
            : [selector];

        const instances = [];
        elements.forEach(el => {
            const id = el.id || `rte-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
            if (!el.id) el.id = id;

            if (this.editors.has(id)) {
                this.editors.get(id).destroy();
            }

            const editor = new RichTextEditor(el, options);
            this.editors.set(id, editor);
            instances.push(editor);
        });

        return instances.length === 1 ? instances[0] : instances;
    },

    get(id) {
        return this.editors.get(id);
    },

    remove(id) {
        if (id) {
            const editor = this.editors.get(id);
            if (editor) {
                editor.destroy();
                this.editors.delete(id);
            }
        } else {
            this.editors.forEach((editor) => {
                editor.destroy();
            });
            this.editors.clear();
        }
    }
};

// Compatibility layer for TinyMCE API
window.tinymce = {
    init: function(config) {
        const selector = config.selector;
        const onImageUpload = config.images_upload_handler;
        const onFilePicker = config.file_picker_callback;

        // Get maxChars from textarea data attribute
        const element = document.querySelector(selector);
        const maxChars = element ? parseInt(element.getAttribute('data-max-chars')) : null;

        console.log(`[tinymce.init] Selector: ${selector}, maxChars: ${maxChars}`);

        RTE.init(selector, {
            height: config.height || 400,
            toolbar: 'bold italic underline | p h1 h2 h3 | bullist numlist | link image | undo redo',
            maxChars: maxChars && maxChars > 0 ? maxChars : null,
            onFilePicker: onFilePicker ? (callback) => {
                onFilePicker(callback, null, { filetype: 'image' });
            } : null,
            onImageUpload: onImageUpload ? async (file) => {
                return new Promise((resolve, reject) => {
                    const reader = new FileReader();
                    reader.onload = () => {
                        const blob = new Blob([reader.result], { type: file.type });
                        onImageUpload({
                            blob: () => blob,
                            filename: () => file.name
                        }, resolve, reject);
                    };
                    reader.readAsArrayBuffer(file);
                });
            } : null,
            onChange: config.setup ? () => {
                const editorInstance = RTE.get(selector.replace('#', ''));
                if (editorInstance) editorInstance.save();
            } : null
        });
    },

    get: function(id) {
        const editor = RTE.get(id);
        if (!editor) return null;

        return {
            getContent: () => editor.getContent(),
            setContent: (content) => editor.setContent(content),
            save: () => editor.save()
        };
    },

    remove: function(selector) {
        if (selector) {
            const id = typeof selector === 'string' ? selector.replace('#', '') : selector;
            RTE.remove(id);
        } else {
            RTE.remove();
        }
    }
};

// Integration functions (compatible with existing code)
function initializeTinyMCE(elementId) {
    // Get the textarea element to read max-chars attribute
    const textarea = document.getElementById(elementId);
    const maxChars = textarea ? textarea.getAttribute('data-max-chars') : null;

    console.log(`initializeTinyMCE: ${elementId}, maxChars from data attr: ${maxChars}`);

    tinymce.init({
        selector: `#${elementId}`,
        height: 400,
        images_upload_handler: function (blobInfo, success, failure, progress) {
            uploadImageToRTE(blobInfo, success, failure, progress);
        },
        file_picker_callback: function (callback, value, meta) {
            openRTEFilePicker(callback, value, meta);
        },
        setup: function(editor) {
            // Store max chars in editor instance for later use
            if (maxChars && parseInt(maxChars) > 0) {
                editor.maxChars = parseInt(maxChars);
                console.log(`Stored maxChars ${editor.maxChars} in editor instance`);
            }
        }
    });
}

async function uploadImageToRTE(blobInfo, success, failure, progress) {
    try {
        const formData = new FormData();
        formData.append('file', blobInfo.blob(), blobInfo.filename());
        formData.append('type', 'tinymce_upload');

        const config = {
            method: 'POST',
            body: formData
        };

        if (authToken) {
            config.headers = {
                'Authorization': `Bearer ${authToken}`
            };
        }

        const response = await fetch(`${API_BASE_URL}/upload`, config);
        const data = await response.json();

        if (!response.ok || !data.success) {
            throw new Error(data.error || 'Upload failed');
        }

        success(data.url);
    } catch (error) {
        console.error('RTE upload error:', error);
        failure('Image upload failed: ' + error.message);
    }
}

function openRTEFilePicker(callback, value, meta) {
    const input = document.createElement('input');
    input.setAttribute('type', 'file');

    if (meta.filetype === 'image') {
        input.setAttribute('accept', 'image/*');
    } else if (meta.filetype === 'media') {
        input.setAttribute('accept', 'image/*,video/*,audio/*');
    } else {
        input.setAttribute('accept', '*/*');
    }

    input.onchange = function() {
        const file = this.files[0];
        if (!file) return;

        const formData = new FormData();
        formData.append('file', file);
        formData.append('type', 'tinymce_upload');

        const config = {
            method: 'POST',
            body: formData
        };

        if (authToken) {
            config.headers = {
                'Authorization': `Bearer ${authToken}`
            };
        }

        fetch(`${API_BASE_URL}/upload`, config)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    callback(data.url, { title: file.name });
                } else {
                    throw new Error(data.error || 'Upload failed');
                }
            })
            .catch(error => {
                console.error('File upload error:', error);
                showToast('File upload failed: ' + error.message, 'error');
            });
    };

    input.click();
}

function cleanupTinyMCE() {
    tinymce.remove();
}

// Character counter for rich paragraph fields
function setupCharacterCounter(fieldId, maxChars, charCounterId) {
    console.log(`[setupCharacterCounter] Starting setup for ${fieldId} with max ${maxChars}`);
    let lastValidContent = '';
    let isBlocking = false;

    // Helper to get text length without HTML tags
    const getTextLength = (html) => {
        if (!html) return 0;
        const temp = document.createElement('div');
        temp.innerHTML = html;
        // Use textContent instead of innerText for more reliable counting
        const text = temp.textContent || temp.innerText || '';
        return text.trim().length;
    };

    const updateCounter = () => {
        // Get the RTE editor instance
        const editor = RTE.get(fieldId);
        if (!editor) {
            console.warn(`Editor not found for ${fieldId}`);
            return;
        }

        // Get content and calculate text length (without HTML tags)
        const content = editor.getContent();
        const currentLength = getTextLength(content);

        console.log(`[${fieldId}] Current: ${currentLength}/${maxChars} chars`);

        // Update counter display
        const counterElement = document.getElementById(charCounterId);
        if (counterElement) {
            const charCountSpan = counterElement.querySelector('.char-count');
            if (charCountSpan) {
                charCountSpan.textContent = currentLength;

                // Change color based on usage
                const percentage = (currentLength / maxChars) * 100;
                if (percentage >= 100) {
                    counterElement.className = 'text-xs text-right font-medium text-red-600 font-bold';
                } else if (percentage >= 90) {
                    counterElement.className = 'text-xs text-right font-medium text-orange-600';
                } else if (percentage >= 75) {
                    counterElement.className = 'text-xs text-right font-medium text-yellow-600';
                } else {
                    counterElement.className = 'text-xs text-right font-medium text-gray-600';
                }
            }
        }

        // Prevent further input if limit exceeded
        if (currentLength > maxChars) {
            console.log(`[${fieldId}] LIMIT EXCEEDED - Reverting!`);

            // Restore the last valid content
            if (lastValidContent) {
                editor.setContent(lastValidContent);

                if (!editor.editor.dataset.limitWarningShown) {
                    showToast(`Character limit reached (${maxChars} characters maximum)`, 'warning');
                    editor.editor.dataset.limitWarningShown = 'true';
                    setTimeout(() => delete editor.editor.dataset.limitWarningShown, 2000);
                }
            }
        } else {
            // Save valid content
            lastValidContent = content;
        }
    };

    // Set up the editor with change callback
    let attempts = 0;
    const maxAttempts = 50; // Try for 5 seconds (100ms intervals)

    const checkEditor = setInterval(() => {
        attempts++;
        const editor = RTE.get(fieldId);

        if (attempts % 10 === 0) {
            console.log(`[${fieldId}] Attempt ${attempts}/${maxAttempts} - Editor found: ${!!editor}, Editor.editor: ${!!(editor && editor.editor)}`);
        }

        if (editor && editor.editor) {
            clearInterval(checkEditor);
            console.log(`[${fieldId}] ✅ Editor found after ${attempts} attempts!`);

            console.log(`[${fieldId}] ✓ Character counter initialized with max ${maxChars}`);

            // Initial count and save initial content
            const initialContent = editor.getContent();
            const initialLength = getTextLength(initialContent);

            console.log(`[${fieldId}] Initial content length: ${initialLength}`);

            if (initialLength <= maxChars) {
                lastValidContent = initialContent;
            }
            updateCounter();

            // Update on input with immediate response
            editor.editor.addEventListener('input', () => {
                updateCounter();
            });

            editor.editor.addEventListener('paste', (e) => {
                // Check current length before paste
                const currentContent = editor.getContent();
                const currentLength = getTextLength(currentContent);

                // If already at limit, prevent paste entirely
                if (currentLength >= maxChars) {
                    e.preventDefault();
                    showToast(`Cannot paste - character limit reached (${maxChars} maximum)`, 'warning');
                    return;
                }

                // Otherwise, let paste happen and check after
                setTimeout(() => {
                    updateCounter();
                    const newLength = getTextLength(editor.getContent());
                    if (newLength > maxChars) {
                        // Content after paste exceeded limit, revert
                        console.log(`[${fieldId}] Paste exceeded limit, reverting`);
                        editor.setContent(lastValidContent);
                        updateCounter();
                        showToast(`Pasted content exceeded character limit. Content was not pasted.`, 'warning');
                    }
                }, 50);
            });

            // Prevent typing when at limit (primary blocking) - MUST be in capture phase
            const handleKeydown = (e) => {
                // Get current state
                const content = editor.getContent();
                const currentLength = getTextLength(content);
                const selection = window.getSelection();
                const hasSelection = selection && selection.toString().length > 0;

                console.log(`[${fieldId}] Keydown: key="${e.key}", length=${currentLength}/${maxChars}, hasSelection=${hasSelection}`);

                // Allow navigation and edit keys
                const allowedKeys = ['Backspace', 'Delete', 'ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown', 'Home', 'End', 'Tab', 'Escape', 'Enter'];

                // Always allow Ctrl/Cmd shortcuts (for copy, select all, etc.)
                if (e.ctrlKey || e.metaKey) {
                    console.log(`[${fieldId}] Allowing Ctrl/Cmd shortcut`);
                    return;
                }

                // Always allow navigation keys
                if (allowedKeys.includes(e.key)) {
                    console.log(`[${fieldId}] Allowing navigation key: ${e.key}`);
                    return;
                }

                // Allow typing if user has selected text (will replace selection, reducing length)
                if (hasSelection) {
                    console.log(`[${fieldId}] Allowing key because text is selected`);
                    return;
                }

                // Block input if at or over limit
                if (currentLength >= maxChars) {
                    isBlocking = true;
                    console.warn(`[${fieldId}] 🚫 BLOCKING KEY: "${e.key}" (${currentLength}/${maxChars})`);
                    e.preventDefault();
                    e.stopPropagation();
                    e.stopImmediatePropagation();

                    if (!editor.editor.dataset.limitWarningShown) {
                        showToast(`Character limit reached: ${maxChars} characters maximum`, 'warning');
                        editor.editor.dataset.limitWarningShown = 'true';
                        setTimeout(() => {
                            delete editor.editor.dataset.limitWarningShown;
                            isBlocking = false;
                        }, 3000);
                    }

                    return false;
                }
            };

            editor.editor.addEventListener('keydown', handleKeydown, true); // Use capture phase

            // Additional blocking with beforeinput event - CRITICAL for blocking all input
            const handleBeforeInput = (e) => {
                const content = editor.getContent();
                const currentLength = getTextLength(content);
                const selection = window.getSelection();
                const hasSelection = selection && selection.toString().length > 0;

                console.log(`[${fieldId}] BeforeInput: type="${e.inputType}", length=${currentLength}/${maxChars}, hasSelection=${hasSelection}`);

                // Allow deletions and formatting changes
                if (e.inputType && (
                    e.inputType.includes('delete') ||
                    e.inputType.includes('Delete') ||
                    e.inputType.includes('format') ||
                    e.inputType === 'historyUndo' ||
                    e.inputType === 'historyRedo'
                )) {
                    console.log(`[${fieldId}] Allowing input type: ${e.inputType}`);
                    return;
                }

                // Allow if user has selected text (will replace selection)
                if (hasSelection) {
                    console.log(`[${fieldId}] Allowing beforeinput because text is selected`);
                    return;
                }

                // Block if at limit
                if (currentLength >= maxChars) {
                    console.warn(`[${fieldId}] 🚫 BLOCKING BEFOREINPUT: ${e.inputType} (${currentLength}/${maxChars})`);
                    e.preventDefault();
                    e.stopPropagation();
                    e.stopImmediatePropagation();
                    return false;
                }
            };

            editor.editor.addEventListener('beforeinput', handleBeforeInput, true); // Use capture phase
        } else if (attempts >= maxAttempts) {
            clearInterval(checkEditor);
            console.error(`[${fieldId}] ✗ Failed to find editor after ${maxAttempts} attempts. Character limit enforcement will not work.`);
            console.error(`[${fieldId}] Debug: RTE.editors has ${RTE.editors.size} editors:`, Array.from(RTE.editors.keys()));

            // Show warning to user
            const counterElement = document.getElementById(charCounterId);
            if (counterElement) {
                counterElement.innerHTML = '<span class="text-red-600">⚠️ Character counter failed to initialize</span>';
            }
        }
    }, 100);
}

// Media selector functions for default values
function openMediaSelectorForDefault(mediaType) {
    // Store the current context
    window.defaultMediaSelectionType = mediaType;
    
    // Load and show media filtered by type
    loadMediaForDefault(mediaType);
    openModal('defaultMediaModal');
}

function clearDefaultImage() {
    document.getElementById('fieldDefaultValue').value = '';
    document.getElementById('selectedDefaultImage').classList.add('hidden');
}

function clearDefaultFile() {
    document.getElementById('fieldDefaultValue').value = '';
    document.getElementById('selectedDefaultFile').classList.add('hidden');
}

function clearDefaultMedia() {
    document.getElementById('fieldDefaultValue').value = '';
    document.getElementById('selectedDefaultMedia').classList.add('hidden');
}

// Media selector functions for record fields
function openMediaSelectorForRecord(fieldId, mediaType) {
    // Store the current context
    window.recordMediaSelectionFieldId = fieldId;
    window.recordMediaSelectionType = mediaType;
    
    // Load and show media filtered by type
    loadMediaForRecord(mediaType);
    openModal('recordMediaModal');
}

function clearRecordMedia(fieldId) {
    document.getElementById(fieldId).value = '';
    document.getElementById(fieldId + '_preview').classList.add('hidden');
    
    // Trigger validation after clearing
    const form = document.getElementById('recordForm');
    if (form) {
        const event = new Event('input', { bubbles: true });
        document.getElementById(fieldId).dispatchEvent(event);
    }
}

function removeSelectedMedia(fieldId, index) {
    const hiddenField = document.getElementById(fieldId);
    const currentValue = hiddenField.value;
    const selections = currentValue ? currentValue.split(',').filter(v => v.trim()) : [];
    
    // Remove the item at the specified index
    selections.splice(index, 1);
    
    // Update the field value
    hiddenField.value = selections.join(',');
    
    // Determine media type from field ID patterns or the presence of preview containers
    let mediaType = 'file';  // default
    if (fieldId.includes('image') || (document.getElementById(fieldId + '_preview_grid') && !fieldId.includes('media'))) {
        mediaType = 'image';
    } else if (fieldId.includes('media') || window.recordMediaSelectionType === 'media') {
        mediaType = 'media';
    }
    
    // Update preview
    updateMediaPreview(fieldId, mediaType, selections);
    
    // Trigger validation
    const form = document.getElementById('recordForm');
    if (form) {
        const event = new Event('input', { bubbles: true });
        hiddenField.dispatchEvent(event);
    }
}

async function loadMediaForRecord(mediaType, page = 1, search = '', searchType = 'name', tag = '') {
    try {
        // Initialize pagination state if not exists
        if (!window.recordMediaPagination) {
            window.recordMediaPagination = {
                currentPage: 1,
                perPage: 12,
                total: 0,
                search: '',
                search_type: 'name',
                tag: ''
            };
        }

        // Update pagination state
        window.recordMediaPagination.currentPage = page;
        window.recordMediaPagination.search = search;
        window.recordMediaPagination.search_type = searchType;
        window.recordMediaPagination.tag = tag;

        let queryParams = new URLSearchParams({
            page: page,
            per_page: window.recordMediaPagination.perPage
        });

        if (search.trim()) {
            if (searchType === 'tag') {
                queryParams.append('tag', search.trim());
                queryParams.append('search_type', 'tag');
            } else {
                queryParams.append('search', search.trim());
                queryParams.append('search_type', 'name');
            }
        }

        if (tag.trim()) {
            queryParams.append('tag', tag.trim());
        }

        if (mediaType !== 'media') {
            queryParams.append('type', mediaType);
        }

        const response = await apiRequest(`/files?${queryParams.toString()}`);

        let files = [];
        let pagination = null;

        if (response && response.items && Array.isArray(response.items)) {
            files = response.items;
            pagination = response.pagination;
        } else if (Array.isArray(response)) {
            files = response;
        }

        // Filter for media type if needed (server-side filtering might not be perfect)
        if (mediaType === 'media') {
            files = files.filter(file =>
                file.file_type.startsWith('image/') ||
                file.file_type.startsWith('video/')
            );
        }

        // Update pagination state
        if (pagination) {
            window.recordMediaPagination.total = pagination.total;
        }

        renderRecordMediaSelector(files, mediaType, pagination);
    } catch (error) {
        console.error('Error loading media for record:', error);
        showToast('Failed to load media files: ' + error.message, 'error');
    }
}

function renderRecordMediaSelector(files, mediaType, pagination = null) {
    // Get currently selected files to show them as selected
    const currentFieldId = window.recordMediaSelectionFieldId;
    const currentValue = currentFieldId ? document.getElementById(currentFieldId)?.value || '' : '';
    const currentSelections = currentValue ? currentValue.split(',').filter(v => v.trim()) : [];

    let modalHtml = `
        <div id="recordMediaModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
            <div class="bg-white rounded-lg shadow-xl max-w-4xl w-full p-6 max-h-[90vh] overflow-y-auto">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Select ${mediaType === 'image' ? 'Images' : mediaType === 'media' ? 'Images & Videos' : 'Files'} (Multiple allowed)</h3>

                <!-- Search Bar -->
                <div class="mb-4 space-y-3">
                    <div class="flex gap-2">
                        <div class="flex-1 relative">
                            <input type="text" id="recordMediaSearch" placeholder="Search files by name..." class="w-full px-4 py-2 pr-10 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" autocomplete="off" value="${window.recordMediaPagination?.search || ''}" onkeyup="handleRecordMediaSearch(event)">
                            <div class="absolute inset-y-0 right-0 flex items-center pr-3">
                                <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                                </svg>
                            </div>
                        </div>
                        <select id="recordMediaSearchType" class="px-3 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" onchange="handleRecordMediaSearchTypeChange()">
                            <option value="name" ${window.recordMediaPagination?.search_type === 'name' || !window.recordMediaPagination?.search_type ? 'selected' : ''}>By Name</option>
                            <option value="tag" ${window.recordMediaPagination?.search_type === 'tag' ? 'selected' : ''}>By Tag</option>
                        </select>
                    </div>

                    <!-- Tag Filter -->
                    <div class="flex gap-2">
                        <div class="flex-1 relative">
                            <input type="text" id="recordMediaTagFilter" placeholder="Filter by tag..." class="w-full px-4 py-2 pr-10 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500" autocomplete="off" value="${window.recordMediaPagination?.tag || ''}" onkeyup="handleRecordMediaTagFilter(event)">
                            <div class="absolute inset-y-0 right-0 flex items-center pr-3">
                                <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 010 2.828l-7 7a2 2 0 01-2.828 0l-7-7A1.994 1.994 0 013 12V7a4 4 0 014-4z"></path>
                                </svg>
                            </div>
                        </div>
                        <button onclick="clearRecordMediaFilters()" class="px-4 py-2 text-gray-600 hover:text-gray-800 border border-gray-300 rounded-lg hover:bg-gray-50">
                            Clear
                        </button>
                    </div>
                </div>

                <div class="mb-4 p-3 bg-blue-50 rounded-lg">
                    <p class="text-sm text-blue-700">Click on items to select/deselect them. Selected items will be highlighted.</p>
                    <div class="mt-2">
                        <span class="text-sm font-medium text-blue-800">Selected: </span>
                        <span id="selectedCount" class="text-sm text-blue-700">${currentSelections.length}</span>
                    </div>
                </div>
                <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 mb-6" id="recordMediaGrid">
                    ${files.map(file => {
                        const isImage = file.file_type.startsWith('image/');
                        const isVideo = file.file_type.startsWith('video/');
                        const isAudio = file.file_type.startsWith('audio/');
                        const isSelected = currentSelections.includes(file.url);
                        return `
                            <div class="border rounded-lg p-3 cursor-pointer transition-all ${isSelected ? 'border-blue-500 bg-blue-50' : 'border-gray-200 hover:bg-gray-50'}"
                                 onclick="toggleRecordMediaSelection('${file.id}', '${file.url}', '${escapeJs(file.original_name)}', ${isImage ? 'true' : 'false'}, ${isVideo ? 'true' : 'false'})"
                                 data-file-url="${file.url}"
                                 data-file-name="${escapeHtml(file.filename)}"
                                 data-original-name="${escapeHtml(file.original_name)}">
                                ${isSelected ?
                                    `<div class="absolute top-1 right-1 bg-blue-500 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs">✓</div>` :
                                    ''
                                }
                                ${isImage ?
                                    `<img src="${file.url}" alt="${escapeHtml(file.original_name)}" class="w-full h-24 object-cover rounded mb-2">` :
                                isVideo ?
                                    `<video src="${file.url}" class="w-full h-24 object-cover rounded mb-2" muted preload="metadata" controls></video>` :
                                isAudio ?
                                    `<div class="w-full h-24 bg-purple-100 rounded mb-2 flex items-center justify-center">
                                        <svg class="w-8 h-8 text-purple-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3"></path>
                                        </svg>
                                    </div>` :
                                    `<div class="w-full h-24 bg-gray-100 rounded mb-2 flex items-center justify-center">
                                        <svg class="w-8 h-8 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                                        </svg>
                                    </div>`
                                }
                                <p class="text-xs text-gray-600 truncate">${escapeHtml(file.original_name)}</p>
                            </div>
                        `;
                    }).join('')}
                </div>

                <!-- Pagination Controls -->
                ${pagination ? `
                <div class="flex items-center justify-between mt-4 pt-4 border-t border-gray-200">
                    <div class="text-sm text-gray-700">
                        Showing ${Math.min((pagination.page - 1) * pagination.per_page + 1, pagination.total)} to ${Math.min(pagination.page * pagination.per_page, pagination.total)} of ${pagination.total} files
                    </div>
                    <div class="flex items-center space-x-2">
                        <button onclick="loadRecordMediaPage(${pagination.page - 1})" class="px-3 py-1 border border-gray-300 rounded-md hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed" ${pagination.page <= 1 ? 'disabled' : ''}>
                            Previous
                        </button>
                        ${generatePageNumbers(pagination.page, Math.ceil(pagination.total / pagination.per_page))}
                        <button onclick="loadRecordMediaPage(${pagination.page + 1})" class="px-3 py-1 border border-gray-300 rounded-md hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed" ${pagination.page >= Math.ceil(pagination.total / pagination.per_page) ? 'disabled' : ''}>
                            Next
                        </button>
                    </div>
                </div>
                ` : ''}

                <div class="flex justify-between items-center mt-4">
                    <button onclick="clearAllSelections()" class="px-4 py-2 text-red-600 hover:text-red-800">Clear All</button>
                    <div class="flex space-x-3">
                        <button onclick="closeModal('recordMediaModal')" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                            Cancel
                        </button>
                        <button onclick="confirmMediaSelection()" class="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                            Done
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing modal if present
    const existingModal = document.getElementById('recordMediaModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Add modal to page
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Initialize selection tracking
    window.currentMediaSelections = [...currentSelections];
}

function toggleRecordMediaSelection(fileId, fileUrl, fileName, isImageStr, isVideoStr) {
    // Convert string parameters to booleans
    const isImage = isImageStr === 'true';
    const isVideo = isVideoStr === 'true';
    
    // Toggle selection
    const index = window.currentMediaSelections.indexOf(fileUrl);
    if (index > -1) {
        // Remove from selection
        window.currentMediaSelections.splice(index, 1);
    } else {
        // Add to selection
        window.currentMediaSelections.push(fileUrl);
    }

    // Update UI
    updateMediaSelectionUI(fileUrl, isImage, isVideo);
    updateSelectedCount();
}

function updateMediaSelectionUI(fileUrl, isImage, isVideo) {
    const mediaItem = document.querySelector(`[data-file-url="${fileUrl}"]`);
    if (!mediaItem) return;
    
    const isSelected = window.currentMediaSelections.includes(fileUrl);
    
    if (isSelected) {
        mediaItem.className = 'border rounded-lg p-3 cursor-pointer transition-all border-blue-500 bg-blue-50 relative';
        if (!mediaItem.querySelector('.absolute')) {
            mediaItem.insertAdjacentHTML('afterbegin', `<div class="absolute top-1 right-1 bg-blue-500 text-white rounded-full w-6 h-6 flex items-center justify-center text-xs">✓</div>`);
        }
    } else {
        mediaItem.className = 'border rounded-lg p-3 cursor-pointer transition-all border-gray-200 hover:bg-gray-50';
        const checkmark = mediaItem.querySelector('.absolute');
        if (checkmark) {
            checkmark.remove();
        }
    }
}

function updateSelectedCount() {
    const countElement = document.getElementById('selectedCount');
    if (countElement) {
        countElement.textContent = window.currentMediaSelections.length;
    }
}

function clearAllSelections() {
    window.currentMediaSelections = [];
    
    // Update all items
    document.querySelectorAll('[data-file-url]').forEach(item => {
        item.className = 'border rounded-lg p-3 cursor-pointer transition-all border-gray-200 hover:bg-gray-50';
        const checkmark = item.querySelector('.absolute');
        if (checkmark) {
            checkmark.remove();
        }
    });
    
    updateSelectedCount();
}

function confirmMediaSelection() {
    const fieldId = window.recordMediaSelectionFieldId;
    const mediaType = window.recordMediaSelectionType;

    // Check if this is for a link entry
    if (window.isLinkEntry && window.currentLinkFieldId && window.currentLinkEntryIndex !== undefined) {
        // Handle link entry media selection
        const linkFieldId = window.currentLinkFieldId;
        const entryIndex = window.currentLinkEntryIndex;
        const selectedMedia = window.currentMediaSelections[0]; // Only take first selection for links

        console.log('Link entry media selection:', {
            linkFieldId,
            entryIndex,
            selectedMedia,
            selections: window.currentMediaSelections
        });

        if (selectedMedia) {
            // Find the link entry and update its media input
            const container = document.getElementById(linkFieldId + '_container');
            console.log('Container found:', !!container);

            const entry = container ? container.querySelector(`[data-index="${entryIndex}"]`) : null;
            console.log('Entry found:', !!entry);

            if (entry) {
                const mediaInput = entry.querySelector('.link-media-input');
                const mediaPreview = entry.querySelector('.link-media-preview');
                const img = mediaPreview ? mediaPreview.querySelector('img') : null;

                console.log('Elements found:', {
                    mediaInput: !!mediaInput,
                    mediaPreview: !!mediaPreview,
                    img: !!img
                });

                if (mediaInput) {
                    mediaInput.value = selectedMedia;
                    console.log('Updated media input value to:', selectedMedia);
                }

                if (mediaPreview && img) {
                    img.src = selectedMedia;
                    mediaPreview.classList.remove('hidden');
                    console.log('Updated image preview and made visible');
                } else {
                    console.warn('Could not update preview - missing elements');
                }

                // Update the main field value
                console.log('Calling updateLinkFieldFromEntry after media selection');
                updateLinkFieldFromEntry(linkFieldId);
            } else {
                console.error('Entry not found with index:', entryIndex);
            }
        } else {
            console.warn('No media selected');
        }

        // Clear link entry state
        window.isLinkEntry = false;
        window.currentLinkFieldId = null;
        window.currentLinkEntryIndex = null;
    } else {
        // Handle regular field media selection
        const hiddenField = document.getElementById(fieldId);
        if (hiddenField) {
            hiddenField.value = window.currentMediaSelections.join(',');

            // Update preview
            updateMediaPreview(fieldId, mediaType, window.currentMediaSelections);

            // Trigger validation after selection
            const form = document.getElementById('recordForm');
            if (form) {
                const event = new Event('input', { bubbles: true });
                hiddenField.dispatchEvent(event);
            }
        }
    }

    // Close modal and clear selection state
    closeModal('recordMediaModal');
    window.currentMediaSelections = [];
}

function updateMediaPreview(fieldId, mediaType, selections) {
    const previewContainer = document.getElementById(fieldId + '_preview');
    if (!previewContainer) {
        console.log('No preview container found for fieldId:', fieldId);
        return;
    }

    console.log('updateMediaPreview called:', { fieldId, mediaType, selections });

    if (selections.length === 0) {
        previewContainer.classList.add('hidden');
        return;
    }

    if (mediaType === 'image' || mediaType === 'media') {
        const previewGrid = document.getElementById(fieldId + '_preview_grid');
        if (previewGrid) {
            previewGrid.innerHTML = selections.map((url, index) => {
                const isVideo = url.match(/\.(mp4|webm|ogg|mov|avi|mkv)$/i);
                console.log('Processing URL:', url, 'isVideo:', !!isVideo, 'mediaType:', mediaType);

                if (mediaType === 'media' && isVideo) {
                    const videoSrc = escapeHtml(makeRelativeUrl(url));
                    console.log('Creating video preview with src:', videoSrc);
                    return `<div class="relative">
                        <video src="${videoSrc}" class="w-20 h-20 object-cover rounded border" muted preload="metadata" onerror="console.error('Video failed to load:', this.src)"></video>
                        <button type="button" onclick="removeSelectedMedia('${fieldId}', ${index})" class="absolute top-1 bg-red-500 text-white rounded-full w-5 h-5 text-xs flex items-center justify-center hover:bg-red-600 shadow-lg">×</button>
                    </div>`;
                } else {
                    const imgSrc = escapeHtml(makeRelativeUrl(url));
                    console.log('Creating image preview with src:', imgSrc);
                    return `<div class="relative">
                        <img src="${imgSrc}" alt="Selected media" class="w-20 h-20 object-cover rounded border" onerror="console.error('Image failed to load:', this.src)">
                        <button type="button" onclick="removeSelectedMedia('${fieldId}', ${index})" class="absolute top-1 bg-red-500 text-white rounded-full w-5 h-5 text-xs flex items-center justify-center hover:bg-red-600 shadow-lg">×</button>
                    </div>`;
                }
            }).join('');

            // Show the preview container
            previewContainer.classList.remove('hidden');
        }
    } else {
        const previewList = document.getElementById(fieldId + '_preview_list');
        if (previewList) {
            previewList.innerHTML = selections.map((url, index) => `
                <div class="relative flex items-center p-2 bg-gray-100 rounded">
                    <div class="flex items-center space-x-2 pr-6">
                        <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                        </svg>
                        <span>${url.split('/').pop()}</span>
                    </div>
                    <button type="button" onclick="removeSelectedMedia('${fieldId}', ${index})" class="absolute right-1 top-1/2 transform -translate-y-1/2 bg-red-500 text-white rounded-full w-5 h-5 text-xs flex items-center justify-center hover:bg-red-600 shadow-lg">×</button>
                </div>
            `).join('');

            // Show the preview container
            previewContainer.classList.remove('hidden');
        }
    }
    
    previewContainer.classList.remove('hidden');
}

// Function to handle media selection from the main media gallery
function selectMediaFromGallery(fileId, fileUrl, fileName, isImage, isVideo) {
    // Check if we're in RTE image selection mode
    if (window.rteMediaSelectionMode && window.rteImageCallback) {
        selectMediaForRTE(fileUrl);
        return;
    }

    // Check if we're in record creation/editing mode and have a form open
    const recordForm = document.getElementById('recordForm');
    if (!recordForm) {
        showToast('Please open a record form first to select media', 'warning');
        return;
    }

    // Try to find the most appropriate field to populate
    // Look for media, image, or file fields in the current form
    const mediaFields = recordForm.querySelectorAll('input[type="hidden"]');
    let targetField = null;

    // Priority order: media field > image field (if it's an image) > video field > file field
    for (let field of mediaFields) {
        const fieldId = field.id;
        const fieldLabel = document.querySelector(`label[for="${fieldId}"]`)?.textContent?.toLowerCase() || '';

        // Check for media field (accepts both images and videos)
        if (fieldId.includes('media') || fieldLabel.includes('media')) {
            targetField = field;
            break;
        }

        // Check for image field if selecting an image
        if (isImage && (fieldId.includes('image') || fieldLabel.includes('image'))) {
            targetField = field;
            break;
        }

        // Check for video field if selecting a video
        if (isVideo && (fieldId.includes('video') || fieldLabel.includes('video'))) {
            targetField = field;
            break;
        }

        // Check for general file field
        if (fieldId.includes('file') || fieldLabel.includes('file')) {
            targetField = field;
        }
    }

    if (!targetField) {
        showToast('No suitable field found to add this media. Please use the "Select from Media" button in a field.', 'warning');
        return;
    }

    // Add the file to the field
    const currentValue = targetField.value;
    const currentSelections = currentValue ? currentValue.split(',').filter(v => v.trim()) : [];

    // Check if already selected
    if (currentSelections.includes(fileUrl)) {
        showToast('This media is already selected in ' + targetField.id.replace('_', ' '), 'info');
        return;
    }

    // Add to selections
    currentSelections.push(fileUrl);
    targetField.value = currentSelections.join(',');

    // Determine media type for preview update
    let mediaType = 'file';
    if (targetField.id.includes('image')) {
        mediaType = 'image';
    } else if (targetField.id.includes('media')) {
        mediaType = 'media';
    }

    // Update preview
    console.log('Calling updateMediaPreview with:', { fieldId: targetField.id, mediaType, selections: currentSelections });
    updateMediaPreview(targetField.id, mediaType, currentSelections);

    // Show the preview container
    const previewContainer = document.getElementById(targetField.id + '_preview');
    if (previewContainer) {
        previewContainer.classList.remove('hidden');
        console.log('Preview container shown for:', targetField.id);
    }

    // Trigger validation
    const event = new Event('input', { bubbles: true });
    targetField.dispatchEvent(event);

    // Show success message
    const mediaType_display = isVideo ? 'Video' : isImage ? 'Image' : 'File';
    showToast(`${mediaType_display} "${fileName}" added to ${targetField.id.replace('_', ' ')}`, 'success');
}

// RTE Media Selection Functions
function openMediaSelectorForRTE(callback) {
    // Store the callback for RTE image insertion
    window.rteImageCallback = callback;
    window.rteMediaSelectionMode = true;

    // Load and show media filtered by images only
    loadMediaForRecord('image');
    openModal('recordMediaModal');
}

function selectMediaForRTE(fileUrl) {
    console.log('selectMediaForRTE called with URL:', fileUrl);
    console.log('Callback exists:', !!window.rteImageCallback);
    console.log('Callback type:', typeof window.rteImageCallback);

    if (window.rteImageCallback && typeof window.rteImageCallback === 'function') {
        try {
            console.log('Calling RTE callback with URL:', fileUrl);
            window.rteImageCallback(fileUrl);
            console.log('RTE callback completed successfully');

            closeModal('recordMediaModal');
            showToast('Image inserted into editor', 'success');

            // Clean up
            window.rteImageCallback = null;
            window.rteMediaSelectionMode = false;
        } catch (error) {
            console.error('Error in RTE callback:', error);
            showToast('Failed to insert image: ' + error.message, 'error');
        }
    } else {
        console.error('RTE callback not found or not a function');
        showToast('Image insertion failed - please try again', 'error');
    }
}

// Record Media Pagination Functions
function loadRecordMediaPage(page) {
    const mediaType = window.recordMediaSelectionType;
    const search = window.recordMediaPagination?.search || '';
    const searchType = window.recordMediaPagination?.search_type || 'name';
    const tag = window.recordMediaPagination?.tag || '';
    loadMediaForRecord(mediaType, page, search, searchType, tag);
}

function handleRecordMediaSearch(event) {
    // Debounce search
    clearTimeout(window.recordMediaSearchTimeout);
    window.recordMediaSearchTimeout = setTimeout(() => {
        const searchValue = event.target.value;
        const searchType = document.getElementById('recordMediaSearchType')?.value || 'name';
        const tag = document.getElementById('recordMediaTagFilter')?.value || '';
        const mediaType = window.recordMediaSelectionType;
        loadMediaForRecord(mediaType, 1, searchValue, searchType, tag);
    }, 300);
}

function handleRecordMediaTagFilter(event) {
    // Debounce tag filter
    clearTimeout(window.recordMediaTagTimeout);
    window.recordMediaTagTimeout = setTimeout(() => {
        const tagValue = event.target.value;
        const search = document.getElementById('recordMediaSearch')?.value || '';
        const searchType = document.getElementById('recordMediaSearchType')?.value || 'name';
        const mediaType = window.recordMediaSelectionType;
        loadMediaForRecord(mediaType, 1, search, searchType, tagValue);
    }, 300);
}

function handleRecordMediaSearchTypeChange() {
    const search = document.getElementById('recordMediaSearch')?.value || '';
    const searchType = document.getElementById('recordMediaSearchType')?.value || 'name';
    const tag = document.getElementById('recordMediaTagFilter')?.value || '';
    const mediaType = window.recordMediaSelectionType;

    // Update placeholder based on search type
    const searchInput = document.getElementById('recordMediaSearch');
    if (searchInput) {
        searchInput.placeholder = searchType === 'tag' ? 'Search files by tag...' : 'Search files by name...';
    }

    loadMediaForRecord(mediaType, 1, search, searchType, tag);
}

function clearRecordMediaFilters() {
    const searchInput = document.getElementById('recordMediaSearch');
    const tagInput = document.getElementById('recordMediaTagFilter');
    const searchTypeSelect = document.getElementById('recordMediaSearchType');

    if (searchInput) searchInput.value = '';
    if (tagInput) tagInput.value = '';
    if (searchTypeSelect) searchTypeSelect.value = 'name';

    const mediaType = window.recordMediaSelectionType;
    loadMediaForRecord(mediaType, 1, '', 'name', '');
}

function generatePageNumbers(currentPage, totalPages) {
    let pages = [];
    const maxVisiblePages = 5;
    let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);

    // Adjust start if we're near the end
    if (endPage - startPage + 1 < maxVisiblePages) {
        startPage = Math.max(1, endPage - maxVisiblePages + 1);
    }

    for (let i = startPage; i <= endPage; i++) {
        pages.push(`
            <button onclick="loadRecordMediaPage(${i})" class="px-3 py-1 border rounded-md text-sm ${i === currentPage ? 'bg-blue-600 text-white border-blue-600' : 'border-gray-300 hover:bg-gray-50'}">
                ${i}
            </button>
        `);
    }

    return pages.join('');
}

async function handleFieldSubmit(event) {
    event.preventDefault();
    
    const categoryId = document.getElementById('fieldsCategorySelect').value;
    const fieldName = document.getElementById('fieldName').value;
    const fieldType = document.getElementById('fieldType').value;
    const fieldRequired = document.getElementById('fieldRequired').checked;
    const fieldDefaultValue = document.getElementById('fieldDefaultValue').value;

    if (!categoryId) {
        showToast('Please select a category first', 'warning');
        return;
    }
    
    // Validate default value based on field type
    const validation = validateFieldDefaultValue(fieldType, fieldDefaultValue);
    if (!validation.valid) {
        showToast(validation.message, 'error');
        return;
    }

    setButtonLoading('fieldForm', 'fieldSpinner', true);

    try {
        const fieldData = {
            category_id: categoryId,
            name: fieldName,
            data_type: fieldType,
            is_required: fieldRequired,
            default_value: fieldDefaultValue
        };

        // Handle foreign key fields
        if (fieldType === 'foreign_key') {
            const foreignKeyTable = document.getElementById('foreignKeyTable').value;
            const foreignKeyColumn = document.getElementById('foreignKeyColumn').value;

            if (!foreignKeyTable || !foreignKeyColumn) {
                showToast('Please select both target table and column for foreign key field', 'error');
                setButtonLoading('fieldForm', 'fieldSpinner', false);
                return;
            }

            fieldData.foreign_key_table = foreignKeyTable;
            fieldData.foreign_key_column = foreignKeyColumn;
            fieldData.default_value = null; // Foreign keys don't have default values
        }

        // Handle array fields
        if (fieldType === 'array') {
            const arrayItemType = document.getElementById('arrayItemType').value;
            const arrayAllowMultiple = document.getElementById('arrayAllowMultiple').checked;

            fieldData.array_options = {
                item_type: arrayItemType,
                allow_multiple: arrayAllowMultiple
            };
            fieldData.default_value = null; // Arrays don't have simple default values
        }

        // Handle JSON fields
        if (fieldType === 'json') {
            const jsonValueType = document.getElementById('jsonValueType').value;
            const jsonAllowCustomKeys = document.getElementById('jsonAllowCustomKeys').checked;
            const jsonPredefinedKeys = document.getElementById('jsonPredefinedKeys').value;

            const predefinedKeysArray = jsonPredefinedKeys
                ? jsonPredefinedKeys.split(',').map(key => key.trim()).filter(key => key.length > 0)
                : [];

            fieldData.json_options = {
                value_type: jsonValueType,
                allow_custom_keys: jsonAllowCustomKeys,
                predefined_keys: predefinedKeysArray
            };
            fieldData.default_value = null; // JSON fields don't have simple default values
        }

        // Handle rich paragraph fields
        if (fieldType === 'rich_paragraph') {
            const maxChars = document.getElementById('richParagraphMaxChars').value;

            if (maxChars && parseInt(maxChars) > 0) {
                // Store in field_options to be consistent with other field types
                fieldData.field_options = {
                    max_chars: parseInt(maxChars)
                };
            }
        }

        if (editingField) {
            await apiRequest(`/fields/${editingField}`, {
                method: 'PUT',
                body: JSON.stringify(fieldData)
            });
            showToast('Field updated successfully');
        } else {
            await apiRequest('/fields', {
                method: 'POST',
                body: JSON.stringify(fieldData)
            });
            showToast('Field created successfully');
        }

        closeModal('fieldModal');
        loadFieldsForCategory();
    } catch (error) {
        console.error('Error saving field:', error);
        showToast('Error saving field: ' + error.message, 'error');
    } finally {
        setButtonLoading('fieldForm', 'fieldSpinner', false);
    }
}

async function handleRecordSubmit(event) {
    event.preventDefault();
    console.log("data 45 :",currentCategoryFields )
    const categoryId = document.getElementById('recordsCategorySelect').value;
    if (!categoryId) {
        showToast('Please select a category first', 'warning');
        return;
    }

    // Validate slug format before submission
    const slugInput = document.getElementById('record_slug');
    if (slugInput && slugInput.value.trim()) {
        const slugValidation = validateSlugFormat(slugInput.value.trim());
        if (!slugValidation.valid) {
            showToast(slugValidation.message, 'error');
            return;
        }
    }

    // Validate rich paragraph character limits
    for (const field of currentCategoryFields) {
        if (field.data_type === 'rich_paragraph' && field.field_options && field.field_options.max_chars) {
            const fieldId = `record_${field.name.replace(/\s+/g, '_')}`;
            const editor = RTE.get(fieldId);

            if (editor) {
                const content = editor.getContent();
                const currentLength = content.length;
                const maxChars = field.field_options.max_chars;

                if (currentLength > maxChars) {
                    showToast(`Field "${field.name}" exceeds maximum character limit (${currentLength}/${maxChars} characters). Please reduce the content.`, 'error');

                    // Highlight the field
                    const counterElement = document.getElementById(`${fieldId}_char_count`);
                    if (counterElement) {
                        counterElement.classList.add('animate-pulse');
                        setTimeout(() => counterElement.classList.remove('animate-pulse'), 2000);
                    }

                    return;
                }
            }
        }
    }

    setButtonLoading('recordForm', 'recordSpinner', true);

    try {
        const values = {};

        // Get slug
        if (slugInput) {
            values.slug = slugInput.value.trim();
        }

        // Get tags
        const tagsInput = document.getElementById('record_tags');
        if (tagsInput) {
            values.tags = tagsInput.value;
        }

        // Get field values
        currentCategoryFields.forEach(field => {
            const fieldId = `record_${field.name.replace(/\s+/g, '_')}`;
            const input = document.getElementById(fieldId);

            if (field.data_type === 'link') {
                console.log(`Looking for LINK field: ${field.name}, fieldId: ${fieldId}, found input:`, input);
                if (input) {
                    console.log(`Link field input value:`, input.value);
                }
            }
            
            if (input) {
                if (field.data_type === 'checkbox') {
                    values[field.name] = input.checked;
                } else if (field.data_type === 'image' || field.data_type === 'file') {
                    // For images and files, the value is now comma-separated URLs
                    const urls = input.value ? input.value.split(',').filter(url => url.trim()) : [];
                    values[field.name] = urls.length > 1 ? urls : (urls.length === 1 ? urls[0] : '');
                } else if (field.data_type === 'array') {
                    // For array fields, parse the JSON value
                    console.log(`Processing array field: ${field.name}, input value:`, input.value);
                    try {
                        if (input.value && input.value.trim() !== '') {
                            // Check if the value looks like JSON (starts with [ or {)
                            const trimmedValue = input.value.trim();
                            if (trimmedValue.startsWith('[') || trimmedValue.startsWith('{')) {
                                // Additional validation for malformed JSON
                                if (trimmedValue === '[' || trimmedValue === '{' || trimmedValue.length < 2) {
                                    console.warn(`Malformed JSON detected for array field ${field.name}:`, trimmedValue);
                                    values[field.name] = [];
                                } else {
                                    try {
                                        const arrayValue = JSON.parse(input.value);
                                        console.log(`Parsed array value for ${field.name}:`, arrayValue);
                                        values[field.name] = arrayValue;
                                    } catch (parseError) {
                                        console.error(`Error parsing array field value for ${field.name}:`, parseError, 'Raw value:', trimmedValue);
                                        values[field.name] = [];
                                    }
                                }
                            } else {
                                // Handle legacy comma-separated values
                                const arrayValue = input.value.split(',').map(v => v.trim()).filter(v => v);
                                console.log(`Converted comma-separated to array for ${field.name}:`, arrayValue);
                                values[field.name] = arrayValue;
                            }
                        } else {
                            console.log(`Empty array field: ${field.name}`);
                            // Always include array fields, even if empty, to ensure proper updates
                            values[field.name] = [];
                        }
                    } catch (e) {
                        console.error(`Error parsing array field value for ${field.name}:`, e, 'Raw value:', input.value);
                        // Try to fallback to comma-separated parsing
                        try {
                            const arrayValue = input.value.split(',').map(v => v.trim()).filter(v => v);
                            console.log(`Fallback comma-separated parsing for ${field.name}:`, arrayValue);
                            values[field.name] = arrayValue;
                        } catch (fallbackError) {
                            console.error(`Fallback parsing also failed for ${field.name}:`, fallbackError);
                            // Always include array fields, even if empty, to ensure proper updates
                            values[field.name] = [];
                        }
                    }
                } else if (field.data_type === 'foreign_key') {
                    // For foreign key fields, use the selected value
                    values[field.name] = input.value || null;
                } else if (field.data_type === 'rich_paragraph') {
                    // For rich paragraph fields, get content from TinyMCE
                    const editorId = fieldId;
                    if (typeof tinymce !== 'undefined' && tinymce.get(editorId)) {
                        values[field.name] = tinymce.get(editorId).getContent();
                    } else {
                        values[field.name] = input.value;
                    }
                } else if (field.data_type === 'json') {
                    // For JSON fields, parse the JSON object value
                    try {
                        if (input.value && input.value.trim() !== '') {
                            const jsonValue = JSON.parse(input.value);
                            values[field.name] = jsonValue;
                        } else {
                            // Always include JSON fields, even if empty, to ensure proper updates
                            values[field.name] = {};
                        }
                    } catch (e) {
                        console.error('Error parsing JSON field value:', e);
                        // Always include JSON fields, even if malformed, to ensure proper updates
                        values[field.name] = {};
                    }
                } else if (field.data_type === 'link') {
                    // For link fields, parse the JSON array value
                    console.log(`Processing link field: ${field.name}, input value:`, input.value);
                    console.log(`Input element found:`, !!input);
                    console.log(`Input type:`, input ? input.type : 'N/A');
                    try {
                        if (input.value && input.value.trim() !== '') {
                            const trimmedValue = input.value.trim();
                            console.log(`Trimmed value:`, trimmedValue);
                            console.log(`Attempting to parse JSON...`);
                            const linkValue = JSON.parse(trimmedValue);
                            console.log(`Successfully parsed link value for ${field.name}:`, linkValue);
                            console.log(`Is array:`, Array.isArray(linkValue));

                            // Filter out empty link entries (where both media and text are empty)
                            const filteredLinks = Array.isArray(linkValue)
                                ? linkValue.filter(link => {
                                    const hasMedia = link.media && link.media.trim() !== '';
                                    const hasText = link.text && link.text.trim() !== '';
                                    return hasMedia || hasText; // Keep if either media or text is present
                                })
                                : [];

                            console.log(`Filtered link value for ${field.name}:`, filteredLinks);
                            values[field.name] = filteredLinks;
                        } else {
                            console.warn(`Empty link field: ${field.name}`);
                            // Always include link fields, even if empty, to ensure proper updates
                            values[field.name] = [];
                        }
                    } catch (e) {
                        console.error(`ERROR parsing link field value for ${field.name}:`);
                        console.error(`Error object:`, e);
                        console.error(`Raw value:`, input.value);
                        console.error(`Value length:`, input.value ? input.value.length : 0);
                        // Always include link fields, even if malformed, to ensure proper updates
                        values[field.name] = [];
                    }
                } else {
                    values[field.name] = input.value;
                }
            }
        });

        const recordData = {
            category_id: categoryId,
            values: values
        };

        console.log('Final record data being sent:', recordData);

        let savedRecordId = editingRecord;

        if (editingRecord) {
            await apiRequest(`/records/${editingRecord}`, {
                method: 'PUT',
                body: JSON.stringify(recordData)
            });
            showToast('Record updated successfully');
        } else {
            const response = await apiRequest('/records', {
                method: 'POST',
                body: JSON.stringify(recordData)
            });
            savedRecordId = response.record?.id || response.id;
            showToast('Record created successfully');
        }

        // Save translations if a non-default language is selected
        const languageSelect = document.getElementById('record_language');
        if (languageSelect && savedRecordId) {
            const selectedLanguageCode = languageSelect.value;
            const defaultLanguageCode = window.recordCurrentLanguage;

            if (selectedLanguageCode && selectedLanguageCode !== defaultLanguageCode) {
                // We're editing a translation, save it
                try {
                    const translationData = {
                        language_code: selectedLanguageCode,
                        translations: {}
                    };

                    // Include all field values as translations
                    currentCategoryFields.forEach(field => {
                        if (values[field.name] !== undefined) {
                            translationData.translations[field.name] = values[field.name];
                        }
                    });

                    await apiRequest(`/records/${savedRecordId}/translations`, {
                        method: 'POST',
                        body: JSON.stringify(translationData)
                    });

                    showToast(`Translation saved for ${selectedLanguageCode}`, 'success');
                } catch (translationError) {
                    console.error('Error saving translation:', translationError);
                    showToast('Record saved but translation failed: ' + translationError.message, 'warning');
                }
            }
        }

        closeModal('recordModal');
        loadRecordsForCategory();
    } catch (error) {
        console.error('Error saving record:', error);
        showToast('Error saving record: ' + error.message, 'error');
    } finally {
        setButtonLoading('recordForm', 'recordSpinner', false);
    }
}

async function handleUploadSubmit(event) {
    event.preventDefault();

    const fileInput = document.getElementById('uploadFile');
    if (!fileInput.files.length) {
        showToast('Please select at least one file', 'warning');
        return;
    }

    const files = Array.from(fileInput.files);
    const tagsInput = document.getElementById('uploadTags');

    console.log('Upload Debug - Multiple files info:', {
        count: files.length,
        files: files.map(f => ({ name: f.name, size: f.size, type: f.type }))
    });

    // Validate file sizes before uploading
    const oversizedFiles = [];
  

    if (oversizedFiles.length > 0) {
        showToast('Some files are too large:', 'error');
        oversizedFiles.forEach(msg => showToast(msg, 'warning'));
        return;
    }

    setButtonLoading('uploadSubmitBtn', 'uploadSpinner', true);

    try {
        let successCount = 0;
        let failCount = 0;
        const errors = [];

        // Upload files one by one (could be done in parallel but sequential is safer)
        for (let i = 0; i < files.length; i++) {
            const file = files[i];

            try {
                const formData = new FormData();
                formData.append('file', file);

                // Add tags if provided
                if (tagsInput && tagsInput.value.trim()) {
                    formData.append('tags', tagsInput.value.trim());
                }

                const config = {
                    method: 'POST',
                    headers: {},
                    body: formData
                };

                if (authToken) {
                    config.headers['Authorization'] = `Bearer ${authToken}`;
                }

                const response = await fetch(`${API_BASE_URL}/upload`, config);
                const responseText = await response.text();

                let data;
                try {
                    data = JSON.parse(responseText);
                } catch (parseError) {
                    throw new Error('Server returned invalid JSON: ' + responseText.substring(0, 200));
                }

                if (!response.ok || !data.success) {
                    throw new Error(data.error || 'Upload failed');
                }

                successCount++;

                // Update progress for user
                if (files.length > 1) {
                    showToast(`Uploaded ${successCount}/${files.length}: ${file.name}`, 'success');
                }

            } catch (fileError) {
                failCount++;
                errors.push(`${file.name}: ${fileError.message}`);
                console.error(`Error uploading ${file.name}:`, fileError);
            }
        }

        // Show final result
        if (successCount === files.length) {
            showToast(`All ${successCount} files uploaded successfully!`, 'success');
        } else if (successCount > 0) {
            showToast(`${successCount} files uploaded, ${failCount} failed`, 'warning');
            if (errors.length > 0) {
                console.error('Upload errors:', errors);
                showToast(`Errors: ${errors.slice(0, 3).join('; ')}${errors.length > 3 ? '...' : ''}`, 'error');
            }
        } else {
            showToast('All uploads failed', 'error');
            if (errors.length > 0) {
                showToast(`Errors: ${errors.slice(0, 2).join('; ')}${errors.length > 2 ? '...' : ''}`, 'error');
            }
        }

        // Clear form and close modal if any uploads succeeded
        if (successCount > 0) {
            fileInput.value = '';
            if (tagsInput) tagsInput.value = '';
            hideSelectedFilesPreview();
            closeModal('uploadModal');

            if (currentTab === 'media') {
                loadMedia();
            }
        }

    } catch (error) {
        console.error('Error in batch upload:', error);
        showToast('Error uploading files: ' + error.message, 'error');
    } finally {
        setButtonLoading('uploadSubmitBtn', 'uploadSpinner', false);
    }
}

// File preview functions
async function showSelectedFilesPreview() {
    const fileInput = document.getElementById('uploadFile');
    const preview = document.getElementById('selectedFilesPreview');
    const filesList = document.getElementById('filesList');

    if (!fileInput.files.length) {
        hideSelectedFilesPreview();
        return;
    }

    const files = Array.from(fileInput.files);
    preview.classList.remove('hidden');

    // No file size limits
    const isTooLarge = false;

    filesList.innerHTML = files.map((file, index) => {
        const size = formatFileSize(file.size);
        const isImage = file.type.startsWith('image/');
        const borderColor = 'bg-gray-50 border';

        return `
        <div class="flex items-center justify-between p-2 rounded ${borderColor}">
            <div class="flex items-center space-x-2 flex-1 min-w-0">
                <div class="flex-shrink-0">
                    ${isTooLarge ?
                        '<svg class="w-4 h-4 text-red-600" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/></svg>' :
                        isImage ?
                        '<svg class="w-4 h-4 text-green-600" fill="currentColor" viewBox="0 0 20 20"><path d="M4 3a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V5a2 2 0 00-2-2H4zm12 12H4l4-8 3 6 2-4 3 6z"/></svg>' :
                        '<svg class="w-4 h-4 text-blue-600" fill="currentColor" viewBox="0 0 20 20"><path d="M4 4a2 2 0 00-2 2v8a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2H4zm0 2h12v8H4V6z"/></svg>'
                    }
                </div>
                <div class="min-w-0 flex-1">
                    <p class="text-sm font-medium ${isTooLarge ? 'text-red-900' : 'text-gray-900'} truncate">${escapeHtml(file.name)}</p>
                    <p class="text-xs ${isTooLarge ? 'text-red-600 font-semibold' : 'text-gray-500'}">${size}${isTooLarge ? ' - TOO LARGE! Max: ' + formatFileSize(maxSize) : ''}</p>
                </div>
            </div>
            <button type="button" onclick="removeSelectedFile(${index})" class="text-red-600 hover:text-red-800 ml-2 flex-shrink-0">
                <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                    <path d="M6 6L14 14M6 14L14 6" stroke="currentColor" stroke-width="2" stroke-linecap="round"/>
                </svg>
            </button>
        </div>
        `;
    }).join('');
}

function hideSelectedFilesPreview() {
    const preview = document.getElementById('selectedFilesPreview');
    if (preview) {
        preview.classList.add('hidden');
    }
}

function removeSelectedFile(index) {
    const fileInput = document.getElementById('uploadFile');
    if (!fileInput.files.length) return;

    const dt = new DataTransfer();
    const files = Array.from(fileInput.files);

    // Add all files except the one to remove
    files.forEach((file, i) => {
        if (i !== index) {
            dt.items.add(file);
        }
    });

    fileInput.files = dt.files;
    showSelectedFilesPreview();
}

// Upload tab switching function
function switchUploadTab(tabType) {
    console.log('switchUploadTab called with:', tabType);

    const fileUploadTab = document.getElementById('fileUploadTab');
    const urlUploadTab = document.getElementById('urlUploadTab');
    const uploadForm = document.getElementById('uploadForm');
    const urlUploadForm = document.getElementById('urlUploadForm');
    const mediaUrlUploadForm = document.getElementById('mediaUrlUploadForm');

    console.log('Elements found:', {
        fileUploadTab: !!fileUploadTab,
        urlUploadTab: !!urlUploadTab,
        uploadForm: !!uploadForm,
        urlUploadForm: !!urlUploadForm,
        mediaUrlUploadForm: !!mediaUrlUploadForm
    });

    if (!fileUploadTab || !urlUploadTab || !uploadForm || !urlUploadForm || !mediaUrlUploadForm) {
        console.error('Missing required elements for tab switching');
        return;
    }

    // Reset all tabs to inactive state
    [fileUploadTab, urlUploadTab].forEach(tab => {
        tab.classList.add('text-gray-500', 'border-transparent');
        tab.classList.remove('text-blue-600', 'border-blue-600');
    });

    // Hide all forms
    [uploadForm, urlUploadForm, mediaUrlUploadForm].forEach(form => {
        form.classList.add('hidden');
    });

    if (tabType === 'file') {
        // Switch to file upload
        fileUploadTab.classList.add('text-blue-600', 'border-blue-600');
        fileUploadTab.classList.remove('text-gray-500', 'border-transparent');
        uploadForm.classList.remove('hidden');
        console.log('Switched to file upload tab');
    } else if (tabType === 'url') {
        // Switch to video URL upload
        urlUploadTab.classList.add('text-blue-600', 'border-blue-600');
        urlUploadTab.classList.remove('text-gray-500', 'border-transparent');
        urlUploadForm.classList.remove('hidden');
        console.log('Switched to URL upload tab');
    } else if (tabType === 'media-url') {
        // Switch to media URL upload
        mediaUrlUploadForm.classList.remove('hidden');
        console.log('Switched to media URL upload tab');
    }
}

// Video URL upload function
async function handleUrlUploadSubmit(event) {
    event.preventDefault();

    const urlInput = document.getElementById('videoUrl');
    const titleInput = document.getElementById('videoTitle');
    const tagsInput = document.getElementById('urlUploadTags');

    if (!urlInput.value.trim()) {
        showToast('Please enter a video URL', 'warning');
        return;
    }

    setButtonLoading('urlUploadSubmitBtn', 'urlUploadSpinner', true);

    try {
        const requestData = {
            url: urlInput.value.trim(),
            title: titleInput.value.trim(),
            tags: tagsInput.value.trim()
        };

        const config = {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(requestData)
        };

        if (authToken) {
            config.headers['Authorization'] = `Bearer ${authToken}`;
        }

        const response = await fetch(`${API_BASE_URL}/upload-video-url`, config);
        const data = await response.json();

        if (!response.ok || !data.success) {
            throw new Error(data.error || 'Video URL upload failed');
        }

        showToast('Video URL added successfully');
        closeModal('uploadModal');

        // Clear form
        urlInput.value = '';
        titleInput.value = '';
        tagsInput.value = '';

        // Refresh media if we're on media tab
        if (currentTab === 'media') {
            loadMedia();
        }
    } catch (error) {
        console.error('Error uploading video URL:', error);
        showToast('Error adding video URL: ' + error.message, 'error');
    } finally {
        setButtonLoading('urlUploadSubmitBtn', 'urlUploadSpinner', false);
    }
}

// Media URL upload function
async function handleMediaUrlUploadSubmit(event) {
    event.preventDefault();

    const mediaTypeInput = document.getElementById('mediaType');
    const mediaUrlInput = document.getElementById('mediaUrl');
    const mediaTitleInput = document.getElementById('mediaUrlTitle');
    const mediaTagsInput = document.getElementById('mediaUrlTags');

    if (!mediaTypeInput.value) {
        showToast('Please select a media type', 'warning');
        return;
    }

    if (!mediaUrlInput.value.trim()) {
        showToast('Please enter a media URL', 'warning');
        return;
    }

    setButtonLoading('mediaUrlUploadSubmitBtn', 'mediaUrlUploadSpinner', true);

    try {
        const requestData = {
            type: mediaTypeInput.value,
            url: mediaUrlInput.value.trim(),
            title: mediaTitleInput.value.trim(),
            tags: mediaTagsInput.value.trim()
        };

        const config = {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(requestData)
        };

        if (authToken) {
            config.headers['Authorization'] = `Bearer ${authToken}`;
        }

        const response = await fetch(`${API_BASE_URL}/upload-media-url`, config);
        const data = await response.json();

        if (!response.ok || !data.success) {
            throw new Error(data.error || 'Media URL upload failed');
        }

        showToast(`${mediaTypeInput.value === 'image' ? 'Image' : 'Video'} URL added successfully`);
        closeModal('uploadModal');

        // Clear form
        mediaTypeInput.value = '';
        mediaUrlInput.value = '';
        mediaTitleInput.value = '';
        mediaTagsInput.value = '';

        // Refresh media if we're on media tab
        if (currentTab === 'media') {
            loadMedia();
        }
    } catch (error) {
        console.error('Error uploading media URL:', error);
        showToast('Error adding media URL: ' + error.message, 'error');
    } finally {
        setButtonLoading('mediaUrlUploadSubmitBtn', 'mediaUrlUploadSpinner', false);
    }
}

// ---- Live URL preview helpers ----
function buildVideoEmbed(url) {
    try {
        const u = new URL(url);
        const host = u.hostname.replace('www.', '');

        // YouTube
        if (host === 'youtube.com' || host === 'm.youtube.com') {
            const id = u.searchParams.get('v');
            if (id) return { type: 'iframe', src: `https://www.youtube.com/embed/${id}` };
        }
        if (host === 'youtu.be') {
            const id = u.pathname.split('/')[1];
            if (id) return { type: 'iframe', src: `https://www.youtube.com/embed/${id}` };
        }

        // Vimeo
        if (host === 'vimeo.com') {
            const id = u.pathname.split('/').filter(Boolean)[0];
            if (id) return { type: 'iframe', src: `https://player.vimeo.com/video/${id}` };
        }

        // Dailymotion
        if (host === 'dailymotion.com') {
            const parts = u.pathname.split('/');
            const id = parts.find(p => p && p.startsWith('video'))?.replace('video/', '') || parts.pop();
            if (id) return { type: 'iframe', src: `https://www.dailymotion.com/embed/video/${id}` };
        }

        // Direct video files
        if (/\.(mp4|webm|ogg)(\?.*)?$/i.test(url)) {
            return { type: 'video', src: url };
        }
    } catch (e) {
        // ignore parse errors
    }
    // Unknown platform; fallback to iframe
    return { type: 'iframe', src: url };
}

function updateVideoUrlPreview() {
    const input = document.getElementById('videoUrl');
    const preview = document.getElementById('videoUrlPreview');
    const url = (input?.value || '').trim();
    if (!preview) return;

    if (!url) {
        preview.classList.add('hidden');
        preview.innerHTML = '';
        return;
    }

    const info = buildVideoEmbed(url);
    preview.classList.remove('hidden');
    if (info.type === 'video') {
        preview.innerHTML = `<video src="${info.src}" controls playsinline class="w-full max-h-[180px]"></video>`;
    } else {
        preview.innerHTML = `<iframe src="${info.src}" class="w-full" style="height: 180px" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen></iframe>`;
    }
}

function updateMediaUrlPreview() {
    const typeSel = document.getElementById('mediaType');
    const input = document.getElementById('mediaUrl');
    const preview = document.getElementById('mediaUrlPreview');
    const type = typeSel?.value;
    const url = (input?.value || '').trim();
    if (!preview) return;

    if (!type || !url) {
        preview.classList.add('hidden');
        preview.innerHTML = '';
        return;
    }

    if (type === 'image') {
        preview.classList.remove('hidden');
        preview.innerHTML = `<img src="${url}" alt="preview" class="max-h-[180px] object-contain">`;
        return;
    }

    const info = buildVideoEmbed(url);
    preview.classList.remove('hidden');
    if (info.type === 'video') {
        preview.innerHTML = `<video src="${info.src}" controls playsinline class="w-full max-h-[180px]"></video>`;
    } else {
        preview.innerHTML = `<iframe src="${info.src}" class="w-full" style="height: 180px" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen></iframe>`;
    }
}

// Attach listeners when modal is opened or on DOM ready
document.addEventListener('input', (e) => {
    if (e.target && e.target.id === 'videoUrl') updateVideoUrlPreview();
    if (e.target && (e.target.id === 'mediaUrl' || e.target.id === 'mediaType')) updateMediaUrlPreview();
});

document.addEventListener('change', (e) => {
    if (e.target && e.target.id === 'mediaType') updateMediaUrlPreview();
});

// Utility Functions
function escapeHtml(text) {
    if (text === null || text === undefined) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

function escapeJs(text) {
    if (text === null || text === undefined) return '';
    return text.replace(/\\/g, '\\\\')
               .replace(/'/g, "\\'")
               .replace(/"/g, '\\"')
               .replace(/\n/g, '\\n')
               .replace(/\r/g, '\\r')
               .replace(/\t/g, '\\t');
}

function getCurrentBaseUrl() {
    // Get current page's base URL dynamically
    const protocol = window.location.protocol;
    const host = window.location.host;
    const pathname = window.location.pathname;

    // Remove the filename from pathname to get directory
    const pathDir = pathname.substring(0, pathname.lastIndexOf('/'));

    return protocol + '//' + host + pathDir;
}

function makeRelativeUrl(url) {
    if (!url || typeof url !== 'string') return url;

    // Convert absolute URLs to relative paths
    // Handle localhost URLs
    url = url.replace(/^https?:\/\/localhost(:[0-9]+)?\//i, '/');
    // Handle any other absolute URLs that might contain domain names
    url = url.replace(/^https?:\/\/[^\/]+\//i, '/');

    return url;
}

// Validation Functions
function validateForm(formId, submitButtonId) {
    const form = document.getElementById(formId);
    const submitButton = document.getElementById(submitButtonId);
    
    if (!form || !submitButton) return;
    
    const validateFields = () => {
        const requiredFields = form.querySelectorAll('[required]');
        let allValid = true;
        
        requiredFields.forEach(field => {
            if (field.type === 'hidden') {
                // For hidden fields (like media selectors), check if they have values
                allValid = allValid && field.value.trim() !== '';
            } else if (field.type === 'checkbox') {
                // Checkbox fields are valid if checked
                allValid = allValid && field.checked;
            } else {
                // Regular fields - check if they have values
                allValid = allValid && field.value.trim() !== '';
            }
        });
        
        // submitButton.disabled = !allValid; // Commented out to always enable save button
    };
    
    // Initial validation
    validateFields();
    
    // Add event listeners to all form fields
    const allInputs = form.querySelectorAll('input, select, textarea');
    allInputs.forEach(input => {
        input.addEventListener('input', validateFields);
        input.addEventListener('change', validateFields);
    });
    
    return validateFields;
}

// Debounce utility function
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function setupFormValidation() {
    // Setup validation for all forms
    validateForm('categoryForm', 'categorySubmitBtn');
    validateForm('fieldForm', 'fieldSubmitBtn');
    validateForm('recordForm', 'recordSubmitBtn');
    validateForm('uploadForm', 'uploadSubmitBtn');
}

// Setup debounced search inputs
function setupDebouncedSearchInputs() {
    // Categories search
    const categoriesSearchInput = document.getElementById('categoriesSearchInput');
    if (categoriesSearchInput) {
        const debouncedCategorySearch = debounce((value) => {
            searchCategories(value);
        }, 500);

        categoriesSearchInput.addEventListener('input', (e) => {
            debouncedCategorySearch(e.target.value);
        });
    }

    // Records search
    const recordsSearchInput = document.getElementById('recordsSearchInput');
    const recordsSearchType = document.getElementById('recordsSearchType');
    if (recordsSearchInput) {
        const debouncedRecordSearch = debounce(() => {
            searchRecords();
        }, 500);

        recordsSearchInput.addEventListener('input', () => {
            debouncedRecordSearch();
        });

        // Also trigger search when search type changes
        if (recordsSearchType) {
            recordsSearchType.addEventListener('change', () => {
                debouncedRecordSearch();
            });
        }
    }

    // Media search
    const mediaSearchInput = document.getElementById('mediaSearchInput');
    const mediaSearchType = document.getElementById('mediaSearchType');
    if (mediaSearchInput) {
        const debouncedMediaSearch = debounce(() => {
            performMediaSearch();
        }, 500);

        mediaSearchInput.addEventListener('input', () => {
            debouncedMediaSearch();
        });

        // Also trigger search when search type changes
        if (mediaSearchType) {
            mediaSearchType.addEventListener('change', () => {
                debouncedMediaSearch();
            });
        }
    }
}

// Search Functions
function setupSearch() {
    const searchInput = document.getElementById('categoriesSearchInput');
    if (searchInput) {
        let searchTimeout;
        let suggestionTimeout;
        
        searchInput.addEventListener('input', function() {
            const query = this.value.trim();
            
            clearTimeout(searchTimeout);
            clearTimeout(suggestionTimeout);
            
            // Hide suggestions if query is empty
            if (!query) {
                hideCategorySuggestions();
                searchTimeout = setTimeout(() => {
                    loadCategories();
                }, 300);
                return;
            }
            
            // Show suggestions after 150ms of typing
            suggestionTimeout = setTimeout(() => {
                showCategorySuggestions(query);
            }, 150);
            
            // Perform search after 500ms of no typing
            searchTimeout = setTimeout(() => {
                loadCategories();
            }, 500);
        });
        
        searchInput.addEventListener('focus', function() {
            const query = this.value.trim();
            if (query) {
                setTimeout(() => showCategorySuggestions(query), 100);
            }
        });
        
        searchInput.addEventListener('blur', function() {
            // Hide suggestions after a small delay to allow clicks
            setTimeout(() => hideCategorySuggestions(), 200);
        });
        
        searchInput.addEventListener('keydown', function(e) {
            const suggestions = document.getElementById('categoriesSearchSuggestions');
            const activeItem = suggestions.querySelector('.suggestion-active');
            
            if (e.key === 'ArrowDown') {
                e.preventDefault();
                const nextItem = activeItem ? activeItem.nextElementSibling : suggestions.firstElementChild;
                if (nextItem) {
                    if (activeItem) activeItem.classList.remove('suggestion-active');
                    nextItem.classList.add('suggestion-active');
                }
            } else if (e.key === 'ArrowUp') {
                e.preventDefault();
                const prevItem = activeItem ? activeItem.previousElementSibling : suggestions.lastElementChild;
                if (prevItem) {
                    if (activeItem) activeItem.classList.remove('suggestion-active');
                    prevItem.classList.add('suggestion-active');
                }
            } else if (e.key === 'Enter') {
                e.preventDefault();
                if (activeItem) {
                    activeItem.click();
                } else {
                    loadCategories();
                }
            } else if (e.key === 'Escape') {
                hideCategorySuggestions();
            }
        });
    }
}

async function showCategorySuggestions(query) {
    if (!query) return;
    
    try {
        // Get categories that match the query for autocomplete
        const params = new URLSearchParams({ 
            q: query, 
            page: '1', 
            per_page: '10' 
        });
        
        const res = await apiRequest(`/categories?${params.toString()}`);
        let suggestions = [];
        
        if (res && res.items && Array.isArray(res.items)) {
            suggestions = res.items;
        } else if (Array.isArray(res)) {
            suggestions = res;
        }
        
        renderCategorySuggestions(suggestions, query);
    } catch (error) {
        console.error('Error loading search suggestions:', error);
    }
}

function renderCategorySuggestions(suggestions, query) {
    const suggestionsContainer = document.getElementById('categoriesSearchSuggestions');
    if (!suggestionsContainer) return;

    if (suggestions.length === 0) {
        // Show create category option when no suggestions found
        if (query && query.trim()) {
            suggestionsContainer.innerHTML = `
                <div class="px-4 py-3 text-center border-b border-gray-100">
                    <p class="text-sm text-gray-500 mb-2">No categories found for "${escapeHtml(query)}"</p>
                    <button onclick="createCategoryFromSearch('${escapeHtml(query)}')"
                            class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 text-sm transition-colors">
                        Create "${escapeHtml(query)}" category
                    </button>
                </div>
            `;
            suggestionsContainer.classList.remove('hidden');
        } else {
            hideCategorySuggestions();
        }
        return;
    }
    
    const highlightQuery = (text, query) => {
        const regex = new RegExp(`(${query})`, 'gi');
        return text.replace(regex, '<mark class="bg-yellow-200">$1</mark>');
    };
    
    suggestionsContainer.innerHTML = suggestions.map(category => `
        <div class="px-4 py-2 hover:bg-gray-100 cursor-pointer border-b border-gray-100 last:border-b-0 suggestion-item" 
             onclick="selectCategorySuggestion('${escapeHtml(category.name)}')">
            <div class="font-medium text-gray-900">${highlightQuery(escapeHtml(category.name), query)}</div>
            <div class="text-sm text-gray-500">${category.field_count || 0} fields • ${category.record_count || 0} records</div>
        </div>
    `).join('');
    
    suggestionsContainer.classList.remove('hidden');
}

function selectCategorySuggestion(categoryName) {
    const searchInput = document.getElementById('categoriesSearchInput');
    if (searchInput) {
        searchInput.value = categoryName;
        loadCategories();
    }
    hideCategorySuggestions();
}

function hideCategorySuggestions() {
    const suggestionsContainer = document.getElementById('categoriesSearchSuggestions');
    if (suggestionsContainer) {
        suggestionsContainer.classList.add('hidden');
        suggestionsContainer.innerHTML = '';
    }
}

function createCategoryFromSearch(categoryName) {
    // Close suggestions
    hideCategorySuggestions();

    // Pre-fill the category modal with the searched name
    const categoryNameField = document.getElementById('categoryName');
    if (categoryNameField) {
        categoryNameField.value = categoryName;
    }

    // Open the add category modal
    openModal('categoryModal');

    // Clear search input
    const searchInput = document.getElementById('categoriesSearchInput');
    if (searchInput) {
        searchInput.value = '';
    }
}

// Upload Tags Management
let availableTags = [];

async function loadAvailableTags() {
    try {
        const tags = await apiRequest('/tags');
        availableTags = Array.isArray(tags) ? tags : [];
    } catch (error) {
        console.error('Error loading available tags:', error);
        availableTags = [];
    }
}

function setupUploadTagsAutocomplete() {
    const tagsInput = document.getElementById('uploadTags');
    if (!tagsInput) return;

    let suggestionsTimeout;

    tagsInput.addEventListener('input', function() {
        const value = this.value;
        const lastCommaIndex = value.lastIndexOf(',');
        const currentTag = lastCommaIndex !== -1 ? value.substring(lastCommaIndex + 1).trim() : value.trim();

        clearTimeout(suggestionsTimeout);

        if (currentTag.length >= 1) {
            suggestionsTimeout = setTimeout(() => {
                showUploadTagsSuggestions(currentTag, lastCommaIndex);
            }, 200);
        } else {
            hideUploadTagsSuggestions();
        }
    });

    tagsInput.addEventListener('focus', function() {
        const value = this.value;
        const lastCommaIndex = value.lastIndexOf(',');
        const currentTag = lastCommaIndex !== -1 ? value.substring(lastCommaIndex + 1).trim() : value.trim();

        if (currentTag.length >= 1) {
            showUploadTagsSuggestions(currentTag, lastCommaIndex);
        }
    });

    tagsInput.addEventListener('blur', function() {
        setTimeout(() => hideUploadTagsSuggestions(), 200);
    });

    tagsInput.addEventListener('keydown', function(e) {
        const suggestions = document.getElementById('uploadTagsSuggestions');
        const activeItem = suggestions.querySelector('.suggestion-active');

        if (e.key === 'ArrowDown') {
            e.preventDefault();
            const nextItem = activeItem ? activeItem.nextElementSibling : suggestions.firstElementChild;
            if (nextItem) {
                if (activeItem) activeItem.classList.remove('suggestion-active');
                nextItem.classList.add('suggestion-active');
            }
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            const prevItem = activeItem ? activeItem.previousElementSibling : suggestions.lastElementChild;
            if (prevItem) {
                if (activeItem) activeItem.classList.remove('suggestion-active');
                prevItem.classList.add('suggestion-active');
            }
        } else if (e.key === 'Enter' || e.key === 'Tab') {
            if (activeItem) {
                e.preventDefault();
                activeItem.click();
            }
        } else if (e.key === 'Escape') {
            hideUploadTagsSuggestions();
        }
    });
}

function showUploadTagsSuggestions(currentTag, lastCommaIndex) {
    if (!currentTag || availableTags.length === 0) {
        hideUploadTagsSuggestions();
        return;
    }

    const filteredTags = availableTags.filter(tag =>
        tag.toLowerCase().includes(currentTag.toLowerCase()) &&
        tag.toLowerCase() !== currentTag.toLowerCase()
    );

    if (filteredTags.length === 0) {
        hideUploadTagsSuggestions();
        return;
    }

    const suggestions = document.getElementById('uploadTagsSuggestions');
    suggestions.innerHTML = filteredTags.slice(0, 8).map(tag => `
        <div class="px-3 py-2 hover:bg-gray-100 cursor-pointer suggestion-item"
             onclick="selectUploadTag('${escapeHtml(tag)}', ${lastCommaIndex})">
            ${escapeHtml(tag)}
        </div>
    `).join('');

    suggestions.classList.remove('hidden');
}

function selectUploadTag(tag, lastCommaIndex) {
    const tagsInput = document.getElementById('uploadTags');
    if (!tagsInput) return;

    const currentValue = tagsInput.value;
    let newValue;

    if (lastCommaIndex !== -1) {
        // Replace the current tag after the last comma
        newValue = currentValue.substring(0, lastCommaIndex + 1) + ' ' + tag + ', ';
    } else {
        // Replace the entire value
        newValue = tag + ', ';
    }

    tagsInput.value = newValue;
    tagsInput.focus();
    hideUploadTagsSuggestions();
}

function hideUploadTagsSuggestions() {
    const suggestions = document.getElementById('uploadTagsSuggestions');
    if (suggestions) {
        suggestions.classList.add('hidden');
        suggestions.innerHTML = '';
    }
}

// URL Upload Tags Autocomplete Functions
function setupUrlUploadTagsAutocomplete() {
    const tagsInput = document.getElementById('urlUploadTags');
    if (!tagsInput) return;

    let suggestionsTimeout;

    tagsInput.addEventListener('input', function() {
        const value = this.value;
        const lastCommaIndex = value.lastIndexOf(',');
        const currentTag = lastCommaIndex !== -1 ? value.substring(lastCommaIndex + 1).trim() : value.trim();

        clearTimeout(suggestionsTimeout);

        if (currentTag.length >= 1) {
            suggestionsTimeout = setTimeout(() => {
                showUrlUploadTagsSuggestions(currentTag, lastCommaIndex);
            }, 200);
        } else {
            hideUrlUploadTagsSuggestions();
        }
    });

    tagsInput.addEventListener('focus', function() {
        const value = this.value;
        const lastCommaIndex = value.lastIndexOf(',');
        const currentTag = lastCommaIndex !== -1 ? value.substring(lastCommaIndex + 1).trim() : value.trim();

        if (currentTag.length >= 1) {
            showUrlUploadTagsSuggestions(currentTag, lastCommaIndex);
        }
    });

    tagsInput.addEventListener('blur', function() {
        setTimeout(() => hideUrlUploadTagsSuggestions(), 200);
    });

    tagsInput.addEventListener('keydown', function(e) {
        const suggestions = document.getElementById('urlUploadTagsSuggestions');
        const activeItem = suggestions.querySelector('.suggestion-active');

        if (e.key === 'ArrowDown') {
            e.preventDefault();
            const nextItem = activeItem ? activeItem.nextElementSibling : suggestions.firstElementChild;
            if (nextItem) {
                if (activeItem) activeItem.classList.remove('suggestion-active');
                nextItem.classList.add('suggestion-active');
            }
        } else if (e.key === 'ArrowUp') {
            e.preventDefault();
            const prevItem = activeItem ? activeItem.previousElementSibling : suggestions.lastElementChild;
            if (prevItem) {
                if (activeItem) activeItem.classList.remove('suggestion-active');
                prevItem.classList.add('suggestion-active');
            }
        } else if (e.key === 'Enter' && activeItem) {
            e.preventDefault();
            activeItem.click();
        } else if (e.key === 'Escape') {
            hideUrlUploadTagsSuggestions();
        }
    });
}

function showUrlUploadTagsSuggestions(currentTag, lastCommaIndex) {
    if (!currentTag || availableTags.length === 0) {
        hideUrlUploadTagsSuggestions();
        return;
    }

    const filteredTags = availableTags.filter(tag =>
        tag.toLowerCase().includes(currentTag.toLowerCase()) &&
        tag.toLowerCase() !== currentTag.toLowerCase()
    );

    if (filteredTags.length === 0) {
        hideUrlUploadTagsSuggestions();
        return;
    }

    const suggestions = document.getElementById('urlUploadTagsSuggestions');
    suggestions.innerHTML = filteredTags.slice(0, 8).map(tag => `
        <div class="px-3 py-2 hover:bg-gray-100 cursor-pointer suggestion-item"
             onclick="selectUrlUploadTag('${escapeHtml(tag)}', ${lastCommaIndex})">
            ${escapeHtml(tag)}
        </div>
    `).join('');

    suggestions.classList.remove('hidden');
}

function selectUrlUploadTag(tag, lastCommaIndex) {
    const tagsInput = document.getElementById('urlUploadTags');
    if (!tagsInput) return;

    const currentValue = tagsInput.value;

    if (lastCommaIndex !== -1) {
        tagsInput.value = currentValue.substring(0, lastCommaIndex + 1) + ' ' + tag + ', ';
    } else {
        tagsInput.value = tag + ', ';
    }

    hideUrlUploadTagsSuggestions();
    tagsInput.focus();
}

function hideUrlUploadTagsSuggestions() {
    const suggestions = document.getElementById('urlUploadTagsSuggestions');
    if (suggestions) {
        suggestions.classList.add('hidden');
        suggestions.innerHTML = '';
    }
}

// GET Request Tester Functions
let lastResponse = null;

async function testGetRequest(path) {
    if (!path || path.trim() === '') {
        showToast('Please enter a valid path', 'error');
        return;
    }

    const responseOutput = document.getElementById('responseOutput');
    const responseEmpty = document.getElementById('responseEmpty');
    const requestInfo = document.getElementById('requestInfo');
    const requestUrl = document.getElementById('requestUrl');
    const requestStatus = document.getElementById('requestStatus');
    const requestTime = document.getElementById('requestTime');
    const copyBtn = document.getElementById('copyResponseBtn');

    // Show loading state
    responseEmpty.classList.add('hidden');
    responseOutput.classList.remove('hidden');
    responseOutput.textContent = 'Loading...';
    copyBtn.style.display = 'none';

    const startTime = Date.now();

    try {
        const response = await fetch(API_BASE_URL + path, {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${authToken}`
            }
        });

        const endTime = Date.now();
        const duration = endTime - startTime;

        const data = await response.json();
        lastResponse = data;

        // Show request info
        requestInfo.classList.remove('hidden');
        requestUrl.textContent = `GET ${API_BASE_URL}${path}`;
        requestTime.textContent = `${duration}ms`;

        if (response.ok) {
            requestStatus.textContent = `${response.status} OK`;
            requestStatus.className = 'text-sm font-medium text-green-600';
        } else {
            requestStatus.textContent = `${response.status} Error`;
            requestStatus.className = 'text-sm font-medium text-red-600';
        }

        // Display response
        responseOutput.textContent = JSON.stringify(data, null, 2);
        copyBtn.style.display = 'block';

    } catch (error) {
        const endTime = Date.now();
        const duration = endTime - startTime;

        requestInfo.classList.remove('hidden');
        requestUrl.textContent = `GET ${API_BASE_URL}${path}`;
        requestTime.textContent = `${duration}ms`;
        requestStatus.textContent = 'Network Error';
        requestStatus.className = 'text-sm font-medium text-red-600';

        responseOutput.textContent = JSON.stringify({
            error: error.message,
            type: 'NetworkError'
        }, null, 2);

        lastResponse = null;
        copyBtn.style.display = 'block';
    }
}

function copyResponse() {
    if (!lastResponse) {
        showToast('No response to copy', 'error');
        return;
    }

    const responseText = JSON.stringify(lastResponse, null, 2);

    navigator.clipboard.writeText(responseText).then(() => {
        showToast('Response copied to clipboard', 'success');
    }).catch(err => {
        console.error('Failed to copy:', err);
        showToast('Failed to copy response', 'error');
    });
}

// Database Schema Viewer
async function loadDatabaseSchema() {
    const canvas = document.getElementById('schemaCanvas');
    if (!canvas) return;

    try {
        console.log('Loading database schema...');
        canvas.innerHTML = '<div class="text-center py-12 text-gray-500">Loading schema...</div>';

        // Get all tables schema information
        const tables = await getTableSchema();

        console.log('Schema loaded:', tables);
        console.log('Number of tables:', tables ? tables.length : 0);

        if (!tables || tables.length === 0) {
            canvas.innerHTML = '<div class="text-center py-12 text-yellow-500">No tables found in database</div>';
            return;
        }

        canvas.innerHTML = '';
        renderSchema(canvas, tables);
    } catch (error) {
        console.error('Error loading schema:', error);
        console.error('Error stack:', error.stack);
        canvas.innerHTML = `<div class="text-center py-12 text-red-500">
            <p class="font-bold mb-2">Failed to load database schema</p>
            <p class="text-sm">${error.message || 'Unknown error'}</p>
        </div>`;
    }
}

async function getTableSchema() {
    try {
        console.log('Calling apiRequest for /schema...');
        const response = await apiRequest('/schema');
        console.log('API response received:', response);

        // Handle the response format
        if (response && typeof response === 'object') {
            // If response has a data property, use it
            if (response.data && Array.isArray(response.data)) {
                return response.data;
            }
            // If response is directly an array
            if (Array.isArray(response)) {
                return response;
            }
        }

        console.warn('Unexpected response format:', response);
        return [];
    } catch (error) {
        console.error('Error fetching schema:', error);
        console.error('Error details:', {
            message: error.message,
            name: error.name,
            stack: error.stack
        });
        showToast('Failed to load database schema: ' + error.message, 'error');
        throw error; // Re-throw to be caught by loadDatabaseSchema
    }
}

function renderSchema(canvas, tables) {
    const padding = 40;
    const tableWidth = 280;
    const tableHeight = 50; // Base height
    const rowHeight = 32;
    const colGap = 100;
    const rowGap = 60;

    // Load saved positions or calculate initial positions
    const savedPositions = JSON.parse(localStorage.getItem('schemaTablePositions') || '{}');

    tables.forEach((table, index) => {
        const tableKey = table.name;
        if (savedPositions[tableKey]) {
            table.x = savedPositions[tableKey].x;
            table.y = savedPositions[tableKey].y;
        } else {
            const col = index % 3;
            const row = Math.floor(index / 3);
            table.x = padding + col * (tableWidth + colGap);
            table.y = padding + row * (tableHeight + table.columns.length * rowHeight + rowGap);
        }
    });

    // Set canvas size
    const maxX = Math.max(...tables.map(t => t.x)) + tableWidth + padding;
    const maxY = Math.max(...tables.map(t => t.y + tableHeight + t.columns.length * rowHeight)) + padding;
    canvas.style.width = Math.max(maxX, 1200) + 'px';
    canvas.style.height = Math.max(maxY, 800) + 'px';

    // Create SVG layer for relationships
    const svgLayer = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
    svgLayer.id = 'relationshipLayer';
    svgLayer.style.position = 'absolute';
    svgLayer.style.top = '0';
    svgLayer.style.left = '0';
    svgLayer.style.width = '100%';
    svgLayer.style.height = '100%';
    svgLayer.style.pointerEvents = 'none';
    svgLayer.style.zIndex = '0';
    canvas.appendChild(svgLayer);

    // Function to draw relationships
    function drawRelationships() {
        svgLayer.innerHTML = '';
        const relationships = [];

        tables.forEach(table => {
            table.columns.forEach(col => {
                if (col.fk) {
                    // Match table name with or without " (virtual)" suffix
                    const targetTableName = col.fk.table;
                    const targetTable = tables.find(t => {
                        const cleanName = t.name.replace(' (virtual)', '');
                        return cleanName === targetTableName || t.name === targetTableName;
                    });

                    if (targetTable) {
                        console.log('Found relationship:', table.name, '->', targetTable.name);
                        relationships.push({
                            from: table,
                            to: targetTable,
                            fromCol: col.name,
                            toCol: col.fk.column
                        });
                    } else {
                        console.warn('Target table not found for FK:', col.fk.table, 'from', table.name);
                    }
                }
            });
        });

        console.log('Total relationships found:', relationships.length);

        // Draw relationships
        relationships.forEach(rel => {
            const fromX = rel.from.x + tableWidth;
            const fromY = rel.from.y + 40;
            const toX = rel.to.x;
            const toY = rel.to.y + 40;

            // Create curved line path
            const midX = (fromX + toX) / 2;
            const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
            path.setAttribute('d', `M ${fromX} ${fromY} Q ${midX} ${fromY}, ${midX} ${(fromY + toY) / 2} Q ${midX} ${toY}, ${toX} ${toY}`);
            path.setAttribute('stroke', '#94a3b8');
            path.setAttribute('stroke-width', '2');
            path.setAttribute('fill', 'none');
            path.setAttribute('stroke-dasharray', '5,5');

            const circle = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
            circle.setAttribute('cx', toX);
            circle.setAttribute('cy', toY);
            circle.setAttribute('r', '4');
            circle.setAttribute('fill', '#3b82f6');

            svgLayer.appendChild(path);
            svgLayer.appendChild(circle);
        });
    }

    drawRelationships();

    // Draw tables
    tables.forEach(table => {
        const tableEl = document.createElement('div');
        const isVirtual = table.virtual === true;
        tableEl.className = `absolute bg-white border-2 ${isVirtual ? 'border-green-400' : 'border-gray-300'} rounded-lg shadow-lg overflow-hidden`;
        tableEl.style.left = table.x + 'px';
        tableEl.style.top = table.y + 'px';
        tableEl.style.width = tableWidth + 'px';
        tableEl.style.zIndex = '10';

        // Table header
        const header = document.createElement('div');
        const headerColor = isVirtual ? 'from-green-600 to-green-500' : 'from-blue-600 to-blue-500';
        header.className = `bg-gradient-to-r ${headerColor} text-white px-4 py-3 font-bold text-sm flex items-center`;
        const icon = isVirtual
            ? '<svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"></path></svg>'
            : '<svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"></path></svg>';
        header.innerHTML = `${icon}${table.name}`;
        tableEl.appendChild(header);

        // Table columns
        table.columns.forEach(col => {
            const colEl = document.createElement('div');
            colEl.className = 'px-4 py-2 text-xs border-b border-gray-200 hover:bg-blue-50 transition-colors flex items-center justify-between';

            const colInfo = document.createElement('div');
            colInfo.className = 'flex items-center';

            // Primary key icon
            if (col.pk) {
                colInfo.innerHTML += '<svg class="w-3 h-3 mr-1 text-yellow-500" fill="currentColor" viewBox="0 0 20 20"><path d="M10 2a5 5 0 00-5 5v2a2 2 0 00-2 2v5a2 2 0 002 2h10a2 2 0 002-2v-5a2 2 0 00-2-2H7V7a3 3 0 015.905-.75 1 1 0 001.937-.5A5.002 5.002 0 0010 2z"></path></svg>';
            }
            // Foreign key icon
            else if (col.fk) {
                colInfo.innerHTML += '<svg class="w-3 h-3 mr-1 text-blue-500" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M12.586 4.586a2 2 0 112.828 2.828l-3 3a2 2 0 01-2.828 0 1 1 0 00-1.414 1.414 4 4 0 005.656 0l3-3a4 4 0 00-5.656-5.656l-1.5 1.5a1 1 0 101.414 1.414l1.5-1.5zm-5 5a2 2 0 012.828 0 1 1 0 101.414-1.414 4 4 0 00-5.656 0l-3 3a4 4 0 105.656 5.656l1.5-1.5a1 1 0 10-1.414-1.414l-1.5 1.5a2 2 0 11-2.828-2.828l3-3z" clip-rule="evenodd"></path></svg>';
            }

            colInfo.innerHTML += `<span class="font-medium text-gray-700">${col.name}</span>`;
            colEl.appendChild(colInfo);

            // Data type
            const typeEl = document.createElement('span');
            typeEl.className = 'text-gray-500 text-xs';
            typeEl.textContent = col.type;
            colEl.appendChild(typeEl);

            tableEl.appendChild(colEl);
        });

        // Add drag and drop functionality
        let isDragging = false;
        let currentX;
        let currentY;
        let initialX;
        let initialY;

        header.style.cursor = 'move';
        header.addEventListener('mousedown', dragStart);

        function dragStart(e) {
            initialX = e.clientX - table.x;
            initialY = e.clientY - table.y;

            if (e.target === header || header.contains(e.target)) {
                isDragging = true;
                tableEl.style.zIndex = '100';
            }
        }

        document.addEventListener('mousemove', drag);
        document.addEventListener('mouseup', dragEnd);

        function drag(e) {
            if (isDragging) {
                e.preventDefault();
                currentX = e.clientX - initialX;
                currentY = e.clientY - initialY;

                table.x = currentX;
                table.y = currentY;

                tableEl.style.left = currentX + 'px';
                tableEl.style.top = currentY + 'px';

                // Redraw relationships
                drawRelationships();
            }
        }

        function dragEnd(e) {
            if (isDragging) {
                isDragging = false;
                tableEl.style.zIndex = '10';

                // Save position
                const positions = JSON.parse(localStorage.getItem('schemaTablePositions') || '{}');
                positions[table.name] = { x: table.x, y: table.y };
                localStorage.setItem('schemaTablePositions', JSON.stringify(positions));
            }
        }

        canvas.appendChild(tableEl);
    });

    // Add reset positions button
    const resetBtn = document.createElement('button');
    resetBtn.className = 'fixed bottom-6 left-6 bg-red-500 text-white px-4 py-2 rounded-lg shadow-xl hover:bg-red-600 transition-colors text-sm z-20';
    resetBtn.innerHTML = '🔄 Reset Positions';
    resetBtn.onclick = () => {
        if (confirm('Reset all table positions to default layout?')) {
            localStorage.removeItem('schemaTablePositions');
            loadDatabaseSchema();
        }
    };
    canvas.appendChild(resetBtn);

    // Add legend
    const legend = document.createElement('div');
    legend.className = 'fixed bottom-6 right-6 bg-white border-2 border-gray-300 rounded-lg shadow-xl p-4 text-xs z-20';
    legend.innerHTML = `
        <div class="font-bold text-gray-700 mb-2">Legend</div>
        <div class="space-y-1">
            <div class="flex items-center">
                <div class="w-4 h-3 bg-gradient-to-r from-blue-600 to-blue-500 rounded mr-2"></div>
                <span class="text-gray-600">Core Table</span>
            </div>
            <div class="flex items-center">
                <div class="w-4 h-3 bg-gradient-to-r from-green-600 to-green-500 rounded mr-2"></div>
                <span class="text-gray-600">Category Table</span>
            </div>
            <div class="flex items-center">
                <svg class="w-3 h-3 mr-2 text-yellow-500" fill="currentColor" viewBox="0 0 20 20"><path d="M10 2a5 5 0 00-5 5v2a2 2 0 00-2 2v5a2 2 0 002 2h10a2 2 0 002-2v-5a2 2 0 00-2-2H7V7a3 3 0 015.905-.75 1 1 0 001.937-.5A5.002 5.002 0 0010 2z"></path></svg>
                <span class="text-gray-600">Primary Key</span>
            </div>
            <div class="flex items-center">
                <svg class="w-3 h-3 mr-2 text-blue-500" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M12.586 4.586a2 2 0 112.828 2.828l-3 3a2 2 0 01-2.828 0 1 1 0 00-1.414 1.414 4 4 0 005.656 0l3-3a4 4 0 00-5.656-5.656l-1.5 1.5a1 1 0 101.414 1.414l1.5-1.5zm-5 5a2 2 0 012.828 0 1 1 0 101.414-1.414 4 4 0 00-5.656 0l-3 3a4 4 0 105.656 5.656l1.5-1.5a1 1 0 10-1.414-1.414l-1.5 1.5a2 2 0 11-2.828-2.828l3-3z" clip-rule="evenodd"></path></svg>
                <span class="text-gray-600">Foreign Key</span>
            </div>
            <div class="flex items-center">
                <svg class="w-4 h-2 mr-2" viewBox="0 0 20 2"><line x1="0" y1="1" x2="20" y2="1" stroke="#94a3b8" stroke-width="2" stroke-dasharray="3,3"/></svg>
                <span class="text-gray-600">Relationship</span>
            </div>
        </div>
    `;
    canvas.appendChild(legend);
}

// Language Management Functions
let languages = [];
let editingLanguage = null;

function openLanguageModal() {
    editingLanguage = null;
    document.getElementById('languageModalTitle').textContent = 'Add Language';
    document.getElementById('languageCode').value = '';
    document.getElementById('languageCode').disabled = false;
    document.getElementById('languageName').value = '';
    document.getElementById('languageTextDirection').value = 'ltr';
    document.getElementById('languageIsActive').checked = true;
    document.getElementById('languageIsDefault').checked = false;
    document.getElementById('languageButtonText').textContent = 'Save';
    openModal('languageModal');
}

async function loadLanguages() {
    try {
        languages = await apiRequest('/languages');
        renderLanguages();
    } catch (error) {
        console.error('Error loading languages:', error);
        showToast('Failed to load languages: ' + error.message, 'error');
    }
}

function renderLanguages() {
    const list = document.getElementById('languagesList');
    if (!list) return;

    if (languages.length === 0) {
        list.innerHTML = `
            <tr>
                <td colspan="7" class="px-6 py-8 text-center text-gray-500">
                    No languages configured. Click "Add Language" to create one.
                </td>
            </tr>
        `;
        return;
    }

    list.innerHTML = languages.map(lang => `
        <tr>
            <td class="px-6 py-4">
                <span class="font-mono text-sm font-medium text-gray-900">${escapeHtml(lang.code)}</span>
            </td>
            <td class="px-6 py-4 text-gray-900">${escapeHtml(lang.name)}</td>
            <td class="px-6 py-4">
                <span class="px-2 py-1 text-xs font-semibold rounded-full ${lang.text_direction === 'rtl' ? 'bg-purple-100 text-purple-800' : 'bg-blue-100 text-blue-800'}">
                    ${lang.text_direction ? lang.text_direction.toUpperCase() : 'LTR'}
                </span>
            </td>
            <td class="px-6 py-4">
                ${lang.is_active ?
                    '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-green-100 text-green-800">Active</span>' :
                    '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-gray-100 text-gray-600">Inactive</span>'
                }
            </td>
            <td class="px-6 py-4">
                ${lang.is_default ?
                    '<span class="px-2 py-1 text-xs font-semibold rounded-full bg-yellow-100 text-yellow-800">Default</span>' :
                    '<span class="text-gray-400">-</span>'
                }
            </td>
            <td class="px-6 py-4 text-sm text-gray-500">
                ${lang.created_at ? new Date(lang.created_at).toLocaleDateString() : '-'}
            </td>
            <td class="px-6 py-4">
                <button onclick="editLanguage(${lang.id})" class="text-blue-600 hover:text-blue-900 mr-3">
                    <svg class="w-5 h-5 inline" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                    </svg>
                </button>
                <button onclick="deleteLanguageConfirm(${lang.id}, '${escapeHtml(lang.name)}')" class="text-red-600 hover:text-red-900">
                    <svg class="w-5 h-5 inline" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                    </svg>
                </button>
            </td>
        </tr>
    `).join('');
}

function editLanguage(languageId) {
    const language = languages.find(l => l.id === languageId);
    if (!language) return;

    editingLanguage = languageId;
    document.getElementById('languageModalTitle').textContent = 'Edit Language';
    document.getElementById('languageCode').value = language.code;
    document.getElementById('languageCode').disabled = true; // Don't allow changing code
    document.getElementById('languageName').value = language.name;
    document.getElementById('languageTextDirection').value = language.text_direction || 'ltr';
    document.getElementById('languageIsActive').checked = language.is_active == 1;
    document.getElementById('languageIsDefault').checked = language.is_default == 1;
    document.getElementById('languageButtonText').textContent = 'Update';
    openModal('languageModal');
}

async function handleLanguageSubmit(event) {
    event.preventDefault();

    const code = document.getElementById('languageCode').value.toLowerCase().trim();
    const name = document.getElementById('languageName').value.trim();
    const textDirection = document.getElementById('languageTextDirection').value;
    const isActive = document.getElementById('languageIsActive').checked;
    const isDefault = document.getElementById('languageIsDefault').checked;

    const data = {
        code,
        name,
        text_direction: textDirection,
        is_active: isActive,
        is_default: isDefault
    };

    try {
        setButtonLoading('languageForm', 'languageSpinner', true);

        if (editingLanguage) {
            await apiRequest(`/languages/${editingLanguage}`, {
                method: 'PUT',
                body: JSON.stringify(data)
            });
            showToast('Language updated successfully');
        } else {
            await apiRequest('/languages', {
                method: 'POST',
                body: JSON.stringify(data)
            });
            showToast('Language added successfully');
        }

        closeModal('languageModal');
        loadLanguages();
    } catch (error) {
        console.error('Error saving language:', error);
        showToast('Error: ' + error.message, 'error');
    } finally {
        setButtonLoading('languageForm', 'languageSpinner', false);
    }
}

async function deleteLanguageConfirm(languageId, languageName) {
    if (!confirm(`Are you sure you want to delete "${languageName}"? All translations in this language will also be deleted.`)) {
        return;
    }

    try {
        showLoading();
        await apiRequest(`/languages/${languageId}`, { method: 'DELETE' });
        showToast('Language deleted successfully');
        loadLanguages();
    } catch (error) {
        console.error('Error deleting language:', error);
        showToast('Error: ' + error.message, 'error');
    } finally {
        hideLoading();
    }
}

// Translation Functions
let currentTranslationRecordId = null;
let currentTranslationData = {};
let translatableFields = [];
let currentOriginalValues = {};
let currentLanguageCode = '';

async function openTranslationsModal() {
    if (!editingRecord) {
        showToast('Please save the record first before managing translations', 'warning');
        return;
    }

    currentTranslationRecordId = editingRecord;

    // Get category ID from the select element
    const categoryId = document.getElementById('recordsCategorySelect').value;
    if (!categoryId) {
        showToast('Please select a category first', 'warning');
        return;
    }

    // Get translatable fields (text, textarea, rich_paragraph)
    const fieldsResponse = await apiRequest(`/fields?category_id=${categoryId}`);
    const allFields = Array.isArray(fieldsResponse) ? fieldsResponse : (fieldsResponse.items || []);
    translatableFields = allFields.filter(f => ['text', 'textarea', 'rich_paragraph'].includes(f.data_type));

    if (translatableFields.length === 0) {
        showToast('No translatable text fields found in this category', 'warning');
        return;
    }

    // Get original values from the record form
    currentOriginalValues = {};
    translatableFields.forEach(field => {
        const fieldId = `record_${field.name.replace(/\s+/g, '_')}`;
        const input = document.getElementById(fieldId);
        if (input) {
            currentOriginalValues[field.name] = input.value;
        }
    });

    // Load existing translations
    try {
        const translations = await apiRequest(`/records/${currentTranslationRecordId}/translations`);
        currentTranslationData = translations || {};
    } catch (error) {
        console.error('Error loading translations:', error);
        currentTranslationData = {};
    }

    // Populate language dropdown with active languages
    await populateLanguageDropdown();

    // Update saved languages buttons
    await updateSavedLanguagesButtons();

    // Reset form
    document.getElementById('translationLanguageSelect').value = '';
    document.getElementById('translationFieldsContainer').classList.add('hidden');
    document.getElementById('saveTranslationsBtn').classList.add('hidden');
    document.getElementById('deleteTranslationBtn').classList.add('hidden');
    document.getElementById('defaultLanguageCheckbox').classList.add('hidden');
    document.getElementById('translationIsDefault').checked = false;

    openModal('translationModal');
}

async function populateLanguageDropdown() {
    try {
        const languagesResponse = await apiRequest('/languages');
        const languages = Array.isArray(languagesResponse) ? languagesResponse : [];

        // Filter to only active languages
        const activeLanguages = languages.filter(lang => {
            const isActive = lang.is_active === 1 || lang.is_active === true || lang.is_active === '1';
            return isActive;
        });

        const dropdown = document.getElementById('translationLanguageSelect');

        // Clear existing options except the first placeholder
        dropdown.innerHTML = '<option value="">-- Select a language --</option>';

        // Add languages to dropdown
        activeLanguages.forEach(lang => {
            const option = document.createElement('option');
            option.value = lang.code;
            option.textContent = `${lang.name} (${lang.code})`;
            option.dataset.languageId = lang.id;
            option.dataset.languageName = lang.name;
            option.dataset.textDirection = lang.text_direction || 'ltr';
            dropdown.appendChild(option);
        });

        console.log(`[Translation] Populated dropdown with ${activeLanguages.length} active languages`);
    } catch (error) {
        console.error('Error loading languages:', error);
        showToast('Error loading languages: ' + error.message, 'error');
    }
}

async function updateSavedLanguagesButtons() {
    const savedLanguages = Object.keys(currentTranslationData);
    const container = document.getElementById('savedLanguagesContainer');
    const buttonsContainer = document.getElementById('savedLanguagesButtons');

    if (savedLanguages.length > 0) {
        // Get language names from the dropdown options
        const dropdown = document.getElementById('translationLanguageSelect');
        const languageMap = {};

        Array.from(dropdown.options).forEach(option => {
            if (option.value) {
                languageMap[option.value] = option.dataset.languageName || option.value;
            }
        });

        container.classList.remove('hidden');
        buttonsContainer.innerHTML = savedLanguages.map(langCode => {
            const langName = languageMap[langCode] || langCode;
            return `
                <button type="button" onclick="loadLanguageTranslation('${langCode}')"
                        class="px-3 py-1 bg-blue-100 text-blue-700 rounded-md hover:bg-blue-200 font-medium text-sm">
                    ${escapeHtml(langName)} (${escapeHtml(langCode)})
                </button>
            `;
        }).join('');
    } else {
        container.classList.add('hidden');
    }
}

async function handleLanguageSelection() {
    const dropdown = document.getElementById('translationLanguageSelect');
    const languageCode = dropdown.value;

    if (!languageCode) {
        document.getElementById('translationFieldsContainer').classList.add('hidden');
        document.getElementById('saveTranslationsBtn').classList.add('hidden');
        document.getElementById('deleteTranslationBtn').classList.add('hidden');
        document.getElementById('defaultLanguageCheckbox').classList.add('hidden');
        return;
    }

    loadLanguageTranslation(languageCode);
}

async function loadLanguageTranslation(languageCode) {
    currentLanguageCode = languageCode;

    // Set the dropdown to the selected language
    const dropdown = document.getElementById('translationLanguageSelect');
    dropdown.value = languageCode;

    // Get language name from dropdown option
    const selectedOption = dropdown.querySelector(`option[value="${languageCode}"]`);
    const languageName = selectedOption ? selectedOption.dataset.languageName : languageCode;

    document.getElementById('currentLanguageLabel').textContent = `${languageName} - ${languageCode}`;

    // Clean up any existing Rich Paragraph editors before rendering new ones
    translatableFields.forEach(field => {
        if (field.data_type === 'rich_paragraph') {
            const fieldId = `trans_${field.name.replace(/\s+/g, '_')}`;
            const editor = RTE.get(fieldId);
            if (editor) {
                console.log(`[Translation-RTE] Removing editor for ${fieldId}`);
                RTE.remove(fieldId);
            }
        }
    });

    const existingTranslations = currentTranslationData[languageCode]?.fields || {};

    // Check if this language exists and get its default status
    try {
        const languagesResponse = await apiRequest('/languages');
        const language = languagesResponse.find(l => l.code === languageCode);

        // Show and set default checkbox
        const defaultCheckbox = document.getElementById('defaultLanguageCheckbox');
        const isDefaultCheckbox = document.getElementById('translationIsDefault');
        defaultCheckbox.classList.remove('hidden');

        if (language) {
            isDefaultCheckbox.checked = language.is_default == 1;
        } else {
            isDefaultCheckbox.checked = false;
        }
    } catch (error) {
        console.error('Error loading language info:', error);
    }

    // Show original fields
    const originalFields = document.getElementById('originalFields');
    originalFields.innerHTML = translatableFields.map(field => {
        const originalValue = currentOriginalValues[field.name] || '';

        if (field.data_type === 'textarea' || field.data_type === 'rich_paragraph') {
            return `
                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700">${escapeHtml(field.name)}</label>
                    <div class="w-full px-3 py-2 bg-gray-50 border border-gray-200 rounded-md text-gray-600 whitespace-pre-wrap min-h-[6rem] max-h-[12rem] overflow-y-auto">
                        ${escapeHtml(originalValue) || '<span class="text-gray-400 italic">No content</span>'}
                    </div>
                </div>
            `;
        } else {
            return `
                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700">${escapeHtml(field.name)}</label>
                    <div class="w-full px-3 py-2 bg-gray-50 border border-gray-200 rounded-md text-gray-600">
                        ${escapeHtml(originalValue) || '<span class="text-gray-400 italic">No content</span>'}
                    </div>
                </div>
            `;
        }
    }).join('');

    // Show translation fields
    const translationFields = document.getElementById('translationFields');
    translationFields.innerHTML = translatableFields.map(field => {
        const currentValue = existingTranslations[field.name] || '';
        const fieldId = `trans_${field.name.replace(/\s+/g, '_')}`;

        if (field.data_type === 'rich_paragraph') {
            // Get character limit from field options
            const fieldOptions = field.field_options ? (typeof field.field_options === 'string' ? JSON.parse(field.field_options) : field.field_options) : {};
            const maxChars = fieldOptions.max_chars || null;
            const charCounterId = `${fieldId}_char_count`;

            return `
                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700">${escapeHtml(field.name)}</label>
                    <textarea id="${fieldId}"
                              class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 tinymce-editor"
                              rows="10"
                              ${field.is_required ? 'required' : ''}
                              data-max-chars="${maxChars || ''}">${escapeHtml(currentValue)}</textarea>
                    ${maxChars ? `<div id="${charCounterId}" class="text-xs text-right font-medium text-gray-600"><span class="char-count">0</span> / ${maxChars} characters</div>` : ''}
                    <div class="text-xs text-gray-500">
                        Rich text editor with formatting support${maxChars ? `. Maximum ${maxChars} characters.` : ''}
                    </div>
                </div>
            `;
        } else if (field.data_type === 'textarea') {
            return `
                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700">${escapeHtml(field.name)}</label>
                    <textarea id="${fieldId}" rows="6"
                              class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 resize-y"
                              ${field.is_required ? 'required' : ''}
                              placeholder="Enter translation...">${escapeHtml(currentValue)}</textarea>
                </div>
            `;
        } else {
            // text field
            return `
                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700">${escapeHtml(field.name)}</label>
                    <input type="text" id="${fieldId}"
                           value="${escapeHtml(currentValue)}"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                           ${field.is_required ? 'required' : ''}
                           placeholder="Enter translation...">
                </div>
            `;
        }
    }).join('');

    document.getElementById('translationFieldsContainer').classList.remove('hidden');
    document.getElementById('saveTranslationsBtn').classList.remove('hidden');

    // Show delete button only if translation exists
    if (currentTranslationData[languageCode]) {
        document.getElementById('deleteTranslationBtn').classList.remove('hidden');
    } else {
        document.getElementById('deleteTranslationBtn').classList.add('hidden');
    }

    // Initialize Rich Paragraph editors for translation fields
    setTimeout(() => {
        translatableFields.forEach(field => {
            if (field.data_type === 'rich_paragraph') {
                const fieldId = `trans_${field.name.replace(/\s+/g, '_')}`;
                const fieldOptions = field.field_options ? (typeof field.field_options === 'string' ? JSON.parse(field.field_options) : field.field_options) : {};
                const maxChars = fieldOptions.max_chars || null;
                const charCounterId = `${fieldId}_char_count`;

                console.log(`[Translation-RTE] Initializing RTE for ${fieldId} with maxChars: ${maxChars}`);

                // Initialize the editor
                initializeTinyMCE(fieldId);

                // Set up character counter if limit is defined
                if (maxChars && maxChars > 0) {
                    setTimeout(() => {
                        setupCharacterCounter(fieldId, maxChars, charCounterId);
                    }, 1500);
                }
            }
        });
    }, 300);
}

async function handleSaveTranslations() {
    const languageCode = currentLanguageCode;
    if (!languageCode) {
        showToast('Please enter a language code', 'warning');
        return;
    }

    // Get is_default checkbox value
    const isDefault = document.getElementById('translationIsDefault').checked;

    // Validate rich paragraph character limits before saving
    for (const field of translatableFields) {
        if (field.data_type === 'rich_paragraph') {
            const fieldOptions = field.field_options ? (typeof field.field_options === 'string' ? JSON.parse(field.field_options) : field.field_options) : {};
            const maxChars = fieldOptions.max_chars;

            if (maxChars) {
                const fieldId = `trans_${field.name.replace(/\s+/g, '_')}`;
                const editor = RTE.get(fieldId);

                if (editor) {
                    const content = editor.getContent();
                    const currentLength = content.length;

                    if (currentLength > maxChars) {
                        showToast(`Translation field "${field.name}" exceeds maximum character limit (${currentLength}/${maxChars} characters). Please reduce the content.`, 'error');
                        return;
                    }
                }
            }
        }
    }

    // Collect translation values
    const translations = {};
    translatableFields.forEach(field => {
        const fieldId = `trans_${field.name.replace(/\s+/g, '_')}`;

        if (field.data_type === 'rich_paragraph') {
            // Get value from TinyMCE editor
            const editor = RTE.get(fieldId);
            if (editor) {
                translations[field.name] = editor.getContent();
            }
        } else {
            // Get value from regular input/textarea
            const input = document.getElementById(fieldId);
            if (input) {
                translations[field.name] = input.value;
            }
        }
    });

    try {
        const saveBtn = document.getElementById('saveTranslationsBtn');
        const spinner = document.getElementById('translationSpinner');
        saveBtn.disabled = true;
        spinner.classList.remove('hidden');

        // Save translations
        await apiRequest(`/records/${currentTranslationRecordId}/translations`, {
            method: 'POST',
            body: JSON.stringify({
                language_code: languageCode,
                translations: translations,
                is_default: isDefault
            })
        });

        showToast(`Translation saved successfully for "${languageCode}"${isDefault ? ' (set as default)' : ''}`);

        // Reload translations
        const updated = await apiRequest(`/records/${currentTranslationRecordId}/translations`);
        currentTranslationData = updated || {};

        // Update saved languages buttons
        await updateSavedLanguagesButtons();

        // Show delete button now
        document.getElementById('deleteTranslationBtn').classList.remove('hidden');

    } catch (error) {
        console.error('Error saving translations:', error);
        showToast('Error: ' + error.message, 'error');
    } finally {
        const saveBtn = document.getElementById('saveTranslationsBtn');
        const spinner = document.getElementById('translationSpinner');
        saveBtn.disabled = false;
        spinner.classList.add('hidden');
    }
}

async function deleteCurrentTranslation() {
    const languageCode = currentLanguageCode;
    if (!languageCode) return;

    if (!confirm(`Are you sure you want to delete all "${languageCode}" translations for this record?`)) {
        return;
    }

    try {
        showLoading();

        // Get language ID
        const languagesResponse = await apiRequest('/languages');
        const language = languagesResponse.find(l => l.code === languageCode);

        if (language) {
            // Delete translations by deleting the language (will cascade)
            // Or we can add a specific endpoint for this
            // For now, we'll just clear the translation data and re-save empty
            await apiRequest(`/records/${currentTranslationRecordId}/translations`, {
                method: 'POST',
                body: JSON.stringify({
                    language_code: languageCode,
                    translations: {} // Empty translations
                })
            });
        }

        showToast(`Translation deleted for "${languageCode}"`);

        // Reload translations
        const updated = await apiRequest(`/records/${currentTranslationRecordId}/translations`);
        currentTranslationData = updated || {};

        // Update UI
        await updateSavedLanguagesButtons();
        document.getElementById('translationLanguageSelect').value = '';
        document.getElementById('translationFieldsContainer').classList.add('hidden');
        document.getElementById('saveTranslationsBtn').classList.add('hidden');
        document.getElementById('deleteTranslationBtn').classList.add('hidden');

    } catch (error) {
        console.error('Error deleting translation:', error);
        showToast('Error: ' + error.message, 'error');
    } finally {
        hideLoading();
    }
}

// Initialize App
function initializeApp() {
    // Setup form validation on page load
    setupFormValidation();

    // Check for stored authentication
    const storedToken = localStorage.getItem('authToken');
    const storedUser = localStorage.getItem('currentUser');

    if (storedToken && storedUser) {
        try {
            authToken = storedToken;
            currentUser = JSON.parse(storedUser);
            showMainApp();
        } catch (error) {
            console.error('Error parsing stored user data:', error);
            localStorage.removeItem('authToken');
            localStorage.removeItem('currentUser');
            showAuthSection();
        }
    } else {
        showAuthSection();
    }

    // Setup search functionality
    setupSearch();

    // Setup debounced search inputs
    setupDebouncedSearchInputs();
}

// Initialize the app when DOM is ready
document.addEventListener('DOMContentLoaded', initializeApp);
</script>
</body>
</html>
<?php
}
