// MaxMind AI Assistant
class MaxMindAssistant {
    constructor() {
        this.isOpen = false;
        this.isTyping = false;
        this.responses = this.initializeResponses();
        this.init();
    }

    init() {
        this.bindEvents();
        this.setupKeyboardShortcuts();
        console.log('🤖 MaxMind AI Assistant initialized');
    }

    bindEvents() {
        // Chat toggle button
        const chatToggle = document.getElementById('chat-toggle');
        const chatMinimize = document.getElementById('chat-minimize');
        const chatSend = document.getElementById('chat-send');
        const chatInput = document.getElementById('chat-input');

        if (chatToggle) {
            chatToggle.addEventListener('click', () => this.toggleChat());
        }

        if (chatMinimize) {
            chatMinimize.addEventListener('click', () => this.closeChat());
        }

        if (chatSend) {
            chatSend.addEventListener('click', () => this.sendMessage());
        }

        if (chatInput) {
            chatInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    this.sendMessage();
                }
            });

            chatInput.addEventListener('input', () => {
                this.updateSendButton();
            });
        }

        // Quick actions
        document.querySelectorAll('.quick-action').forEach(button => {
            button.addEventListener('click', (e) => {
                const action = e.currentTarget.dataset.action;
                this.handleQuickAction(action);
            });
        });

        // Close chat when clicking outside
        document.addEventListener('click', (e) => {
            const chatWindow = document.getElementById('chat-window');
            const chatToggle = document.getElementById('chat-toggle');
            
            if (this.isOpen && chatWindow && !chatWindow.contains(e.target) && !chatToggle.contains(e.target)) {
                this.closeChat();
            }
        });
    }

    setupKeyboardShortcuts() {
        document.addEventListener('keydown', (e) => {
            // ESC to close chat
            if (e.key === 'Escape' && this.isOpen) {
                this.closeChat();
            }
        });
    }

    toggleChat() {
        if (this.isOpen) {
            this.closeChat();
        } else {
            this.openChat();
        }
    }

    openChat() {
        const chatWindow = document.getElementById('chat-window');
        const chatToggle = document.getElementById('chat-toggle');
        
        if (chatWindow && chatToggle) {
            chatWindow.classList.remove('hidden');
            setTimeout(() => {
                chatWindow.classList.add('show');
            }, 10);
            
            chatToggle.classList.add('active');
            chatToggle.querySelector('.chat-icon').classList.add('hidden');
            chatToggle.querySelector('.chat-close-icon').classList.remove('hidden');
            
            this.isOpen = true;
            this.focusInput();
            
            // Send welcome message if it's the first time
            if (!this.hasWelcomed) {
                setTimeout(() => {
                    this.showQuickWelcome();
                }, 500);
                this.hasWelcomed = true;
            }
        }
    }

    closeChat() {
        const chatWindow = document.getElementById('chat-window');
        const chatToggle = document.getElementById('chat-toggle');
        
        if (chatWindow && chatToggle) {
            chatWindow.classList.remove('show');
            setTimeout(() => {
                chatWindow.classList.add('hidden');
            }, 300);
            
            chatToggle.classList.remove('active');
            chatToggle.querySelector('.chat-icon').classList.remove('hidden');
            chatToggle.querySelector('.chat-close-icon').classList.add('hidden');
            
            this.isOpen = false;
        }
    }

    focusInput() {
        const chatInput = document.getElementById('chat-input');
        if (chatInput) {
            setTimeout(() => {
                chatInput.focus();
            }, 100);
        }
    }

    sendMessage() {
        const chatInput = document.getElementById('chat-input');
        if (!chatInput) return;

        const message = chatInput.value.trim();
        if (!message) return;

        // Add user message
        this.addMessage(message, 'user');
        chatInput.value = '';
        this.updateSendButton();

        // Show typing indicator and respond
        this.showTypingIndicator();
        setTimeout(() => {
            this.hideTypingIndicator();
            this.respondToMessage(message);
        }, 1000 + Math.random() * 1000); // Random delay for realism
    }

    addMessage(content, sender = 'assistant', showTime = true) {
        const chatMessages = document.getElementById('chat-messages');
        if (!chatMessages) return;

        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${sender}-message`;

        const avatarDiv = document.createElement('div');
        avatarDiv.className = 'message-avatar';
        
        if (sender === 'assistant') {
            avatarDiv.innerHTML = '<img src="imgs/LOGO_MAXMIND_WHITE.png" alt="MaxMind" class="w-6 h-6">';
        } else {
            avatarDiv.innerHTML = '<i data-lucide="user" class="w-4 h-4 text-white"></i>';
        }

        const contentDiv = document.createElement('div');
        contentDiv.className = 'message-content';
        
        const messageP = document.createElement('p');
        messageP.textContent = content;
        contentDiv.appendChild(messageP);

        if (showTime) {
            const timeSpan = document.createElement('span');
            timeSpan.className = 'message-time';
            timeSpan.textContent = this.getCurrentTime();
            contentDiv.appendChild(timeSpan);
        }

        messageDiv.appendChild(avatarDiv);
        messageDiv.appendChild(contentDiv);
        chatMessages.appendChild(messageDiv);

        // Scroll to bottom
        chatMessages.scrollTop = chatMessages.scrollHeight;

        // Initialize Lucide icons for new content
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

    showTypingIndicator() {
        if (this.isTyping) return;
        
        const chatMessages = document.getElementById('chat-messages');
        if (!chatMessages) return;

        this.isTyping = true;
        
        const typingDiv = document.createElement('div');
        typingDiv.className = 'message assistant-message';
        typingDiv.id = 'typing-indicator';

        typingDiv.innerHTML = `
            <div class="message-avatar">
                <img src="imgs/LOGO_MAXMIND_WHITE.png" alt="MaxMind" class="w-6 h-6">
            </div>
            <div class="typing-indicator">
                <div class="typing-dots">
                    <div class="typing-dot"></div>
                    <div class="typing-dot"></div>
                    <div class="typing-dot"></div>
                </div>
            </div>
        `;

        chatMessages.appendChild(typingDiv);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    hideTypingIndicator() {
        const typingIndicator = document.getElementById('typing-indicator');
        if (typingIndicator) {
            typingIndicator.remove();
        }
        this.isTyping = false;
    }

    updateSendButton() {
        const chatInput = document.getElementById('chat-input');
        const chatSend = document.getElementById('chat-send');
        
        if (chatInput && chatSend) {
            chatSend.disabled = !chatInput.value.trim();
        }
    }

    getCurrentTime() {
        const now = new Date();
        return now.toLocaleTimeString('fr-FR', { 
            hour: '2-digit', 
            minute: '2-digit' 
        });
    }

    showQuickWelcome() {
        const welcomeMessages = [
            "Je peux vous aider avec nos services de développement web, applications mobiles, et solutions digitales.",
            "N'hésitez pas à utiliser les boutons rapides ci-dessous ou à me poser directement vos questions ! 😊"
        ];

        welcomeMessages.forEach((msg, index) => {
            setTimeout(() => {
                this.addMessage(msg, 'assistant');
            }, index * 1500);
        });
    }

    handleQuickAction(action) {
        const quickActionResponses = {
            services: "Nos services incluent :\n\n🌐 Développement Web\n📱 Applications Mobiles\n🛒 E-commerce\n🎨 Design UI/UX\n📈 Marketing Digital\n\nQuel service vous intéresse le plus ?",
            portfolio: "Nous avons réalisé plus de 100 projets pour des clients au Maroc et à l'international. Visitez notre section 'Projets' pour voir nos réalisations ou contactez-nous pour des exemples spécifiques à votre secteur.",
            contact: "📞 Contactez-nous :\n\n📧 Email : contact@maxmind.ma\n📱 Téléphone : +212 XXX XXX XXX\n📍 Adresse : Casablanca, Maroc\n\nOu utilisez le formulaire de contact sur notre site !",
            devis: "🎯 Devis gratuit !\n\nPour recevoir un devis personnalisé, j'ai besoin de quelques informations :\n\n• Type de projet (site web, app mobile, etc.)\n• Fonctionnalités souhaitées\n• Budget approximatif\n• Délai souhaité\n\nPouvez-vous me parler de votre projet ?"
        };

        const response = quickActionResponses[action];
        if (response) {
            this.showTypingIndicator();
            setTimeout(() => {
                this.hideTypingIndicator();
                this.addMessage(response, 'assistant');
            }, 800);
        }
    }

    initializeResponses() {
        return {
            greetings: [
                "Bonjour ! Comment puis-je vous aider aujourd'hui ?",
                "Salut ! Je suis là pour répondre à vos questions sur MaxMind.",
                "Hello ! Que puis-je faire pour vous ?"
            ],
            services: [
                "MaxMind propose des services complets en développement web, applications mobiles, e-commerce, design UI/UX et marketing digital. Quel service vous intéresse ?",
                "Nous sommes spécialisés dans les solutions digitales : sites web, apps mobiles, boutiques en ligne et stratégies marketing. Sur quoi souhaitez-vous en savoir plus ?"
            ],
            pricing: [
                "Nos tarifs varient selon la complexité du projet. Nous proposons toujours un devis gratuit et personnalisé. Voulez-vous que je vous mette en contact avec notre équipe ?",
                "Chaque projet est unique ! Nous offrons des devis gratuits adaptés à vos besoins et budget. Parlez-moi de votre projet ?"
            ],
            contact: [
                "Vous pouvez nous contacter par email à contact@maxmind.ma ou utiliser le formulaire de contact sur notre site. Notre équipe vous répondra rapidement !",
                "Pour nous joindre : email contact@maxmind.ma, téléphone +212 XXX XXX XXX, ou via le formulaire de contact. Comment préférez-vous être contacté ?"
            ],
            default: [
                "C'est une excellente question ! Notre équipe peut vous donner plus de détails. Souhaitez-vous que je vous mette en contact avec un expert ?",
                "Je vais transmettre votre question à notre équipe technique. En attendant, puis-je vous aider avec autre chose ?",
                "Intéressant ! Pour une réponse détaillée, je recommande de parler directement avec notre équipe. Voulez-vous planifier un appel ?"
            ]
        };
    }

    respondToMessage(message) {
        const lowerMessage = message.toLowerCase();
        let response;

        // Determine response category
        if (this.containsWords(lowerMessage, ['bonjour', 'salut', 'hello', 'bonsoir', 'hey'])) {
            response = this.getRandomResponse('greetings');
        } else if (this.containsWords(lowerMessage, ['service', 'développement', 'web', 'mobile', 'app', 'site'])) {
            response = this.getRandomResponse('services');
        } else if (this.containsWords(lowerMessage, ['prix', 'tarif', 'coût', 'budget', 'devis'])) {
            response = this.getRandomResponse('pricing');
        } else if (this.containsWords(lowerMessage, ['contact', 'téléphone', 'email', 'adresse', 'joindre'])) {
            response = this.getRandomResponse('contact');
        } else {
            response = this.getRandomResponse('default');
        }

        this.addMessage(response, 'assistant');
    }

    containsWords(text, words) {
        return words.some(word => text.includes(word));
    }

    getRandomResponse(category) {
        const responses = this.responses[category];
        return responses[Math.floor(Math.random() * responses.length)];
    }

    // Additional utility methods
    showNotification(message, type = 'info') {
        // Create a simple notification system
        const notification = document.createElement('div');
        notification.className = `ai-notification ai-notification-${type}`;
        notification.textContent = message;

        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#01acb6'};
            color: white;
            padding: 12px 20px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            z-index: 1001;
            font-size: 14px;
            font-weight: 500;
            transform: translateX(100%);
            transition: transform 0.3s ease;
        `;

        document.body.appendChild(notification);

        // Animate in
        setTimeout(() => {
            notification.style.transform = 'translateX(0)';
        }, 100);

        // Remove after 3 seconds
        setTimeout(() => {
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 3000);
    }

    // Analytics tracking (placeholder)
    trackEvent(action, category = 'AI Assistant') {
        console.log(`📊 Analytics: ${category} - ${action}`);
        // Here you could integrate with Google Analytics, Mixpanel, etc.
    }

    // Enhanced message handling with rich content
    addRichMessage(content, sender = 'assistant', options = {}) {
        if (typeof content === 'string') {
            this.addMessage(content, sender);
            return;
        }

        // Handle rich content (buttons, links, etc.)
        const chatMessages = document.getElementById('chat-messages');
        if (!chatMessages) return;

        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${sender}-message`;

        // Add avatar
        const avatarDiv = document.createElement('div');
        avatarDiv.className = 'message-avatar';

        if (sender === 'assistant') {
            avatarDiv.innerHTML = '<img src="imgs/LOGO_MAXMIND_WHITE.png" alt="MaxMind" class="w-6 h-6">';
        } else {
            avatarDiv.innerHTML = '<i data-lucide="user" class="w-4 h-4 text-white"></i>';
        }

        // Add content
        const contentDiv = document.createElement('div');
        contentDiv.className = 'message-content';

        if (content.text) {
            const messageP = document.createElement('p');
            messageP.textContent = content.text;
            contentDiv.appendChild(messageP);
        }

        // Add buttons if provided
        if (content.buttons && content.buttons.length > 0) {
            const buttonsDiv = document.createElement('div');
            buttonsDiv.className = 'message-buttons';
            buttonsDiv.style.cssText = 'margin-top: 8px; display: flex; flex-wrap: wrap; gap: 6px;';

            content.buttons.forEach(button => {
                const btn = document.createElement('button');
                btn.textContent = button.text;
                btn.className = 'message-button';
                btn.style.cssText = `
                    background: #f1f5f9;
                    border: 1px solid #e2e8f0;
                    border-radius: 6px;
                    padding: 4px 8px;
                    font-size: 12px;
                    cursor: pointer;
                    transition: all 0.2s ease;
                `;

                btn.addEventListener('click', () => {
                    if (button.action) {
                        this.handleQuickAction(button.action);
                    } else if (button.url) {
                        window.open(button.url, '_blank');
                    }
                });

                btn.addEventListener('mouseenter', () => {
                    btn.style.background = '#01acb6';
                    btn.style.color = 'white';
                });

                btn.addEventListener('mouseleave', () => {
                    btn.style.background = '#f1f5f9';
                    btn.style.color = 'inherit';
                });

                buttonsDiv.appendChild(btn);
            });

            contentDiv.appendChild(buttonsDiv);
        }

        // Add timestamp
        if (options.showTime !== false) {
            const timeSpan = document.createElement('span');
            timeSpan.className = 'message-time';
            timeSpan.textContent = this.getCurrentTime();
            contentDiv.appendChild(timeSpan);
        }

        messageDiv.appendChild(avatarDiv);
        messageDiv.appendChild(contentDiv);
        chatMessages.appendChild(messageDiv);

        // Scroll to bottom
        chatMessages.scrollTop = chatMessages.scrollHeight;

        // Initialize Lucide icons
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

    // Method to clear chat history
    clearChat() {
        const chatMessages = document.getElementById('chat-messages');
        if (chatMessages) {
            chatMessages.innerHTML = `
                <div class="message assistant-message">
                    <div class="message-avatar">
                        <img src="imgs/LOGO_MAXMIND_WHITE.png" alt="MaxMind" class="w-6 h-6">
                    </div>
                    <div class="message-content">
                        <p>Bonjour ! 👋 Je suis l'assistant virtuel de MaxMind. Comment puis-je vous aider aujourd'hui ?</p>
                        <span class="message-time">Maintenant</span>
                    </div>
                </div>
            `;
        }
    }

    // Test function to verify integration
    testIntegration() {
        const tests = {
            elementsExist: this.testElementsExist(),
            eventsWork: this.testEvents(),
            responsiveDesign: this.testResponsiveDesign(),
            accessibility: this.testAccessibility(),
            performance: this.testPerformance()
        };

        console.log('🧪 AI Assistant Integration Test Results:', tests);

        const allPassed = Object.values(tests).every(test => test.passed);

     

        return tests;
    }

    testElementsExist() {
        const requiredElements = [
            'chat-toggle',
            'chat-window',
            'chat-messages',
            'chat-input',
            'chat-send'
        ];

        const missing = requiredElements.filter(id => !document.getElementById(id));

        return {
            passed: missing.length === 0,
            details: missing.length === 0 ? 'All elements found' : `Missing elements: ${missing.join(', ')}`
        };
    }

    testEvents() {
        try {
            const chatToggle = document.getElementById('chat-toggle');
            const chatInput = document.getElementById('chat-input');

            // Test if event listeners are attached
            const hasToggleEvents = chatToggle && chatToggle.onclick !== null;
            const hasInputEvents = chatInput && (chatInput.onkeypress !== null || chatInput.addEventListener);

            return {
                passed: true, // Events are attached in constructor
                details: 'Event listeners attached successfully'
            };
        } catch (error) {
            return {
                passed: false,
                details: `Event test failed: ${error.message}`
            };
        }
    }

    testResponsiveDesign() {
        const chatWindow = document.getElementById('chat-window');
        if (!chatWindow) {
            return { passed: false, details: 'Chat window not found' };
        }

        const styles = getComputedStyle(chatWindow);
        const hasResponsiveStyles = styles.width && styles.height;

        return {
            passed: hasResponsiveStyles,
            details: hasResponsiveStyles ? 'Responsive styles applied' : 'Missing responsive styles'
        };
    }

    testAccessibility() {
        const chatToggle = document.getElementById('chat-toggle');
        const chatInput = document.getElementById('chat-input');

        const hasAriaLabels = chatToggle && chatToggle.getAttribute('aria-label');
        const hasProperInput = chatInput && chatInput.getAttribute('placeholder');

        return {
            passed: hasAriaLabels && hasProperInput,
            details: 'Accessibility attributes present'
        };
    }

    testPerformance() {
        const startTime = performance.now();

        // Simulate some operations
        this.getCurrentTime();
        this.getRandomResponse('greetings');

        const endTime = performance.now();
        const executionTime = endTime - startTime;

        return {
            passed: executionTime < 10, // Should be very fast
            details: `Execution time: ${executionTime.toFixed(2)}ms`
        };
    }

    // Debug method for development
    debug() {
        console.log('🔍 AI Assistant Debug Info:', {
            isOpen: this.isOpen,
            isTyping: this.isTyping,
            hasWelcomed: this.hasWelcomed,
            responses: Object.keys(this.responses),
            elements: {
                chatToggle: !!document.getElementById('chat-toggle'),
                chatWindow: !!document.getElementById('chat-window'),
                chatMessages: !!document.getElementById('chat-messages'),
                chatInput: !!document.getElementById('chat-input')
            }
        });
    }
}

// Initialize AI Assistant when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.maxmindAssistant = new MaxMindAssistant();

    // Make debug functions available globally
    window.testAIAssistant = () => window.maxmindAssistant.testIntegration();
    window.debugAIAssistant = () => window.maxmindAssistant.debug();

    // Add some initial delay before showing the assistant
    setTimeout(() => {
        const chatToggle = document.getElementById('chat-toggle');
        if (chatToggle) {
            chatToggle.style.opacity = '0';
            chatToggle.style.transform = 'scale(0.8)';
            chatToggle.style.transition = 'all 0.5s cubic-bezier(0.4, 0, 0.2, 1)';

            setTimeout(() => {
                chatToggle.style.opacity = '1';
                chatToggle.style.transform = 'scale(1)';

                // Run integration test after initialization
                setTimeout(() => {
                    const testResults = window.maxmindAssistant.testIntegration();
                    console.log('🚀 MaxMind AI Assistant ready!');

                    // Show a subtle notification that the assistant is ready
                    if (testResults && Object.values(testResults).every(test => test.passed)) {
                        console.log('✅ AI Assistant fully operational');
                    }
                }, 500);
            }, 100);
        }
    }, 2000);

    // Add keyboard shortcut to open chat (Ctrl/Cmd + K)
    document.addEventListener('keydown', (e) => {
        if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
            e.preventDefault();
            if (window.maxmindAssistant) {
                window.maxmindAssistant.openChat();
            }
        }
    });
});
