<?php
include __DIR__ . '/../admin/config.php';

function getSlugFromURL()
{
    global $debugMode;

    $request_uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);

    if ($debugMode) {
        echo "<!-- DEBUG: getSlugFromURL() called -->\n";
        echo "<!-- DEBUG: Parsed URI: " . $request_uri . " -->\n";
    }

    // Check if the request is for portfolio.php with a slug
    if (preg_match('#^(.*/)?portfolio\.php/(.+)$#', $request_uri, $matches)) {
        $slug = trim($matches[2], '/');
        if ($debugMode) {
            echo "<!-- DEBUG: Found slug: " . $slug . " -->\n";
        }
        return $slug;
    }

    // If it's just portfolio.php without a slug, return empty string
    if (preg_match('#^(.*/)?portfolio\.php/?$#', $request_uri)) {
        if ($debugMode) {
            echo "<!-- DEBUG: Portfolio.php without slug detected -->\n";
        }
        return '';
    }

    if ($debugMode) {
        echo "<!-- DEBUG: No matching pattern found, returning empty string -->\n";
    }
    return '';
}

function fetchPortfolioData($slug)
{
    if (empty($slug)) {
        return null;
    }
    $baseUrl = getBaseUrl();
    $adminPath = (strpos($baseUrl, 'localhost') !== false) ? '/admin' : 'admin';
    $api_url = $baseUrl . $adminPath . "/cms.php/Records?category_id=4&q=" . urlencode($slug);
    echo "<script>console.log('Debug from PHP: " . $api_url . "');</script>";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        'Content-Type: application/json',
        'Accept: application/json',
        "X-API-TOKEN: {$GLOBALS['API_TOKEN']}"
    ));

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);

    curl_close($ch);

    if ($curl_error) {
        error_log("cURL Error: " . $curl_error);
        return null;
    }

    if ($http_code !== 200) {
        error_log("HTTP Error: " . $http_code);
        return null;
    }

    $data = json_decode($response, true);

    if ($data && isset($data['success']) && $data['success'] && isset($data['data']['items'][0])) {
        return $data['data']['items'][0];
    }

    return null;
}


function processPortfolioMedia($portfolio_items)
{
    $all_media = [];
    $categories = [];

    foreach ($portfolio_items as $item) {
        if (!isset($item['values'][3]) || empty($item['values'][3])) {
            continue;
        }

        $media_files = $item['values'][3];
        if (is_string($media_files)) {
            $media_files = json_decode($media_files, true) ?: explode(',', $media_files);
        }

        if (!is_array($media_files)) {
            continue;
        }

        $title = $item['values'][0] ?? 'Portfolio Item';
        $description = $item['values'][1] ?? '';
        $category = $item['values']['category'] ?? 'General';

        // Add category to our list
        if (!in_array($category, $categories)) {
            $categories[] = $category;
        }

        foreach ($media_files as $media_url) {
            $media_url = trim($media_url);
            if (!empty($media_url)) {
                // Check for local video files
                $is_local_video = preg_match('/\.(mp4|webm|ogg|mov|avi|mkv)$/i', $media_url);

                // Check for external video URLs
                $external_video_type = isExternalVideo($media_url);

                // Determine if this is a video (local or external)
                $is_video = $is_local_video || $external_video_type;

                $all_media[] = [
                    'url' => $media_url,
                    'title' => $title,
                    'description' => $description,
                    'category' => $category,
                    'is_video' => $is_video,
                    'is_external_video' => $external_video_type,
                    'item_id' => $item['id'] ?? 0
                ];
            }
        }
    }

    return ['media' => $all_media, 'categories' => $categories];
}
function generateMasonryGrid($media_items, $max_items = 50)
{
    if (empty($media_items)) {
        return '<div class="col-span-full text-center py-20"><p class="text-gray-500 text-xl">No portfolio items available</p></div>';
    }

    // Shuffle for variety, then limit
    shuffle($media_items);
    $media_items = array_slice($media_items, 0, $max_items);

    $grid_html = '';

    foreach ($media_items as $index => $media) {
        $grid_html .= generateMasonryCell($media, $index);
    }

    return $grid_html;
}

function generateMasonryCell($media, $index)
{
    $media_type = $media['is_video'] ? 'videos' : 'images';
    $category_color = getCategoryColor($media['category']);

    $onclick = "openLightbox('" . htmlspecialchars($media['url']) . "', '" .
        htmlspecialchars($media['title']) . "', '" .
        htmlspecialchars($media['category']) . "')";

    if ($media['is_video']) {
        if ($media['is_external_video']) {
            // Handle external videos with iframes
            $embed_url = '';
            switch ($media['is_external_video']) {
                case 'youtube':
                    $embed_url = getYouTubeEmbedUrl($media['url']);
                    break;
                case 'instagram':
                    $embed_url = getInstagramEmbedUrl($media['url']);
                    break;
                case 'vimeo':
                    $embed_url = getVimeoEmbedUrl($media['url']);
                    break;
                default:
                    $embed_url = $media['url'];
                    break;
            }

            $media_element = "
                <div class=\"iframe-container relative w-full h-full bg-gray-900\">
                    <iframe class=\"w-full h-full object-cover transition-all duration-500\"
                            src=\"" . htmlspecialchars($embed_url) . "\"
                            frameborder=\"0\"
                            allow=\"accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture\"
                            allowfullscreen
                            data-external-video=\"" . htmlspecialchars($media['is_external_video']) . "\"
                            style=\"min-height: 250px;\">
                    </iframe>
                </div>
                <div class=\"video-overlay absolute inset-0 transition-opacity duration-300 bg-black bg-opacity-20\"></div>
                <div class=\"video-play-indicator\">
                    <svg class=\"w-6 h-6 text-white ml-1\" fill=\"currentColor\" viewBox=\"0 0 24 24\">
                        <path d=\"M8 5v14l11-7z\"/>
                    </svg>
                </div>";
        } else {
            // Handle local video files
            $media_element = "
                <video class=\"masonry-video w-full h-auto object-cover transition-all duration-500\"
                       muted loop preload=\"metadata\" src=\"" . htmlspecialchars($media['url']) . "\"
                       poster=\"" . htmlspecialchars($media['url']) . "\"
                       onloadedmetadata=\"this.currentTime=0\"
                       data-video=\"true\">
                    <source src=\"" . htmlspecialchars($media['url']) . "\" type=\"video/mp4\">
                </video>
                <div class=\"video-overlay absolute inset-0 transition-opacity duration-300\"></div>
                <div class=\"video-play-indicator\">
                    <svg class=\"w-6 h-6 text-white ml-1\" fill=\"currentColor\" viewBox=\"0 0 24 24\">
                        <path d=\"M8 5v14l11-7z\"/>
                    </svg>
                </div>";
        }
    } else {
        $media_element = "
            <img class=\"masonry-image w-full h-auto object-cover transition-transform duration-700 group-hover:scale-105\"
                 src=\"" . htmlspecialchars($media['url']) . "\"
                 alt=\"" . htmlspecialchars($media['title']) . "\"
                 onerror=\"this.style.display='none'\">";
    }

    return "
    <div class=\"masonry-item portfolio-item show animate-fade-in\"
         data-category=\"" . $media_type . "\"
         data-index=\"" . $index . "\"
         data-orientation=\"pending\"
         style=\"opacity: 0; transform: translateY(20px); transition: all 0.6s cubic-bezier(0.4, 0, 0.2, 1);\">
        <div class=\"group relative overflow-hidden rounded-xl shadow-lg hover:shadow-2xl transition-all duration-500 cursor-pointer bg-gray-100 h-full\"
             onclick=\"{$onclick}\">
            {$media_element}
            <div class=\"absolute inset-0 opacity-0 group-hover:opacity-100 transition-opacity duration-300\"></div>
            <div class=\"absolute bottom-4 left-4 right-4 text-white transform translate-y-4 group-hover:translate-y-0 opacity-0 duration-300\">
                <h3 class=\"text-sm font-semibold leading-tight\">" . htmlspecialchars($media['title']) . "</h3>
            </div>
            <div class=\"absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity duration-300\">
                <div class=\"bg-white/20 backdrop-blur-sm rounded-full p-2\">
                    <svg class=\"w-5 h-5 text-white\" fill=\"none\" stroke=\"currentColor\" viewBox=\"0 0 24 24\">
                        <path stroke-linecap=\"round\" stroke-linejoin=\"round\" stroke-width=\"2\" d=\"M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z\"/>
                    </svg>
                </div>
            </div>
        </div>
    </div>";
}

function getCategoryColor($category)
{
    $colors = [
        'branding' => 'bg-blue-500/20',
        'web' => 'bg-green-500/20',
        'mobile' => 'bg-purple-500/20',
        'print' => 'bg-orange-500/20',
        'general' => 'bg-gray-500/20'
    ];

    $key = strtolower($category);
    return $colors[$key] ?? $colors['general'];
}

function isExternalVideo($url)
{
    // Check for YouTube URLs
    if (preg_match('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/i', $url)) {
        return 'youtube';
    }

    // Check for Instagram URLs
    if (preg_match('/instagram\.com\/(?:p|reel)\/([a-zA-Z0-9_-]+)/i', $url)) {
        return 'instagram';
    }

    // Check for other external video URLs (Vimeo, TikTok, etc.)
    if (preg_match('/vimeo\.com\/(\d+)/i', $url)) {
        return 'vimeo';
    }

    return false;
}

function getYouTubeEmbedUrl($url)
{
    if (preg_match('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/i', $url, $matches)) {
        return 'https://www.youtube.com/embed/' . $matches[1] . '?autoplay=0&mute=1&controls=1&rel=0';
    }
    return $url;
}

function getInstagramEmbedUrl($url)
{
    if (preg_match('/instagram\.com\/(?:p|reel)\/([a-zA-Z0-9_-]+)/i', $url, $matches)) {
        return 'https://www.instagram.com/p/' . $matches[1] . '/embed/';
    }
    return $url;
}

function getVimeoEmbedUrl($url)
{
    if (preg_match('/vimeo\.com\/(\d+)/i', $url, $matches)) {
        return 'https://player.vimeo.com/video/' . $matches[1] . '?autoplay=0&muted=1';
    }
    return $url;
}

function generateFilterButtons($categories)
{
    $buttons_html = '
    <div class="text-center mb-12">
        <div class="flex flex-wrap justify-center gap-3">
            <button class="portfolio-filter active" data-filter="all">All Media</button>
            <button class="portfolio-filter" data-filter="images">Images</button>
            <button class="portfolio-filter" data-filter="videos">Videos</button>';

    $buttons_html .= '
        </div>
    </div>';

    return $buttons_html;
}

// Get slug and fetch data
$slug = getSlugFromURL();

// Redirect to previous page if no slug is provided
if (empty($slug)) {
    echo '<script>
        if (document.referrer && document.referrer !== window.location.href) {
            window.location.href = document.referrer;
        } else {
            window.location.href = "/";
        }
    </script>';
    exit();
}

$portfolio_data = null;
$page_title = "MaxMind Blog | Portfolio website, company showcase, professional design";
$page_description = "Découvrez comment notre portfolio website valorise la company showcase de MaxMind: professional design, UX moderne, projets innovants et réussites clients.";

if ($slug) {
    $portfolio_data = fetchPortfolioData($slug);
}

// Process portfolio media from the single item (not all items)
$portfolio_media = [];
$portfolio_categories = [];
if ($portfolio_data) {
    $processed_data = processPortfolioMedia([$portfolio_data]);
    $portfolio_media = $processed_data['media'];
    $portfolio_categories = $processed_data['categories'];

    // Update page meta data based on portfolio data
    $page_title = isset($portfolio_data['values'][0]) ?
        htmlspecialchars($portfolio_data['values'][0]) . " | MaxMind Portfolio" :
        $page_title;

    $page_description = isset($portfolio_data['values'][1]) ?
        htmlspecialchars($portfolio_data['values'][1]) :
        $page_description;
}

?>