function changeItemsPerPage(perPage) {
    const url = new URL(window.location);
    url.searchParams.set('per_page', perPage);
    url.searchParams.set('page', '1'); // Reset to first page when changing items per page
    window.location.href = url.toString();
}

// Projects Carousel Functionality
class ProjectsCarousel {
    constructor() {
        this.carousel = document.getElementById('projectsCarousel');
        this.prevBtn = document.getElementById('projectsPrevBtn');
        this.nextBtn = document.getElementById('projectsNextBtn');
        this.dotsContainer = document.getElementById('projectsPaginationDots');

        if (!this.carousel) return;

        this.slides = this.carousel.querySelectorAll('.project-slide');
        this.currentIndex = 0;
        this.slidesToShow = this.getSlidesToShow();
        this.maxIndex = Math.max(0, this.slides.length - this.slidesToShow);

        this.init();

        // Handle window resize
        window.addEventListener('resize', () => {
            this.slidesToShow = this.getSlidesToShow();
            this.maxIndex = Math.max(0, this.slides.length - this.slidesToShow);
            this.updateCarousel();
            this.generateDots();
        });
    }

    // Reset carousel to beginning (for filtering)
    reset() {
        this.currentIndex = 0;
        // Only count visible slides (not hidden by filters)
        this.slides = this.carousel.querySelectorAll('.project-slide:not([style*="display: none"])');
        this.maxIndex = Math.max(0, this.slides.length - this.slidesToShow);
        this.updateCarousel();
        this.generateDots();
    }

    getSlidesToShow() {
        const width = window.innerWidth;
        if (width >= 1024) return 3; // lg: 3 slides
        if (width >= 768) return 2;  // md: 2 slides
        return 1; // sm: 1 slide
    }

    init() {
        this.generateDots();
        this.updateCarousel();
        this.attachEventListeners();
    }

    generateDots() {
        if (!this.dotsContainer) return;

        this.dotsContainer.innerHTML = '';

        // Only count visible slides for dots calculation
        const visibleSlides = this.carousel.querySelectorAll('.project-slide:not([style*="display: none"])');
        const totalDots = Math.ceil(visibleSlides.length / this.slidesToShow);

        // Only show dots if there are more slides than can be displayed at once
        if (totalDots <= 1) {
            this.dotsContainer.style.display = 'none';
            return;
        } else {
            this.dotsContainer.style.display = 'flex';
        }

        for (let i = 0; i < totalDots; i++) {
            const dot = document.createElement('button');
            dot.className = `w-3 h-3 rounded-full transition-all duration-300 ${i === 0 ? 'bg-[#01acb6]' : 'bg-slate-300'}`;
            dot.addEventListener('click', () => this.goToSlide(i * this.slidesToShow));
            this.dotsContainer.appendChild(dot);
        }
    }

    updateCarousel() {
        if (!this.carousel) return;

        // Get visible slides only
        const visibleSlides = this.carousel.querySelectorAll('.project-slide:not([style*="display: none"])');
        this.maxIndex = Math.max(0, visibleSlides.length - this.slidesToShow);

        const slideWidth = 100 / this.slidesToShow;
        const translateX = -(this.currentIndex * slideWidth);
        this.carousel.style.transform = `translateX(${translateX}%)`;

        // Update navigation buttons based on visible slides
        if (this.prevBtn) {
            const shouldDisablePrev = this.currentIndex === 0;
            this.prevBtn.style.opacity = shouldDisablePrev ? '0.5' : '1';
            this.prevBtn.style.pointerEvents = shouldDisablePrev ? 'none' : 'auto';
        }

        if (this.nextBtn) {
            const shouldDisableNext = this.currentIndex >= this.maxIndex || visibleSlides.length <= this.slidesToShow;
            this.nextBtn.style.opacity = shouldDisableNext ? '0.5' : '1';
            this.nextBtn.style.pointerEvents = shouldDisableNext ? 'none' : 'auto';
        }

        // Update dots based on visible slides
        const dots = this.dotsContainer?.querySelectorAll('button');
        if (dots) {
            dots.forEach((dot, index) => {
                const isActive = Math.floor(this.currentIndex / this.slidesToShow) === index;
                dot.className = `w-3 h-3 rounded-full transition-all duration-300 ${isActive ? 'bg-[#01acb6]' : 'bg-slate-300'}`;
            });
        }
    }

    goToSlide(index) {
        this.currentIndex = Math.max(0, Math.min(index, this.maxIndex));
        this.updateCarousel();
    }

    nextSlide() {
        const visibleSlides = this.carousel.querySelectorAll('.project-slide:not([style*="display: none"])');
        const maxIndex = Math.max(0, visibleSlides.length - this.slidesToShow);

        if (this.currentIndex < maxIndex) {
            this.currentIndex += this.slidesToShow;
            this.currentIndex = Math.min(this.currentIndex, maxIndex);
            this.updateCarousel();
        }
    }

    prevSlide() {
        if (this.currentIndex > 0) {
            this.currentIndex -= this.slidesToShow;
            this.currentIndex = Math.max(this.currentIndex, 0);
            this.updateCarousel();
        }
    }

    attachEventListeners() {
        if (this.prevBtn) {
            this.prevBtn.addEventListener('click', () => this.prevSlide());
        }

        if (this.nextBtn) {
            this.nextBtn.addEventListener('click', () => this.nextSlide());
        }

        // Touch/swipe support
        let startX = 0;
        let isDragging = false;

        this.carousel.addEventListener('touchstart', (e) => {
            startX = e.touches[0].clientX;
            isDragging = true;
        });

        this.carousel.addEventListener('touchmove', (e) => {
            if (!isDragging) return;
            e.preventDefault();
        });

        this.carousel.addEventListener('touchend', (e) => {
            if (!isDragging) return;

            const endX = e.changedTouches[0].clientX;
            const diff = startX - endX;

            if (Math.abs(diff) > 50) {
                if (diff > 0) {
                    this.nextSlide();
                } else {
                    this.prevSlide();
                }
            }

            isDragging = false;
        });
    }
}

// Initialize Lucide icons and carousel
document.addEventListener('DOMContentLoaded', function () {
    if (typeof lucide !== 'undefined') {
        lucide.createIcons();
    }

    // Initialize the projects carousel
    const projectsCarousel = new ProjectsCarousel();

    // Enhanced filter functionality for carousel
    const filterButtons = document.querySelectorAll('.filter-btn');
    const projectSlides = document.querySelectorAll('.project-slide');

    function filterProjectsCarousel(category) {
        // Add smooth transition effect
        const carousel = document.getElementById('projectsCarousel');
        if (carousel) {
            carousel.style.opacity = '0.7';
        }

        setTimeout(() => {
            projectSlides.forEach((slide) => {
                const slideCategory = slide.getAttribute('data-category');
                const shouldShow = category === 'all' || slideCategory === category;

                if (shouldShow) {
                    slide.style.display = 'block';
                    // Add fade-in animation
                    slide.style.opacity = '0';
                    setTimeout(() => {
                        slide.style.opacity = '1';
                    }, 50);
                } else {
                    slide.style.display = 'none';
                }
            });

            // Reset carousel to beginning and update
            if (projectsCarousel && projectsCarousel.reset) {
                projectsCarousel.reset();
            }

            // Restore carousel opacity
            if (carousel) {
                carousel.style.opacity = '1';
            }

            // Update project count
            updateProjectCount();
        }, 150);
    }

    // Project count display (console only)
    function updateProjectCount() {
        const visibleProjects = document.querySelectorAll('.project-slide:not([style*="display: none"])').length;
        const totalProjects = projectSlides.length;

        console.log(`${visibleProjects} projets sur ${totalProjects} affichés`);
    }

    // Handle filter button clicks
    filterButtons.forEach(button => {
        button.addEventListener('click', () => {
            // Remove active class from all buttons
            filterButtons.forEach(btn => {
                btn.classList.remove('active-filter');
                btn.classList.add('bg-white', 'text-slate-700', 'border-slate-200');
                btn.classList.remove('bg-[#01acb6]', 'text-white', 'border-[#01acb6]');
            });

            // Add active class to clicked button
            button.classList.add('active-filter');
            button.classList.remove('bg-white', 'text-slate-700', 'border-slate-200');
            button.classList.add('bg-[#01acb6]', 'text-white', 'border-[#01acb6]');

            // Add click feedback animation
            button.style.transform = 'scale(0.95)';
            setTimeout(() => {
                button.style.transform = 'scale(1)';
            }, 100);

            // Apply filter
            const category = button.getAttribute('data-filter');
            filterProjectsCarousel(category);
        });
    });

    // Initialize smooth transitions
    projectSlides.forEach(slide => {
        slide.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
    });

    // Filter buttons transition
    filterButtons.forEach(btn => {
        btn.style.transition = 'all 0.3s ease, transform 0.1s ease';
    });

    // Initialize the project count
    updateProjectCount();
});