// Utility: Debounce function for performance optimization
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Utility: Create skeleton loading card
function createSkeletonCard() {
    const skeleton = document.createElement('div');
    skeleton.className = 'project-card skeleton-card bg-white rounded-2xl overflow-hidden shadow-lg flex flex-col animate-pulse';
    skeleton.innerHTML = `
        <div class="relative h-64 bg-slate-200"></div>
        <div class="p-6 flex-1 flex flex-col">
            <div class="h-6 bg-slate-200 rounded mb-3 w-3/4"></div>
            <div class="space-y-2 mb-4 flex-1">
                <div class="h-4 bg-slate-200 rounded w-full"></div>
                <div class="h-4 bg-slate-200 rounded w-5/6"></div>
                <div class="h-4 bg-slate-200 rounded w-4/6"></div>
            </div>
            <div class="flex gap-2 mb-4">
                <div class="h-6 bg-slate-200 rounded-full w-16"></div>
                <div class="h-6 bg-slate-200 rounded-full w-20"></div>
                <div class="h-6 bg-slate-200 rounded-full w-16"></div>
            </div>
            <div class="h-5 bg-slate-200 rounded w-32"></div>
        </div>
    `;
    return skeleton;
}

// Utility: Remove all skeleton cards
function removeSkeletons() {
    document.querySelectorAll('.skeleton-card').forEach(skeleton => skeleton.remove());
}

class AdvancedCarousel {
    constructor() {
        console.log('AdvancedCarousel constructor called');

        this.carousel = document.getElementById('carousel');
        this.dotsContainer = document.getElementById('dotsContainer');
        this.prevBtn = document.getElementById('prevBtn');
        this.nextBtn = document.getElementById('nextBtn');

        console.log('Carousel elements found:', {
            carousel: !!this.carousel,
            dotsContainer: !!this.dotsContainer,
            prevBtn: !!this.prevBtn,
            nextBtn: !!this.nextBtn
        });

        if (!this.carousel) {
            console.log('No carousel found, exiting');
            return; // Exit if carousel doesn't exist on this page
        }

        this.currentIndex = 0;
        this.cardWidth = 370; // 350px width + 20px gap
        this.visibleDotsCount = 5;
        this.dotsStartIndex = 0;

        console.log('Initializing carousel...');
        this.init();
    }

    init() {
        this.cards = Array.from(this.carousel.children);
        this.totalSlides = this.cards.length;

        if (this.totalSlides === 0) return;

        this.setupImages();
        this.generateDots();
        this.updateCarousel();
        this.attachEventListeners();
        this.updateNavigationButtons();
        this.startAutoPlay();
    }

    setupImages() {
        console.log('Setting up images for', this.cards.length, 'cards');

        this.cards.forEach((card, index) => {
            const placeholder = card.querySelector('.advanced-loading-placeholder');
            const img = card.querySelector('img');

            if (img && placeholder) {
                console.log('Setting up image for card', index, 'with src:', img.src);

                // Check if image is already loaded (for cached images)
                if (img.complete && img.naturalHeight !== 0) {
                    console.log('Image already loaded for card', index);
                    placeholder.style.display = 'none';
                    return;
                }

                // Show image when loaded, hide placeholder
                img.onload = () => {
                    console.log('Image loaded successfully for card', index);
                    if (placeholder) {
                        placeholder.style.display = 'none';
                    }
                };

                // Handle image load error
                img.onerror = () => {
                    console.error('Image failed to load for card', index, 'src:', img.src);
                    if (placeholder) {
                        placeholder.textContent = 'Failed to load';
                        placeholder.style.background = 'linear-gradient(135deg, #ff6b6b 0%, #ee5a24 100%)';
                    }
                };

                // For already loaded images (immediate check)
                setTimeout(() => {
                    if (img.complete && img.naturalHeight !== 0) {
                        console.log('Image was already loaded (delayed check) for card', index);
                        if (placeholder) {
                            placeholder.style.display = 'none';
                        }
                    }
                }, 100);
            } else {
                console.warn('Missing image or placeholder for card', index);
            }
        });
    }

    generateDots() {
        if (!this.dotsContainer) return;

        this.dotsContainer.innerHTML = '';

        for (let i = 0; i < this.totalSlides; i++) {
            const dot = document.createElement('div');
            dot.className = 'advanced-dot';
            dot.dataset.index = i;
            dot.addEventListener('click', () => this.goToSlide(i));
            this.dotsContainer.appendChild(dot);
        }

        this.dots = Array.from(this.dotsContainer.children);
        this.updateDotsVisibility();
    }

    updateDotsVisibility() {
        if (!this.dots) return;

        // Hide all dots first
        this.dots.forEach(dot => {
            dot.style.display = 'none';
            dot.classList.remove('center', 'adjacent', 'outer');
        });

        // Calculate the range of dots to show
        let start = Math.max(0, this.currentIndex - 2);
        let end = Math.min(this.totalSlides - 1, start + 4);

        // Adjust if we're near the end
        if (end - start < 4 && this.totalSlides >= 5) {
            start = Math.max(0, end - 4);
        }

        // Show and style the visible dots
        for (let i = start; i <= end; i++) {
            const dot = this.dots[i];
            if (dot) {
                dot.style.display = 'block';

                const relativePos = i - start;
                const centerPos = Math.floor((end - start) / 2);

                if (i === this.currentIndex) {
                    dot.classList.add('center');
                } else if (Math.abs(relativePos - centerPos) === 1) {
                    dot.classList.add('adjacent');
                } else {
                    dot.classList.add('outer');
                }
            }
        }
    }

    updateCarousel() {
        if (!this.carousel) return;

        const translateX = -this.currentIndex * this.cardWidth;
        this.carousel.style.transform = `translateX(${translateX}px)`;
        this.updateDotsVisibility();
    }

    updateNavigationButtons() {
        if (this.prevBtn) this.prevBtn.disabled = this.currentIndex === 0;
        if (this.nextBtn) this.nextBtn.disabled = this.currentIndex === this.totalSlides - 1;
    }

    goToSlide(index) {
        if (index >= 0 && index < this.totalSlides) {
            this.currentIndex = index;
            this.updateCarousel();
            this.updateNavigationButtons();
        }
    }

    nextSlide() {
        if (this.currentIndex < this.totalSlides - 1) {
            this.goToSlide(this.currentIndex + 1);
        }
    }

    prevSlide() {
        if (this.currentIndex > 0) {
            this.goToSlide(this.currentIndex - 1);
        }
    }

    startAutoPlay() {
        this.stopAutoPlay(); // Clear any existing interval
        this.autoPlayInterval = setInterval(() => {
            if (this.currentIndex >= this.totalSlides - 1) {
                // Reset to first slide when reaching the end
                this.goToSlide(0);
            } else {
                this.nextSlide();
            }
        }, 4000); // Move every 1 second
    }

    stopAutoPlay() {
        if (this.autoPlayInterval) {
            clearInterval(this.autoPlayInterval);
            this.autoPlayInterval = null;
        }
    }

    attachEventListeners() {
        if (this.prevBtn) {
            this.prevBtn.addEventListener('click', () => {
                this.stopAutoPlay();
                this.prevSlide();
                this.startAutoPlay();
            });
        }
        if (this.nextBtn) {
            this.nextBtn.addEventListener('click', () => {
                this.stopAutoPlay();
                this.nextSlide();
                this.startAutoPlay();
            });
        }

        // Pause auto-play on hover
        if (this.carousel.parentElement) {
            this.carousel.parentElement.addEventListener('mouseenter', () => {
                this.stopAutoPlay();
            });
            this.carousel.parentElement.addEventListener('mouseleave', () => {
                this.startAutoPlay();
            });
        }

        // Card click events
        this.cards.forEach((card, index) => {
            card.addEventListener('click', (e) => {
                // If clicked directly on the card (not the link icon)
                if (!e.target.closest('.link-icon')) {
                    const link = card.dataset.link;
                    if (link) {
                        window.location.href = link;
                    }
                }
            });

            // Link icon click events
            const linkIcon = card.querySelector('.link-icon');
            if (linkIcon) {
                linkIcon.addEventListener('click', (e) => {
                    e.stopPropagation(); // Prevent card click
                    const link = card.dataset.link;
                    if (link) {
                        window.open(link, '_blank');
                    }
                });
            }
        });

        // Keyboard navigation
        document.addEventListener('keydown', (e) => {
            if (e.key === 'ArrowLeft') {
                this.stopAutoPlay();
                this.prevSlide();
                this.startAutoPlay();
            } else if (e.key === 'ArrowRight') {
                this.stopAutoPlay();
                this.nextSlide();
                this.startAutoPlay();
            }
        });

        // Touch/swipe support for mobile
        let startX = 0;
        let isDragging = false;

        if (this.carousel) {
            this.carousel.addEventListener('touchstart', (e) => {
                startX = e.touches[0].clientX;
                isDragging = true;
                this.stopAutoPlay();
            });

            this.carousel.addEventListener('touchmove', (e) => {
                if (!isDragging) return;
                e.preventDefault();
            });

            this.carousel.addEventListener('touchend', (e) => {
                if (!isDragging) return;

                const endX = e.changedTouches[0].clientX;
                const diff = startX - endX;

                if (Math.abs(diff) > 50) { // Minimum swipe distance
                    if (diff > 0) {
                        this.nextSlide();
                    } else {
                        this.prevSlide();
                    }
                }

                isDragging = false;
                this.startAutoPlay();
            });
        }
    }
}

// Initialiser Lucide icons
lucide.createIcons();

// Fonction de filtrage améliorée - will be initialized after DOM loads
let filterButtons;
let projectCards;
let projectsGrid;

// Compteur de projets visible (console only)
function updateProjectCount() {
    const visibleProjects = document.querySelectorAll('.project-card:not([style*="display: none"])').length;
    const totalProjects = projectCards.length;

    console.log(`${visibleProjects} projets sur ${totalProjects} affichés`);
}

function filterProjects(category, categoryId = null) {
    // Add smooth transition effect
    if (projectsGrid) {
        projectsGrid.style.opacity = '0.7';
    }

    setTimeout(() => {
        // Get all project cards (including dynamically loaded ones)
        const allProjectCards = document.querySelectorAll('.project-card');

        allProjectCards.forEach((card) => {
            const cardCategory = card.getAttribute('data-category');
            const cardCategoryId = card.getAttribute('data-category-id');

            let shouldShow = false;

            // Check if filtering by category ID (Design Projects or Web Projects)
            if (categoryId && categoryId !== 'all') {
                shouldShow = cardCategoryId === categoryId;
            }
            // Otherwise filter by type slug
            else if (category === 'all') {
                shouldShow = true;
            } else {
                shouldShow = cardCategory === category;
            }

            if (shouldShow) {
                card.style.display = 'flex';
                // Add fade-in animation
                card.style.opacity = '0';
                setTimeout(() => {
                    card.style.opacity = '1';
                }, 50);
            } else {
                card.style.display = 'none';
            }
        });

        // Restore grid opacity
        if (projectsGrid) {
            projectsGrid.style.opacity = '1';
        }

        updateProjectCount();
    }, 150);
}

// Global reference to LoadMoreProjects instance
let loadMoreProjectsInstance = null;

// Function to initialize filter buttons
function initializeFilterButtons() {
    filterButtons = document.querySelectorAll('.filter-btn');
    projectCards = document.querySelectorAll('.project-card');
    projectsGrid = document.querySelector('.grid.grid-cols-1') ||
                   document.querySelector('#projects .grid') ||
                   document.querySelector('[class*="grid-cols"]');

    console.log('Filter initialization:', {
        filterButtons: filterButtons.length,
        projectCards: projectCards.length,
        projectsGrid: !!projectsGrid
    });

    // Gestion des clics sur les boutons de filtre
    filterButtons.forEach(button => {
        button.addEventListener('click', () => {
            // Retirer la classe active de tous les boutons
            filterButtons.forEach(btn => {
                btn.classList.remove('active-filter');
                btn.classList.add('bg-white', 'text-slate-700', 'border-slate-200');
                btn.classList.remove('bg-[#01acb6]', 'text-white', 'border-[#01acb6]');
            });

            // Ajouter la classe active au bouton cliqué
            button.classList.add('active-filter');
            button.classList.remove('bg-white', 'text-slate-700', 'border-slate-200');
            button.classList.add('bg-[#01acb6]', 'text-white', 'border-[#01acb6]');

            // Add click feedback animation
            button.style.transform = 'scale(0.95)';
            setTimeout(() => {
                button.style.transform = 'scale(1)';
            }, 100);

            // Effectuer le filtrage
            const category = button.getAttribute('data-filter');
            const categoryId = button.getAttribute('data-category-id');
            filterProjects(category, categoryId);

            // Update LoadMoreProjects filter if instance exists
            if (loadMoreProjectsInstance) {
                loadMoreProjectsInstance.setFilter(category, categoryId);
            }
        });
    });

    // Initialize smooth transitions
    projectCards.forEach(card => {
        card.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
    });

    if (projectsGrid) {
        projectsGrid.style.transition = 'opacity 0.3s ease';
    }

    // Filter buttons transition
    filterButtons.forEach(btn => {
        btn.style.transition = 'all 0.3s ease, transform 0.1s ease';
    });

    // Initialiser le compteur
    if (projectCards.length > 0) {
        updateProjectCount();
    }

    // Initialize tag click handlers
    initializeTagClickHandlers();
}

// Function to filter projects by tag
function filterProjectsByTag(tagName) {
    // Add smooth transition effect
    if (projectsGrid) {
        projectsGrid.style.opacity = '0.7';
    }

    setTimeout(() => {
        // Get all project cards (including dynamically loaded ones)
        const allProjectCards = document.querySelectorAll('.project-card');

        allProjectCards.forEach((card) => {
            // Get all tags within this card
            const cardTags = Array.from(card.querySelectorAll('.project-tag')).map(tag => tag.getAttribute('data-tag'));

            // Check if this card has the selected tag
            const hasTag = cardTags.includes(tagName);

            if (hasTag) {
                card.style.display = 'flex';
                // Add fade-in animation
                card.style.opacity = '0';
                setTimeout(() => {
                    card.style.opacity = '1';
                }, 50);
            } else {
                card.style.display = 'none';
            }
        });

        // Restore grid opacity
        if (projectsGrid) {
            projectsGrid.style.opacity = '1';
        }

        // Reset filter buttons to show none is active
        filterButtons.forEach(btn => {
            btn.classList.remove('active');
        });

        updateProjectCount();
    }, 150);
}

// Function to initialize tag click handlers
function initializeTagClickHandlers() {
    // Use event delegation for dynamically added tags
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('project-tag')) {
            e.stopPropagation(); // Prevent card click event
            const tagName = e.target.getAttribute('data-tag');
            filterProjectsByTag(tagName);
        }
    });
}

// Web Projects Carousel Class
class WebProjectsCarousel {
    constructor() {
        console.log('WebProjectsCarousel constructor called');

        this.carousel = document.getElementById('webProjectsCarousel');
        this.dotsContainer = document.getElementById('webDotsContainer');
        this.prevBtn = document.getElementById('webPrevBtn');
        this.nextBtn = document.getElementById('webNextBtn');

        console.log('Web Projects Carousel elements found:', {
            carousel: !!this.carousel,
            dotsContainer: !!this.dotsContainer,
            prevBtn: !!this.prevBtn,
            nextBtn: !!this.nextBtn
        });

        if (!this.carousel) {
            console.log('No web projects carousel found, exiting');
            return;
        }

        this.currentIndex = 0;
        this.cardWidth = 404; // 380px width + 24px gap
        this.visibleCards = this.getVisibleCards();
        this.autoPlayInterval = null;
        this.autoPlayDelay = 5000;

        console.log('Initializing web projects carousel...');
        this.init();
    }

    init() {
        this.cards = Array.from(this.carousel.children);
        this.totalSlides = this.cards.length;

        if (this.totalSlides === 0) return;

        this.generateDots();
        this.updateCarousel();
        this.attachEventListeners();
        this.updateNavigationButtons();
        this.startAutoPlay();

        // Handle window resize
        window.addEventListener('resize', () => {
            this.visibleCards = this.getVisibleCards();
            this.updateCarousel();
            this.updateNavigationButtons();
        });
    }

    getVisibleCards() {
        const containerWidth = this.carousel.parentElement.offsetWidth;
        if (containerWidth <= 480) return 1;
        if (containerWidth <= 768) return 2;
        if (containerWidth <= 1200) return 3;
        return 3; // Default for larger screens
    }

    generateDots() {
        if (!this.dotsContainer) return;

        this.dotsContainer.innerHTML = '';
        const totalDots = Math.max(0, this.totalSlides - this.visibleCards + 1);

        for (let i = 0; i < totalDots; i++) {
            const dot = document.createElement('div');
            dot.className = `web-projects-dot ${i === 0 ? 'active' : ''}`;
            dot.addEventListener('click', () => {
                this.stopAutoPlay();
                this.goToSlide(i);
                this.startAutoPlay();
            });
            this.dotsContainer.appendChild(dot);
        }
    }

    updateCarousel() {
        if (!this.carousel) return;

        const maxIndex = Math.max(0, this.totalSlides - this.visibleCards);
        this.currentIndex = Math.min(this.currentIndex, maxIndex);

        const translateX = -(this.currentIndex * this.cardWidth);
        this.carousel.style.transform = `translateX(${translateX}px)`;

        this.updateDots();
    }

    updateDots() {
        if (!this.dotsContainer) return;

        const dots = this.dotsContainer.querySelectorAll('.web-projects-dot');
        dots.forEach((dot, index) => {
            dot.classList.toggle('active', index === this.currentIndex);
        });
    }

    updateNavigationButtons() {
        if (!this.prevBtn || !this.nextBtn) return;

        const maxIndex = Math.max(0, this.totalSlides - this.visibleCards);

        this.prevBtn.style.opacity = this.currentIndex === 0 ? '0.5' : '1';
        this.prevBtn.style.pointerEvents = this.currentIndex === 0 ? 'none' : 'auto';

        this.nextBtn.style.opacity = this.currentIndex >= maxIndex ? '0.5' : '1';
        this.nextBtn.style.pointerEvents = this.currentIndex >= maxIndex ? 'none' : 'auto';
    }

    prevSlide() {
        if (this.currentIndex > 0) {
            this.currentIndex--;
            this.updateCarousel();
            this.updateNavigationButtons();
        }
    }

    nextSlide() {
        const maxIndex = Math.max(0, this.totalSlides - this.visibleCards);
        if (this.currentIndex < maxIndex) {
            this.currentIndex++;
            this.updateCarousel();
            this.updateNavigationButtons();
        }
    }

    goToSlide(index) {
        const maxIndex = Math.max(0, this.totalSlides - this.visibleCards);
        this.currentIndex = Math.min(Math.max(0, index), maxIndex);
        this.updateCarousel();
        this.updateNavigationButtons();
    }

    startAutoPlay() {
        this.stopAutoPlay();
        if (this.totalSlides > this.visibleCards) {
            this.autoPlayInterval = setInterval(() => {
                const maxIndex = Math.max(0, this.totalSlides - this.visibleCards);
                if (this.currentIndex >= maxIndex) {
                    this.currentIndex = 0;
                } else {
                    this.currentIndex++;
                }
                this.updateCarousel();
                this.updateNavigationButtons();
            }, this.autoPlayDelay);
        }
    }

    stopAutoPlay() {
        if (this.autoPlayInterval) {
            clearInterval(this.autoPlayInterval);
            this.autoPlayInterval = null;
        }
    }

    attachEventListeners() {
        if (this.prevBtn) {
            this.prevBtn.addEventListener('click', () => {
                this.stopAutoPlay();
                this.prevSlide();
                this.startAutoPlay();
            });
        }

        if (this.nextBtn) {
            this.nextBtn.addEventListener('click', () => {
                this.stopAutoPlay();
                this.nextSlide();
                this.startAutoPlay();
            });
        }

        // Pause auto-play on hover
        if (this.carousel.parentElement) {
            this.carousel.parentElement.addEventListener('mouseenter', () => {
                this.stopAutoPlay();
            });
            this.carousel.parentElement.addEventListener('mouseleave', () => {
                this.startAutoPlay();
            });
        }

        // Handle card clicks
        this.cards.forEach((card) => {
            const cardElement = card.querySelector('.group');
            if (cardElement) {
                cardElement.addEventListener('click', (e) => {
                    // The onclick is already handled in the HTML
                });
            }
        });
    }
}

// Load More Projects Functionality
class LoadMoreProjects {
    constructor() {
        this.currentPage = 1;
        this.perPage = 10;
        this.isLoading = false;
        this.hasMore = true;
        this.type = 'all'; // Default to all projects (both Design ID 8 and Web ID 11)
        this.activeFilter = 'all'; // Track current filter
        this.activeCategoryId = 'all'; // Track current category ID filter

        this.loadMoreBtn2 = document.getElementById('loadMoreBtn2');
        // Try multiple selectors to find the projects grid
        this.projectsGrid = document.querySelector('.grid.grid-cols-1') ||
                           document.querySelector('#projects .grid') ||
                           document.querySelector('[class*="grid-cols"]');
        this.loadingIndicator = null;

        console.log('LoadMoreProjects initialized:', {
            loadMoreBtn2: this.loadMoreBtn2,
            projectsGrid: this.projectsGrid,
            gridClassName: this.projectsGrid ? this.projectsGrid.className : 'not found'
        });

        if (this.loadMoreBtn2 && this.projectsGrid) {
            this.init();
        } else {
            console.error('LoadMoreProjects: Missing required elements', {
                loadMoreBtn2: !!this.loadMoreBtn2,
                projectsGrid: !!this.projectsGrid
            });
        }
    }

    init() {
        console.log('==========================================');
        console.log('LoadMoreProjects init() called');
        console.log('Button element:', this.loadMoreBtn2);
        console.log('Grid element:', this.projectsGrid);
        console.log('==========================================');

        // Create loading indicator
        this.createLoadingIndicator();

        // Attach click event to Load More button with detailed debugging
        if (this.loadMoreBtn2) {
            console.log('Attaching click event to Load More button');

            this.loadMoreBtn2.addEventListener('click', (e) => {
                console.log('==========================================');
                console.log('LOAD MORE BUTTON CLICKED!');
                console.log('Event:', e);
                console.log('Current page:', this.currentPage);
                console.log('Has more:', this.hasMore);
                console.log('Is loading:', this.isLoading);
                console.log('Type filter:', this.type);
                console.log('==========================================');

                e.preventDefault();
                e.stopPropagation();
                this.loadMore();
            });

            // Add additional event listener to test if button is clickable
            this.loadMoreBtn2.addEventListener('mouseenter', () => {
                console.log('Mouse entered Load More button - button is interactive');
            });

            console.log('Click event listener attached successfully');
        } else {
            console.error('ERROR: loadMoreBtn2 is null! Cannot attach event listener');
        }

        // Check if there's initial pagination data
        const paginationInfo = document.querySelector('.pagination-container');
        if (paginationInfo) {
            console.log('Hiding default pagination container');
            paginationInfo.style.display = 'none';
        }

        console.log('LoadMoreProjects initialization complete');
        console.log('==========================================');
    }

    createLoadingIndicator() {
        this.loadingIndicator = document.createElement('div');
        this.loadingIndicator.className = 'loading-indicator hidden flex items-center justify-center py-8';
        this.loadingIndicator.innerHTML = `
            <div class="flex flex-col items-center space-y-4">
                <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-[#01acb6]"></div>
                <p class="text-slate-600 font-medium">Chargement des projets...</p>
            </div>
        `;

        // Insert loading indicator before the button's parent
        const buttonContainer = this.loadMoreBtn2.parentNode;
        buttonContainer.parentNode.insertBefore(this.loadingIndicator, buttonContainer);

        console.log('Loading indicator created');
    }

    showLoading() {
        this.isLoading = true;
        this.loadingIndicator.classList.remove('hidden');
        this.loadMoreBtn2.disabled = true;
        this.loadMoreBtn2.style.opacity = '0.5';
        this.loadMoreBtn2.style.cursor = 'not-allowed';

        // Add skeleton cards for better UX
        for (let i = 0; i < 3; i++) {
            this.projectsGrid.appendChild(createSkeletonCard());
        }
    }

    hideLoading() {
        this.isLoading = false;
        this.loadingIndicator.classList.add('hidden');
        this.loadMoreBtn2.disabled = false;
        this.loadMoreBtn2.style.opacity = '1';
        this.loadMoreBtn2.style.cursor = 'pointer';

        // Remove skeleton cards
        removeSkeletons();
    }

    setFilter(category, categoryId = null) {
        // Reset state when filter changes
        this.activeFilter = category;
        this.activeCategoryId = categoryId || 'all';
        this.currentPage = 1;
        this.hasMore = true;

        // Map category slugs to type IDs
        if (category === 'all') {
            this.type = 'all';
        } else {
            // Extract type from visible cards in that category
            const sampleCard = document.querySelector(`[data-category="${category}"]`);
            if (sampleCard) {
                const typeAttr = sampleCard.getAttribute('data-type');
                // Map type names to IDs (you may need to adjust based on your data)
                // For now, we'll pass 'all' and let client-side filtering handle it
                this.type = 'all';
            }
        }

        // Reset button state
        this.resetLoadMoreButton();
    }

    resetLoadMoreButton() {
        this.loadMoreBtn2.innerHTML = `
            <i data-lucide="plus-circle" class="w-5 h-5 mr-2 group-hover:rotate-90 transition-transform duration-300"></i>
            <span>Charger plus de projets</span>
            <i data-lucide="chevron-down" class="w-5 h-5 ml-2 group-hover:translate-y-1 transition-transform duration-300"></i>
        `;
        this.loadMoreBtn2.disabled = false;
        this.loadMoreBtn2.style.cursor = 'pointer';
        this.loadMoreBtn2.style.opacity = '1';
        this.loadMoreBtn2.classList.remove('bg-slate-400', 'cursor-not-allowed');
        this.loadMoreBtn2.classList.add('bg-[#01acb6]', 'hover:bg-[#018a92]', 'hover:scale-105', 'active:scale-95');

        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

    async loadMore() {
        console.log('==========================================');
        console.log('loadMore() method called');
        console.log('State check:', {
            isLoading: this.isLoading,
            hasMore: this.hasMore,
            currentPage: this.currentPage,
            perPage: this.perPage,
            type: this.type,
            activeFilter: this.activeFilter
        });
        console.log('==========================================');

        if (this.isLoading) {
            console.warn('⚠️ Load more blocked: Already loading');
            return;
        }

        if (!this.hasMore) {
            console.warn('⚠️ Load more blocked: No more items to load');
            return;
        }

        this.showLoading();
        console.log('✓ Loading state activated');

        this.currentPage++;
        console.log('✓ Page incremented to:', this.currentPage);

        // Build CMS API URL directly
        const basePath = window.location.pathname.substring(0, window.location.pathname.lastIndexOf('/'));
        const url = `${basePath}/admin/cms.php/Records?category_id=4&page=${this.currentPage}&per_page=${this.perPage}`;

        console.log('==========================================');
        console.log('🌐 Making AJAX request to CMS API:');
        console.log('   Base path:', basePath);
        console.log('   Relative URL:', url);
        console.log('   Full URL:', window.location.origin + url);
        console.log('   Category ID: 4 (Projects)');
        console.log('   Page:', this.currentPage);
        console.log('   Per page:', this.perPage);
        console.log('   Active filter:', this.activeFilter);
        console.log('==========================================');

        try {
            console.log('📡 Sending fetch request to CMS...');
            const response = await fetch(url, 
                {
                    headers: {
                    "X-API-TOKEN": "2250c47a4397a41fe5ff8f2c2a92b240ddf46aaeded8ffddb4f29f02ffef8720",
                    "Content-Type": "application/json",
                }
                }
            );

            console.log('📥 Response received:', {
                ok: response.ok,
                status: response.status,
                statusText: response.statusText,
                headers: {
                    contentType: response.headers.get('content-type')
                }
            });

            if (!response.ok) {
                throw new Error(`Network response was not ok: ${response.status} ${response.statusText}`);
            }

            const responseText = await response.text();
            console.log('📄 Raw response text (first 500 chars):', responseText.substring(0, 500));

            let data;
            try {
                data = JSON.parse(responseText);
                console.log('✓ JSON parsed successfully');
                console.log('📦 Full data received:', data);
            } catch (parseError) {
                console.error('❌ JSON parse error:', parseError);
                console.error('Response text:', responseText);
                throw new Error('Invalid JSON response');
            }

            if (data.success && data.data && data.data.items && data.data.items.length > 0) {
                console.log('✓ Success! Received', data.data.items.length, 'total items from CMS');

                // Filter based on active category ID filter
                const filteredProjects = data.data.items.filter(project => {
                    if (!project.values) return false;

                    // If filtering by specific category ID (8 or 11)
                    if (this.activeCategoryId && this.activeCategoryId !== 'all') {
                        const matches = project.values[5] == this.activeCategoryId;
                        if (matches) {
                            console.log('   ✓ Project included (Category ' + this.activeCategoryId + '):', project.values[0]);
                        } else {
                            console.log('   ✗ Project excluded (Category ' + project.values[5] + '):', project.values[0]);
                        }
                        return matches;
                    }

                    // If filtering by type slug (e.g., 'ecommerce', 'portfolio')
                    if (this.activeFilter && this.activeFilter !== 'all') {
                        const projectType = (project.values[8] || '').toLowerCase().replace(/[^a-z0-9\-]/g, '').replace(/-+/g, '-');
                        const matches = (project.values[5] == 8 || project.values[5] == 11) && projectType === this.activeFilter;
                        if (matches) {
                            console.log('   ✓ Project included (Type ' + projectType + '):', project.values[0]);
                        }
                        return matches;
                    }

                    // Default: show all Design (8) and Web (11) projects
                    const hasValidType = project.values[5] == 8 || project.values[5] == 11;
                    if (hasValidType) {
                        console.log('   ✓ Project included:', project.values[0], '(Type:', project.values[5] + ')');
                    } else {
                        console.log('   ✗ Project excluded:', project.values[0], '(Type:', project.values?.[5] + ')');
                    }
                    return hasValidType;
                });

                console.log('✓ Filtered to', filteredProjects.length, 'projects matching current filter');

                if (filteredProjects.length > 0) {
                    this.renderProjects(filteredProjects, this.activeFilter);

                    // Check if there are more pages from CMS
                    const totalPages = data.data.pagination?.total_pages || 1;
                    this.hasMore = this.currentPage < totalPages;

                    console.log('✓ Projects rendered');
                    console.log('   Current page:', this.currentPage);
                    console.log('   Total pages:', totalPages);
                    console.log('   Has more:', this.hasMore);

                    // Update button text
                    if (!this.hasMore) {
                        console.log('ℹ️ No more pages - updating button to "all displayed"');
                        this.loadMoreBtn2.innerHTML = `
                            <i data-lucide="check-circle" class="w-5 h-5 mr-2"></i>
                            Tous les projets affichés
                        `;
                        this.loadMoreBtn2.disabled = true;
                        this.loadMoreBtn2.style.cursor = 'not-allowed';
                        this.loadMoreBtn2.classList.remove('hover:bg-[#018a92]');
                        this.loadMoreBtn2.classList.add('bg-slate-400');

                        if (typeof lucide !== 'undefined') {
                            lucide.createIcons();
                        }
                    } else {
                        const remaining = (totalPages - this.currentPage) * this.perPage;
                        console.log('ℹ️ More pages available. Approx remaining:', remaining);
                        this.loadMoreBtn2.querySelector('span').textContent =
                            `Charger plus de projets`;
                    }

                    // Smooth scroll to first new item
                    setTimeout(() => {
                        const newItems = this.projectsGrid.querySelectorAll('.project-card');
                        const firstNewItem = newItems[newItems.length - filteredProjects.length];
                        if (firstNewItem) {
                            firstNewItem.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
                        }
                    }, 100);
                } else {
                    console.warn('⚠️ No projects matching current filter in this batch, trying next page...');
                    // No matching projects in this batch, automatically try next page if available
                    if (this.currentPage < (data.data.pagination?.total_pages || 1)) {
                        this.hideLoading();
                        this.currentPage--; // Reset increment
                        setTimeout(() => this.loadMore(), 500); // Try next page
                    } else {
                        this.hasMore = false;
                        this.showNoMoreMessage();
                    }
                }
            } else {
                console.warn('⚠️ No items returned or success=false');
                console.log('   Data:', data);
                this.hasMore = false;
                this.showNoMoreMessage();
            }

        } catch (error) {
            console.error('==========================================');
            console.error('❌ ERROR loading more projects:');
            console.error('   Error message:', error.message);
            console.error('   Error stack:', error.stack);
            console.error('==========================================');
            this.showErrorMessage();
        } finally {
            console.log('✓ Hiding loading state');
            this.hideLoading();
            console.log('==========================================');
        }
    }

    renderProjects(projects, activeFilter = 'all') {
        const baseUrl = window.location.origin + window.location.pathname.replace(/\/[^\/]*$/, '');
        const adminPath = window.location.hostname === 'localhost' ? '/' : '';

        projects.forEach((project, index) => {
            const card = this.createProjectCard(project, baseUrl, adminPath);
            const cardCategory = card.getAttribute('data-category');
            const cardCategoryId = card.getAttribute('data-category-id');

            // Apply filter visibility - check both category ID and type slug
            let shouldShow = true;
            if (this.activeCategoryId && this.activeCategoryId !== 'all') {
                shouldShow = cardCategoryId === this.activeCategoryId;
            } else if (activeFilter !== 'all' && cardCategory !== activeFilter) {
                shouldShow = false;
            }

            if (!shouldShow) {
                card.style.display = 'none';
            }

            // Add fade-in animation
            card.style.opacity = '0';
            card.style.transform = 'translateY(20px)';
            this.projectsGrid.appendChild(card);

            // Trigger animation
            setTimeout(() => {
                card.style.transition = 'all 0.5s ease';
                card.style.opacity = '1';
                card.style.transform = 'translateY(0)';
            }, index * 100);
        });

        // Re-initialize Lucide icons for new cards
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

    createProjectCard(project, baseUrl, adminPath) {
        const card = document.createElement('div');
        const type = project.values[8] || 'Portfolio';
        const typeSlug = type.toLowerCase().replace(/[^a-z0-9\-]/g, '').replace(/-+/g, '-');
        const image = project.values[2];
        const tags = project.values[6] || [];
        const categoryId = project.values[5] || ''; // Get the category ID (8 or 11)

        card.className = 'project-card group relative bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-500 hover:-translate-y-2 flex flex-col cursor-pointer';
        card.setAttribute('data-category', typeSlug);
        card.setAttribute('data-type', type);
        card.setAttribute('data-category-id', categoryId);
        card.onclick = () => window.open(`${baseUrl}${adminPath}/portfolio.php/${project.slug}`);

        // Only render if image exists
        if (image && Array.isArray(image) && image.length > 0) {
            card.innerHTML = `
                <div class="relative h-64 overflow-hidden bg-slate-200">
                    <img src="${image[0]}"
                        alt="${this.escapeHtml(project.values[0])}"
                        loading="lazy"
                        class="w-full h-full object-cover group-hover:scale-110 transition-all duration-500 origin-center"
                        onload="this.style.opacity=1"
                        style="opacity:0;transition:opacity 0.3s ease-in-out">

                    <div class="absolute inset-0 bg-black/60 opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                    <div class="absolute inset-0 bg-black/20 group-hover:bg-black/10 transition-colors duration-300"></div>

                    <div class="absolute top-4 left-4 px-3 py-1 bg-[#01acb6] text-white text-sm font-medium rounded-full">
                        ${this.escapeHtml(type)}
                    </div>
                </div>

                <div class="p-6 flex-1 flex flex-col">
                    <h3 class="text-xl font-bold text-slate-800 mb-3 group-hover:text-slate-900 transition-colors">
                        ${this.escapeHtml(project.values[0])}
                    </h3>

                    <p class="text-slate-600 text-sm leading-relaxed mb-4 flex-1">
                        ${this.escapeHtml(this.truncateText(this.stripHtml(project.values[1]), 120))}
                    </p>

                    ${Array.isArray(tags) && tags.length > 0 ? `
                        <div class="flex flex-wrap gap-2 mb-4">
                            ${tags.map(tag => `<span class="project-tag px-2 py-1 bg-slate-100 text-slate-600 text-xs rounded-full hover:bg-[#01acb6] hover:text-white transition-colors cursor-pointer" data-tag="${this.escapeHtml(tag)}">${this.escapeHtml(tag)}</span>`).join('')}
                        </div>
                    ` : ''}

                    <div class="flex items-center text-slate-400 group-hover:text-slate-600 transition-colors cursor-pointer mt-auto">
                        <a href="" class="font-medium mr-2 text-sm">Voir le projet</a>
                        <i data-lucide="arrow-right" class="w-4 h-4 group-hover:translate-x-1 transition-transform"></i>
                    </div>
                </div>
            `;
        }

        return card;
    }

    stripHtml(html) {
        const tmp = document.createElement('div');
        tmp.innerHTML = html;
        return tmp.textContent || tmp.innerText || '';
    }

    truncateText(text, maxLength) {
        if (text.length <= maxLength) return text;
        return text.substr(0, maxLength) + '...';
    }

    escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, m => map[m]);
    }

    showNoMoreMessage() {
        this.loadMoreBtn2.innerHTML = `
            <i data-lucide="check-circle" class="w-5 h-5 mr-2"></i>
            Tous les projets affichés
        `;
        this.loadMoreBtn2.disabled = true;
        this.loadMoreBtn2.classList.add('bg-slate-400');

        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }

    showErrorMessage() {
        const errorDiv = document.createElement('div');
        errorDiv.className = 'text-center py-4 px-6 bg-red-50 border border-red-200 rounded-lg mb-4';
        errorDiv.innerHTML = `
            <p class="text-red-600 font-medium">
                <i data-lucide="alert-circle" class="w-5 h-5 inline mr-2"></i>
                Erreur lors du chargement des projets. Veuillez réessayer.
            </p>
        `;
        this.loadMoreBtn2.parentNode.insertBefore(errorDiv, this.loadMoreBtn2);

        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }

        // Remove error message after 5 seconds
        setTimeout(() => errorDiv.remove(), 5000);

        // Go back one page
        this.currentPage--;
    }
}

// Initialize everything when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM Content Loaded - Initializing all components');

    // Initialize the original design carousel
    new AdvancedCarousel();

    // Initialize the web projects carousel
    new WebProjectsCarousel();

    // Initialize filter buttons and project cards
    initializeFilterButtons();

    // Initialize Load More functionality and store instance
    loadMoreProjectsInstance = new LoadMoreProjects();

    console.log('All components initialized');
});
