document.addEventListener('DOMContentLoaded', function() {
    initializeDesignCarousel();
    initializePortfolioCarousel();
    initializeHeroCarousel();
    initializeTimelineAnimations();
    initializeProcessAnimations();
    initializeMobileNavigation();
    initializeAdvancedCarousel();
    initializeContactForm();
    initializeQuoteModal();
    initializeProjectFilters();
    initializeImagePopups();
    initializeFooterFix();
    initializeLucideIcons();
});
const apikey = "2250c47a4397a41fe5ff8f2c2a92b240ddf46aaeded8ffddb4f29f02ffef8720"
// ==================== DESIGN CAROUSEL ====================
function initializeDesignCarousel() {
    const designCarousel = {
        currentSlide: 0,
        slides: document.querySelectorAll('.design-carousel-slide'),
        dots: document.querySelectorAll('.design-carousel-dot'),
        slidesContainer: document.querySelector('.design-carousel-slides'),
        prevBtn: document.querySelector('.design-carousel-prev'),
        nextBtn: document.querySelector('.design-carousel-next'),
        autoChangeInterval: null,

        init() {
            if (!this.slides.length || !this.dots.length || !this.slidesContainer) {
                console.log('Design carousel elements not found');
                return;
            }

            console.log('Initializing design carousel with', this.slides.length, 'slides');

            // Add click events to navigation buttons
            if (this.prevBtn) {
                this.prevBtn.addEventListener('click', () => {
                    this.prevSlide();
                });
            }

            if (this.nextBtn) {
                this.nextBtn.addEventListener('click', () => {
                    this.nextSlide();
                });
            }

            // Add click events to dots
            this.dots.forEach((dot, index) => {
                dot.addEventListener('click', () => {
                    this.goToSlide(index);
                });
            });

            // Start auto-change every 4 seconds
            this.startAutoChange();

            // Pause on hover
            const container = document.querySelector('.design-carousel-container');
            if (container) {
                container.addEventListener('mouseenter', () => {
                    this.stopAutoChange();
                });

                container.addEventListener('mouseleave', () => {
                    this.startAutoChange();
                });
            }
        },

        goToSlide(index) {
            // Remove active class from current dot
            if (this.dots[this.currentSlide]) {
                this.dots[this.currentSlide].classList.remove('active');
                this.dots[this.currentSlide].classList.remove('bg-[#01acb6]');
                this.dots[this.currentSlide].classList.add('bg-white/30');
            }

            // Update current slide
            this.currentSlide = index;

            // Add active class to new dot
            if (this.dots[this.currentSlide]) {
                this.dots[this.currentSlide].classList.add('active');
                this.dots[this.currentSlide].classList.add('bg-[#01acb6]');
                this.dots[this.currentSlide].classList.remove('bg-white/30');
            }

            // Move carousel
            const translateX = -this.currentSlide * 100;
            this.slidesContainer.style.transform = `translateX(${translateX}%)`;

            console.log('Switched to design slide', index);
        },

        nextSlide() {
            const nextIndex = (this.currentSlide + 1) % this.slides.length;
            this.goToSlide(nextIndex);
        },

        prevSlide() {
            const prevIndex = (this.currentSlide - 1 + this.slides.length) % this.slides.length;
            this.goToSlide(prevIndex);
        },

        startAutoChange() {
            this.autoChangeInterval = setInterval(() => {
                this.nextSlide();
            }, 4000);
            console.log('Design carousel auto-change started');
        },

        stopAutoChange() {
            if (this.autoChangeInterval) {
                clearInterval(this.autoChangeInterval);
                this.autoChangeInterval = null;
            }
        }
    };

    designCarousel.init();

    // Handle visibility change to pause carousel when tab is not visible
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            designCarousel.stopAutoChange();
        } else {
            designCarousel.startAutoChange();
        }
    });
}

// ==================== PORTFOLIO CAROUSEL ====================
function initializePortfolioCarousel() {
    const portfolioCarousel = {
        currentSlide: 0,
        slides: document.querySelectorAll('.carousel-slide-highlight'),
        dots: document.querySelectorAll('.carousel-dot-highlight'),
        autoChangeInterval: null,

        init() {
            if (this.slides.length === 0 || this.dots.length === 0) {
                console.log('Portfolio carousel elements not found');
                return;
            }

            console.log('Initializing portfolio carousel with', this.slides.length, 'slides');

            // Add click events to dots
            this.dots.forEach((dot, index) => {
                dot.addEventListener('click', () => {
                    this.goToSlide(index);
                });
            });

            // Start auto-change every 0.5 seconds
            this.startAutoChange();
        },

        goToSlide(index) {
            // Remove active class from current slide and dot
            this.slides[this.currentSlide].classList.remove('active');
            this.slides[this.currentSlide].style.opacity = '0';
            this.dots[this.currentSlide].classList.remove('active');
            this.dots[this.currentSlide].style.backgroundColor = 'rgba(255, 255, 255, 0.5)';

            // Update current slide
            this.currentSlide = index;

            // Add active class to new slide and dot
            this.slides[this.currentSlide].classList.add('active');
            this.slides[this.currentSlide].style.opacity = '1';
            this.dots[this.currentSlide].classList.add('active');
            this.dots[this.currentSlide].style.backgroundColor = 'white';
            this.dots[this.currentSlide].style.transform = 'scale(1.2)';

            // Reset transform for other dots
            this.dots.forEach((dot, i) => {
                if (i !== this.currentSlide) {
                    dot.style.transform = 'scale(1)';
                }
            });
        },

        nextSlide() {
            const nextIndex = (this.currentSlide + 1) % this.slides.length;
            this.goToSlide(nextIndex);
        },

        startAutoChange() {
            this.stopAutoChange();
            this.autoChangeInterval = setInterval(() => {
                this.nextSlide();
            }, 500);
        },

        stopAutoChange() {
            if (this.autoChangeInterval) {
                clearInterval(this.autoChangeInterval);
                this.autoChangeInterval = null;
            }
        }
    };

    portfolioCarousel.init();

    // Handle visibility change
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            portfolioCarousel.stopAutoChange();
        } else {
            portfolioCarousel.startAutoChange();
        }
    });
}

// ==================== HERO CAROUSEL ====================
function initializeHeroCarousel() {
    const heroCarousel = {
        currentSlide: 0,
        slides: document.querySelectorAll('.carousel-slide'),
        dots: document.querySelectorAll('.carousel-dot'),
        nextBtn: document.getElementById('nextSlide'),
        prevBtn: document.getElementById('prevSlide'),
        progressBar: document.getElementById('carouselProgress'),
        autoChangeInterval: null,
        progressInterval: null,
        slideInterval: 6000,

        init() {
            if (this.slides.length === 0) {
                console.log('Hero carousel elements not found');
                return;
            }

            console.log('Initializing hero carousel with', this.slides.length, 'slides');

            this.setupEventListeners();
            this.startAutoChange();
            this.startProgress();
            this.updateSlide(0);
        },

        setupEventListeners() {
            // Dot navigation
            this.dots.forEach((dot, index) => {
                dot.addEventListener('click', () => {
                    this.goToSlide(index);
                });
            });

            // Arrow navigation
            if (this.nextBtn) {
                this.nextBtn.addEventListener('click', () => {
                    this.nextSlide();
                });
            }

            if (this.prevBtn) {
                this.prevBtn.addEventListener('click', () => {
                    this.prevSlide();
                });
            }

            // Keyboard navigation
            document.addEventListener('keydown', (e) => {
                if (e.key === 'ArrowLeft') {
                    this.prevSlide();
                } else if (e.key === 'ArrowRight') {
                    this.nextSlide();
                }
            });
        },

        goToSlide(index) {
            if (index === this.currentSlide) return;

            // Remove active class from current slide and dot
            this.slides[this.currentSlide].classList.remove('active');
            this.dots[this.currentSlide].classList.remove('active');

            // Update current slide
            this.currentSlide = index;

            // Add active class to new slide and dot
            this.slides[this.currentSlide].classList.add('active');
            this.dots[this.currentSlide].classList.add('active');

            // Restart auto change and progress
            this.startAutoChange();
            this.startProgress();
        },

        updateSlide(index) {
            this.goToSlide(index);
        },

        nextSlide() {
            const nextIndex = (this.currentSlide + 1) % this.slides.length;
            this.goToSlide(nextIndex);
        },

        prevSlide() {
            const prevIndex = (this.currentSlide - 1 + this.slides.length) % this.slides.length;
            this.goToSlide(prevIndex);
        },

        startAutoChange() {
            this.stopAutoChange();
            this.autoChangeInterval = setInterval(() => {
                this.nextSlide();
            }, this.slideInterval);
        },

        stopAutoChange() {
            if (this.autoChangeInterval) {
                clearInterval(this.autoChangeInterval);
                this.autoChangeInterval = null;
            }
        },

        startProgress() {
            this.stopProgress();
            if (this.progressBar) {
                this.progressBar.style.width = '0%';
                let width = 0;
                const increment = 100 / (this.slideInterval / 100);

                this.progressInterval = setInterval(() => {
                    width += increment;
                    if (width >= 100) {
                        width = 0;
                    }
                    this.progressBar.style.width = width + '%';
                }, 100);
            }
        },

        stopProgress() {
            if (this.progressInterval) {
                clearInterval(this.progressInterval);
                this.progressInterval = null;
            }
        }
    };

    heroCarousel.init();

    // Handle visibility change
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            heroCarousel.stopAutoChange();
            heroCarousel.stopProgress();
        } else {
            heroCarousel.startAutoChange();
            heroCarousel.startProgress();
        }
    });
}

// ==================== TIMELINE ANIMATIONS ====================
function initializeTimelineAnimations() {
    const timelineItems = document.querySelectorAll('.timeline-item');
    const timelineProgress = document.querySelector('.timeline-progress');

    // Track scroll direction with debouncing
    let lastScrollTop = 0;
    let isScrollingDown = true;
    let scrollTimeout;

    window.addEventListener('scroll', () => {
        clearTimeout(scrollTimeout);
        scrollTimeout = setTimeout(() => {
            const st = window.pageYOffset || document.documentElement.scrollTop;
            if (Math.abs(lastScrollTop - st) > 5) {
                isScrollingDown = st > lastScrollTop;
                lastScrollTop = st <= 0 ? 0 : st;
            }
        }, 10);
    });

    // Track animated elements
    const animatedElements = new Set();

    // Create intersection observer for timeline items
    const timelineObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            const element = entry.target;

            if (entry.isIntersecting &&
                isScrollingDown &&
                !element.classList.contains('animate-in') &&
                !animatedElements.has(element) &&
                !element.hasAttribute('data-animated')) {

                element.classList.add('animate-in');
                element.setAttribute('data-animated', 'true');
                animatedElements.add(element);

                // Add sweep animation to timeline content
                const timelineContent = element.querySelector('.timeline-content');
                if (timelineContent) {
                    setTimeout(() => {
                        timelineContent.classList.add('animate-sweep');
                        timelineContent.setAttribute('data-swept', 'true');
                    }, 300);
                }

                updateTimelineProgress();
                timelineObserver.unobserve(element);
            }
        });
    }, {
        threshold: 0.3,
        rootMargin: '-50px 0px'
    });

    // Observe all timeline items
    timelineItems.forEach(item => {
        timelineObserver.observe(item);
    });

    // Function to update timeline progress
    function updateTimelineProgress() {
        const visibleItems = document.querySelectorAll('.timeline-item.animate-in');
        const totalItems = timelineItems.length;
        const progressPercentage = (visibleItems.length / totalItems) * 100;

        if (timelineProgress) {
            timelineProgress.style.height = progressPercentage + '%';
        }
    }

    // GSAP ScrollTrigger integration (if available)
    if (typeof gsap !== 'undefined' && gsap.registerPlugin) {
        gsap.registerPlugin(ScrollTrigger);

        // Animate timeline title
        gsap.fromTo('.timeline-title', {
            opacity: 0,
            y: -50
        }, {
            opacity: 1,
            y: 0,
            duration: 1,
            ease: 'power3.out',
            scrollTrigger: {
                trigger: '.timeline-title',
                start: 'top 80%',
                end: 'bottom 20%',
                toggleActions: 'play none none none'
            }
        });

        // Enhanced timeline items animation
        timelineItems.forEach((item, index) => {
            const isEven = index % 2 === 0;

            gsap.fromTo(item, {
                opacity: 0,
                x: isEven ? -100 : 100,
                y: -50
            }, {
                opacity: 1,
                x: 0,
                y: 0,
                duration: 0.6,
                ease: 'power3.out',
                scrollTrigger: {
                    trigger: item,
                    start: 'top 95%',
                    end: 'bottom 15%',
                    toggleActions: 'play none none none',
                    once: true
                }
            });
        });

        // Timeline progress line animation
        if (timelineProgress) {
            gsap.fromTo(timelineProgress, {
                height: '0%'
            }, {
                height: '100%',
                duration: 2,
                ease: 'power2.out',
                scrollTrigger: {
                    trigger: '#timeline',
                    start: 'top 60%',
                    end: 'bottom 40%',
                    scrub: 1
                }
            });
        }
    }
}

// ==================== PROCESS ANIMATIONS ====================
// ==================== PROCESS ANIMATIONS ====================
function initializeProcessAnimations() {
    const processSteps = document.querySelectorAll('.process-step');
    const processTimelineProgress = document.querySelector('.timeline-progress');
    let activeProcessStep = 0;
    let processInterval;

    // Check if mobile device
    const isMobile = window.innerWidth <= 768;

    // Mobile: Show everything immediately without animations
    if (isMobile) {
        processSteps.forEach((step, index) => {
            step.classList.add('animate-in');
            step.style.opacity = '1';
            step.style.transform = 'translateX(0) translateY(0)';
            
            const stepContent = step.querySelector('.process-content');
            if (stepContent) {
                stepContent.style.opacity = '1';
                stepContent.style.transform = 'translateX(0) translateY(0)';
            }
        });

        if (processTimelineProgress) {
            processTimelineProgress.style.height = '100%';
        }

        console.log('Process timeline: Mobile mode - all steps shown immediately');
        return; // Skip desktop animations
    }

    // Desktop: Use intersection observer and animations
    const processObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-in');
                updateProcessProgress();

                // Start live animation when first step is visible
                if (entry.target.dataset.step === '1' && !processInterval) {
                    startLiveProcessAnimation();
                }
            }
        });
    }, {
        threshold: 0.3,
        rootMargin: '-50px 0px'
    });

    // Observe all process steps
    processSteps.forEach(step => {
        processObserver.observe(step);
    });

    // Function to update process timeline progress
    function updateProcessProgress() {
        const visibleSteps = document.querySelectorAll('.process-step.animate-in');
        const totalSteps = processSteps.length;
        const progressPercentage = (visibleSteps.length / totalSteps) * 100;

        if (processTimelineProgress) {
            processTimelineProgress.style.height = progressPercentage + '%';
        }
    }

    // Live process animation for desktop
    function startLiveProcessAnimation() {
        processInterval = setInterval(() => {
            // Highlight current step
            processSteps.forEach((step, index) => {
                const stepContent = step.querySelector('.process-content');
                const stepNumber = step.querySelector('.process-step-number');
                const stepIcon = step.querySelector('.process-icon');

                if (!stepContent) return;

                if (index === activeProcessStep) {
                    // Active step styling
                    stepContent.style.transform = 'scale(1.01)';
                    stepContent.style.boxShadow = '0 15px 35px rgba(0, 0, 0, 0.12)';
                    if (stepNumber) stepNumber.style.animation = 'pulse-number 1s ease-in-out infinite';
                    if (stepIcon) stepIcon.style.transform = 'scale(1.05) rotate(3deg)';
                } else {
                    // Reset other steps
                    stepContent.style.transform = 'scale(1)';
                    stepContent.style.boxShadow = '';
                    if (stepNumber) stepNumber.style.animation = '';
                    if (stepIcon) stepIcon.style.transform = 'scale(1) rotate(0deg)';
                }
            });

            // Move to next step
            activeProcessStep = (activeProcessStep + 1) % processSteps.length;
        }, 4000);
    }

    // Stop animation when user scrolls away (desktop only)
    const processSection = document.getElementById('timeline');
    if (processSection) {
        const sectionObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (!entry.isIntersecting && processInterval) {
                    clearInterval(processInterval);
                    processInterval = null;

                    // Reset all steps
                    processSteps.forEach(step => {
                        const stepContent = step.querySelector('.process-content');
                        const stepNumber = step.querySelector('.process-step-number');
                        const stepIcon = step.querySelector('.process-icon');

                        if (stepContent) {
                            stepContent.style.transform = 'scale(1)';
                            stepContent.style.boxShadow = '';
                        }
                        if (stepNumber) stepNumber.style.animation = '';
                        if (stepIcon) stepIcon.style.transform = 'scale(1) rotate(0deg)';
                    });
                }
            });
        }, {
            threshold: 0.1
        });

        sectionObserver.observe(processSection);
    }

    // GSAP ScrollTrigger integration (if available) - Desktop only
    if (typeof gsap !== 'undefined' && gsap.registerPlugin) {
        gsap.registerPlugin(ScrollTrigger);

        // Animate process title
        const processTitle = document.querySelector('#timeline h2');
        if (processTitle) {
            gsap.fromTo(processTitle, {
                opacity: 0,
                y: 50
            }, {
                opacity: 1,
                y: 0,
                duration: 1,
                ease: 'power3.out',
                scrollTrigger: {
                    trigger: processTitle,
                    start: 'top 80%',
                    end: 'bottom 20%',
                    toggleActions: 'play none none reverse'
                }
            });
        }

        // Process steps are visible by default (no animation needed)
        // The CSS already handles the visibility

        // Process timeline progress animation
        if (processTimelineProgress) {
            gsap.fromTo(processTimelineProgress, {
                height: '0%'
            }, {
                height: '100%',
                duration: 2,
                ease: 'power2.out',
                scrollTrigger: {
                    trigger: '#timeline',
                    start: 'top 60%',
                    end: 'bottom 40%',
                    scrub: 1
                }
            });
        }
    }

    // Add click handlers for process steps
    processSteps.forEach((step, index) => {
        step.addEventListener('click', () => {
            activeProcessStep = index;

            // Immediately highlight clicked step
            processSteps.forEach((s, i) => {
                const stepContent = s.querySelector('.process-content');
                const stepNumber = s.querySelector('.process-step-number');
                const stepIcon = s.querySelector('.process-icon');

                if (!stepContent) return;

                if (i === index) {
                    stepContent.style.transform = 'scale(1.02)';
                    stepContent.style.boxShadow = '0 25px 50px rgba(0, 0, 0, 0.15)';
                    if (stepNumber) stepNumber.style.animation = 'pulse-number 1s ease-in-out infinite';
                    if (stepIcon) stepIcon.style.transform = 'scale(1.1) rotate(5deg)';
                } else {
                    stepContent.style.transform = 'scale(1)';
                    stepContent.style.boxShadow = '';
                    if (stepNumber) stepNumber.style.animation = '';
                    if (stepIcon) stepIcon.style.transform = 'scale(1) rotate(0deg)';
                }
            });
        });
    });

    // Handle window resize
    let resizeTimeout;
    window.addEventListener('resize', () => {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(() => {
            const newIsMobile = window.innerWidth <= 768;
            if (newIsMobile !== isMobile) {
                location.reload(); // Reload to reinitialize with correct mode
            }
        }, 250);
    });

    console.log('Process timeline: Desktop mode initialized with animations');
}
// ==================== MOBILE NAVIGATION ====================
function initializeMobileNavigation() {
    const mobileMenuBtn = document.getElementById('mobile-menu-btn');
    const mobileMenu = document.getElementById('mobile-menu');

    if (mobileMenuBtn && mobileMenu) {
        mobileMenuBtn.addEventListener('click', function() {
            // Toggle the hidden class
            mobileMenu.classList.toggle('hidden');

            // Update ARIA attributes for accessibility
            const isOpen = !mobileMenu.classList.contains('hidden');
            mobileMenuBtn.setAttribute('aria-expanded', isOpen);

            // Change icon based on menu state
            const icon = mobileMenuBtn.querySelector('i');
            if (icon) {
                if (isOpen) {
                    icon.setAttribute('data-lucide', 'x');
                } else {
                    icon.setAttribute('data-lucide', 'menu');
                }
                // Refresh Lucide icons
                if (typeof lucide !== 'undefined') {
                    lucide.createIcons();
                }
            }
        });

        // Close mobile menu when clicking on nav links
        const mobileNavLinks = mobileMenu.querySelectorAll('a');
        mobileNavLinks.forEach(link => {
            link.addEventListener('click', function() {
                mobileMenu.classList.add('hidden');
                mobileMenuBtn.setAttribute('aria-expanded', 'false');

                // Reset icon
                const icon = mobileMenuBtn.querySelector('i');
                if (icon) {
                    icon.setAttribute('data-lucide', 'menu');
                    if (typeof lucide !== 'undefined') {
                        lucide.createIcons();
                    }
                }
            });
        });

        // Close mobile menu when clicking outside
        document.addEventListener('click', function(event) {
            const isClickInsideMenu = mobileMenu.contains(event.target);
            const isClickOnButton = mobileMenuBtn.contains(event.target);

            if (!isClickInsideMenu && !isClickOnButton && !mobileMenu.classList.contains('hidden')) {
                mobileMenu.classList.add('hidden');
                mobileMenuBtn.setAttribute('aria-expanded', 'false');

                // Reset icon
                const icon = mobileMenuBtn.querySelector('i');
                if (icon) {
                    icon.setAttribute('data-lucide', 'menu');
                    if (typeof lucide !== 'undefined') {
                        lucide.createIcons();
                    }
                }
            }
        });
    }
}

// ==================== ADVANCED CAROUSEL ====================
function initializeAdvancedCarousel() {
    class AdvancedCarousel {
        constructor() {
            console.log('AdvancedCarousel constructor called');

            this.carousel = document.getElementById('carousel');
            this.dotsContainer = document.getElementById('dotsContainer');
            this.prevBtn = document.getElementById('prevBtn');
            this.nextBtn = document.getElementById('nextBtn');

            console.log('Carousel elements found:', {
                carousel: !!this.carousel,
                dotsContainer: !!this.dotsContainer,
                prevBtn: !!this.prevBtn,
                nextBtn: !!this.nextBtn
            });

            if (!this.carousel) {
                console.log('No carousel found, exiting');
                return;
            }

            this.currentIndex = 0;
            this.cardWidth = 370;
            this.visibleDotsCount = 5;
            this.dotsStartIndex = 0;

            console.log('Initializing carousel...');
            this.init();
        }

        init() {
            this.cards = Array.from(this.carousel.children);
            this.totalSlides = this.cards.length;

            if (this.totalSlides === 0) return;

            this.setupImages();
            this.generateDots();
            this.updateCarousel();
            this.attachEventListeners();
            this.updateNavigationButtons();
        }

        setupImages() {
            console.log('Setting up images for', this.cards.length, 'cards');

            this.cards.forEach((card, index) => {
                const placeholder = card.querySelector('.advanced-loading-placeholder');
                const img = card.querySelector('img');

                if (img && placeholder) {
                    console.log('Setting up image for card', index, 'with src:', img.src);

                    // Check if image is already loaded
                    if (img.complete && img.naturalHeight !== 0) {
                        console.log('Image already loaded for card', index);
                        placeholder.style.display = 'none';
                        return;
                    }

                    // Show image when loaded, hide placeholder
                    img.onload = () => {
                        console.log('Image loaded successfully for card', index);
                        if (placeholder) {
                            placeholder.style.display = 'none';
                        }
                    };

                    // Handle image load error
                    img.onerror = () => {
                        console.error('Image failed to load for card', index, 'src:', img.src);
                        if (placeholder) {
                            placeholder.textContent = 'Failed to load';
                            placeholder.style.background = 'linear-gradient(135deg, #ff6b6b 0%, #ee5a24 100%)';
                        }
                    };

                    // For already loaded images
                    setTimeout(() => {
                        if (img.complete && img.naturalHeight !== 0) {
                            console.log('Image was already loaded (delayed check) for card', index);
                            if (placeholder) {
                                placeholder.style.display = 'none';
                            }
                        }
                    }, 100);
                } else {
                    console.warn('Missing image or placeholder for card', index);
                }
            });
        }

        generateDots() {
            if (!this.dotsContainer) return;

            this.dotsContainer.innerHTML = '';

            for (let i = 0; i < this.totalSlides; i++) {
                const dot = document.createElement('div');
                dot.className = 'advanced-dot';
                dot.dataset.index = i;
                dot.addEventListener('click', () => this.goToSlide(i));
                this.dotsContainer.appendChild(dot);
            }

            this.dots = Array.from(this.dotsContainer.children);
            this.updateDotsVisibility();
        }

        updateDotsVisibility() {
            if (!this.dots) return;

            // Hide all dots first
            this.dots.forEach(dot => {
                dot.style.display = 'none';
                dot.classList.remove('center', 'adjacent', 'outer');
            });

            // Calculate the range of dots to show
            let start = Math.max(0, this.currentIndex - 2);
            let end = Math.min(this.totalSlides - 1, start + 4);

            // Adjust if we're near the end
            if (end - start < 4 && this.totalSlides >= 5) {
                start = Math.max(0, end - 4);
            }

            // Show and style the visible dots
            for (let i = start; i <= end; i++) {
                const dot = this.dots[i];
                if (dot) {
                    dot.style.display = 'block';

                    const relativePos = i - start;
                    const centerPos = Math.floor((end - start) / 2);

                    if (i === this.currentIndex) {
                        dot.classList.add('center');
                    } else if (Math.abs(relativePos - centerPos) === 1) {
                        dot.classList.add('adjacent');
                    } else {
                        dot.classList.add('outer');
                    }
                }
            }
        }

        updateCarousel() {
            if (!this.carousel) return;

            const translateX = -this.currentIndex * this.cardWidth;
            this.carousel.style.transform = `translateX(${translateX}px)`;
            this.updateDotsVisibility();
        }

        updateNavigationButtons() {
            if (this.prevBtn) this.prevBtn.disabled = this.currentIndex === 0;
            if (this.nextBtn) this.nextBtn.disabled = this.currentIndex === this.totalSlides - 1;
        }

        goToSlide(index) {
            if (index >= 0 && index < this.totalSlides) {
                this.currentIndex = index;
                this.updateCarousel();
                this.updateNavigationButtons();
            }
        }

        nextSlide() {
            if (this.currentIndex < this.totalSlides - 1) {
                this.goToSlide(this.currentIndex + 1);
            }
        }

        prevSlide() {
            if (this.currentIndex > 0) {
                this.goToSlide(this.currentIndex - 1);
            }
        }

        attachEventListeners() {
            if (this.prevBtn) {
                this.prevBtn.addEventListener('click', () => this.prevSlide());
            }
            if (this.nextBtn) {
                this.nextBtn.addEventListener('click', () => this.nextSlide());
            }

            // Card click events
            this.cards.forEach((card, index) => {
                card.addEventListener('click', (e) => {
                    // If clicked directly on the card (not the link icon)
                    if (!e.target.closest('.link-icon')) {
                        const link = card.dataset.link;
                        if (link) {
                            window.location.href = link;
                        }
                    }
                });

                // Link icon click events
                const linkIcon = card.querySelector('.link-icon');
                if (linkIcon) {
                    linkIcon.addEventListener('click', (e) => {
                        e.stopPropagation();
                        const link = card.dataset.link;
                        if (link) {
                            window.open(link, '_blank');
                        }
                    });
                }
            });

            // Keyboard navigation
            document.addEventListener('keydown', (e) => {
                if (e.key === 'ArrowLeft') {
                    this.prevSlide();
                } else if (e.key === 'ArrowRight') {
                    this.nextSlide();
                }
            });

            // Touch/swipe support for mobile
            let startX = 0;
            let isDragging = false;

            if (this.carousel) {
                this.carousel.addEventListener('touchstart', (e) => {
                    startX = e.touches[0].clientX;
                    isDragging = true;
                });

                this.carousel.addEventListener('touchmove', (e) => {
                    if (!isDragging) return;
                    e.preventDefault();
                });

                this.carousel.addEventListener('touchend', (e) => {
                    if (!isDragging) return;

                    const endX = e.changedTouches[0].clientX;
                    const diff = startX - endX;

                    if (Math.abs(diff) > 50) {
                        if (diff > 0) {
                            this.nextSlide();
                        } else {
                            this.prevSlide();
                        }
                    }

                    isDragging = false;
                });
            }
        }
    }

    new AdvancedCarousel();
}

// ==================== CONTACT FORM ====================
function initializeContactForm() {
    const contactForm = document.getElementById('contactForm');

    if (contactForm) {
        // Add real-time validation
        const inputs = contactForm.querySelectorAll('input[required]');
        inputs.forEach(input => {
            // Validate on blur
            input.addEventListener('blur', function() {
                validateField(this);
            });

            // Clear error on input
            input.addEventListener('input', function() {
                clearFieldError(this);
            });
        });

        contactForm.addEventListener('submit', function(e) {
            e.preventDefault();

            // Validate all fields
            let isValid = true;
            inputs.forEach(input => {
                if (!validateField(input)) {
                    isValid = false;
                }
            });

            if (!isValid) {
                showAlert('Erreur de validation !', 'Veuillez corriger les erreurs dans le formulaire', 'error');
                return;
            }

            // Get form data
            const formData = new FormData(this);

            // Show loading state
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<span class="flex items-center justify-center">Envoi en cours...</span>';
            submitBtn.disabled = true;

            // Send AJAX request
            fetch(this.action, {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'success') {
                        showAlert('Succès !', data.message, 'success');
                        contactForm.reset();
                        clearAllErrors();
                    } else {
                        showAlert('Erreur !', data.message || 'Une erreur est survenue', 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showAlert('Erreur !', 'Une erreur de connexion est survenue', 'error');
                })
                .finally(() => {
                    submitBtn.innerHTML = originalText;
                    submitBtn.disabled = false;
                });
        });
    }

    // Validation functions
    function validateField(field) {
        const value = field.value.trim();
        const fieldName = field.name;
        let isValid = true;
        let errorMessage = '';

        clearFieldError(field);

        if (!value) {
            errorMessage = 'Ce champ est requis';
            isValid = false;
        } else {
            switch (fieldName) {
                case 'full-name':
                    if (value.length < 2) {
                        errorMessage = 'Le nom doit contenir au moins 2 caractères';
                        isValid = false;
                    } else if (!/^[A-Za-zÀ-ÿ\s]+$/.test(value)) {
                        errorMessage = 'Le nom ne peut contenir que des lettres et espaces';
                        isValid = false;
                    }
                    break;
                case 'ville':
                    if (value.length < 2) {
                        errorMessage = 'La ville doit contenir au moins 2 caractères';
                        isValid = false;
                    } else if (!/^[A-Za-zÀ-ÿ\s-]+$/.test(value)) {
                        errorMessage = 'La ville ne peut contenir que des lettres, espaces et tirets';
                        isValid = false;
                    }
                    break;
                case 'phone-number':
                    if (!/^(\+212|0)([5-7][0-9]{8})$/.test(value)) {
                        errorMessage = 'Veuillez entrer un numéro de téléphone marocain valide';
                        isValid = false;
                    }
                    break;
                case 'email':
                    if (!/^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/.test(value)) {
                        errorMessage = 'Veuillez entrer une adresse email valide';
                        isValid = false;
                    }
                    break;
            }
        }

        if (!isValid) {
            showFieldError(field, errorMessage);
        }

        return isValid;
    }

    function showFieldError(field, message) {
        field.classList.add('border-red-500');
        field.classList.remove('border-[#01acb6]/30');
        const errorDiv = field.nextElementSibling;
        if (errorDiv && errorDiv.classList.contains('error-message')) {
            errorDiv.textContent = message;
            errorDiv.classList.remove('hidden');
        }
    }

    function clearFieldError(field) {
        field.classList.remove('border-red-500');
        field.classList.add('border-[#01acb6]/30');
        const errorDiv = field.nextElementSibling;
        if (errorDiv && errorDiv.classList.contains('error-message')) {
            errorDiv.classList.add('hidden');
            errorDiv.textContent = '';
        }
    }

    function clearAllErrors() {
        const inputs = contactForm.querySelectorAll('input');
        inputs.forEach(input => {
            clearFieldError(input);
        });
    }

    // Alert function
    function showAlert(title, message, type) {
        const alertModal = document.createElement('div');
        alertModal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-[9999] opacity-0 transition-opacity duration-300';

        const alertContent = document.createElement('div');
        alertContent.className = `bg-white rounded-2xl p-8 max-w-md mx-4 transform transition-all duration-300 scale-95 ${
            type === 'success' ? 'border-l-4 border-[#01acb6]' : 'border-l-4 border-red-500'
        }`;

        alertContent.innerHTML = `
            <div class="flex items-center mb-4">
                <div class="w-12 h-12 rounded-full flex items-center justify-center mr-4 ${
                    type === 'success' ? 'bg-[#01acb6]/10' : 'bg-red-100'
                }">
                    <i data-lucide="${type === 'success' ? 'check-circle' : 'x-circle'}"
                       class="w-6 h-6 ${type === 'success' ? 'text-[#01acb6]' : 'text-red-500'}"></i>
                </div>
                <h3 class="text-xl font-bold text-gray-800">${title}</h3>
            </div>
            <p class="text-gray-600 leading-relaxed">${message}</p>
        `;

        alertModal.appendChild(alertContent);
        document.body.appendChild(alertModal);

        // Initialize icons for the alert
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }

        // Animate in
        setTimeout(() => {
            alertModal.classList.remove('opacity-0');
            alertModal.classList.add('opacity-100');
            alertContent.classList.remove('scale-95');
            alertContent.classList.add('scale-100');
        }, 10);

        // Auto close after 1.2 seconds
        setTimeout(() => {
            alertModal.classList.remove('opacity-100');
            alertModal.classList.add('opacity-0');
            alertContent.classList.remove('scale-100');
            alertContent.classList.add('scale-95');

            setTimeout(() => {
                if (alertModal.parentNode) {
                    alertModal.remove();
                }
            }, 300);
        }, 1200);
    }
}

// ==================== QUOTE MODAL ====================
function initializeQuoteModal() {
    const devisBtn = document.getElementById('devis-btn');
    const modal = document.getElementById('devis-modal');
    const modalContent = document.getElementById('modal-content');
    const closeModal = document.getElementById('close-modal');
    const cancelBtn = document.getElementById('cancel-btn');
    const devisForm = document.getElementById('devis-form');

    // Open modal function
    function openModal() {
        if (modal && modalContent) {
            modal.classList.remove('hidden');
            modal.classList.add('flex');
            modalContent.classList.remove('scale-95', 'opacity-0');
            modalContent.classList.add('scale-100', 'opacity-100');
            document.body.style.overflow = 'hidden';
        }
    }

    // Close modal function
    function closeModalFunc() {
        if (modal && modalContent) {
            modalContent.classList.remove('scale-100', 'opacity-100');
            modalContent.classList.add('scale-95', 'opacity-0');
            setTimeout(() => {
                modal.classList.remove('flex');
                modal.classList.add('hidden');
                document.body.style.overflow = '';
            }, 300);
        }
    }

    // Event listeners
    devisBtn?.addEventListener('click', openModal);
    closeModal?.addEventListener('click', closeModalFunc);
    cancelBtn?.addEventListener('click', closeModalFunc);

    // Close on backdrop click
    modal?.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeModalFunc();
        }
    });

    // Close with Escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && modal && !modal.classList.contains('hidden')) {
            closeModalFunc();
        }
    });

    // Form submission
    devisForm?.addEventListener('submit', function(e) {
        e.preventDefault();

        const formData = new FormData(devisForm);
        const data = {
            prenom: formData.get('prenom'),
            nom: formData.get('nom'),
            email: formData.get('email'),
            telephone: formData.get('telephone'),
            message: formData.get('message')
        };

        // Create mailto link
        const subject = encodeURIComponent('Demande de devis - ' + data.prenom + ' ' + data.nom);
        const body = encodeURIComponent(
            `Bonjour,\n\n` +
            `Je souhaite obtenir un devis pour mon projet.\n\n` +
            `Mes informations :\n` +
            `- Prénom : ${data.prenom}\n` +
            `- Nom : ${data.nom}\n` +
            `- Email : ${data.email}\n` +
            `- Téléphone : ${data.telephone || 'Non renseigné'}\n\n` +
            `Description du projet :\n${data.message || 'Aucune description fournie'}\n\n` +
            `Cordialement,\n${data.prenom} ${data.nom}`
        );

        const mailtoLink = `mailto:contact@maxmind.ma?subject=${subject}&body=${body}`;
        window.location.href = mailtoLink;

        // Close modal and reset form
        setTimeout(() => {
            closeModalFunc();
            devisForm.reset();
        }, 1000);
    });
}

// ==================== PROJECT FILTERS ====================
function initializeProjectFilters() {
    const filterButtons = document.querySelectorAll('.filter-btn');
    const projectCards = document.querySelectorAll('.project-card');
    const projectsGrid = document.querySelector('.projects-grid');

    // Update project count (console only)
    function updateProjectCount() {
        const visibleProjects = document.querySelectorAll('.project-card:not([style*="display: none"])').length;
        const totalProjects = projectCards.length;

        console.log(`${visibleProjects} projets sur ${totalProjects} affichés`);
    }

    // Filter projects
    function filterProjects(category) {
        if (projectsGrid) {
            projectsGrid.style.opacity = '0.7';
        }

        setTimeout(() => {
            projectCards.forEach((card) => {
                const cardCategory = card.getAttribute('data-category');
                const shouldShow = category === 'all' || cardCategory === category;

                if (shouldShow) {
                    card.style.display = 'flex';
                    card.style.opacity = '0';
                    setTimeout(() => {
                        card.style.opacity = '1';
                    }, 50);
                } else {
                    card.style.display = 'none';
                }
            });

            if (projectsGrid) {
                projectsGrid.style.opacity = '1';
            }

            updateProjectCount();
        }, 150);
    }

    // Filter button event listeners
    filterButtons.forEach(button => {
        button.addEventListener('click', () => {
            // Remove active class from all buttons
            filterButtons.forEach(btn => {
                btn.classList.remove('active-filter');
                btn.classList.add('bg-white', 'text-slate-700', 'border-slate-200');
                btn.classList.remove('bg-[#01acb6]', 'text-white', 'border-[#01acb6]');
            });

            // Add active class to clicked button
            button.classList.add('active-filter');
            button.classList.remove('bg-white', 'text-slate-700', 'border-slate-200');
            button.classList.add('bg-[#01acb6]', 'text-white', 'border-[#01acb6]');

            // Click feedback animation
            button.style.transform = 'scale(0.95)';
            setTimeout(() => {
                button.style.transform = 'scale(1)';
            }, 100);

            // Perform filtering
            const category = button.getAttribute('data-filter');
            filterProjects(category);
        });
    });

    // Initialize smooth transitions
    projectCards.forEach(card => {
        card.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
    });

    if (projectsGrid) {
        projectsGrid.style.transition = 'opacity 0.3s ease';
    }

    filterButtons.forEach(btn => {
        btn.style.transition = 'all 0.3s ease, transform 0.1s ease';
    });

    // Initialize project count
    updateProjectCount();
}

// ==================== IMAGE POPUPS ====================
function initializeImagePopups() {
    // Image modal functions
    window.openModal = function(imageSrc, title) {
        const modal = document.getElementById('imageModal');
        const modalImage = document.getElementById('modalImage');
        const modalTitle = document.getElementById('modalTitle');

        // Use preloaded images instead of loading from source
        let preloadedImg;
        switch (imageSrc) {
            case 'design/2835 copy.png?w=800&h=600&fit=crop':
                preloadedImg = document.getElementById('preload-img1');
                break;
            case 'design/dwdwdw.png?w=800&h=600&fit=crop':
                preloadedImg = document.getElementById('preload-img2');
                break;
            case 'newImages/COVER-COPIMA.png?w=800&h=600&fit=crop':
                preloadedImg = document.getElementById('preload-img3');
                break;
            case 'newImages/COVER-SESAME.png?w=800&h=600&fit=crop':
                preloadedImg = document.getElementById('preload-img4');
                break;
            default:
                modalImage.src = imageSrc;
        }

        if (preloadedImg) {
            modalImage.src = preloadedImg.src;
        }

        modalImage.alt = title;
        modalTitle.textContent = title;

        modal.classList.remove('hidden');
        document.body.style.overflow = 'hidden';
    };

    window.openPopup = function(imageSrc, title) {
        const popup = document.getElementById('imagePopup');
        const popupImage = document.getElementById('popupImage');
        const popupTitle = document.getElementById('popupTitle');

        popupImage.src = imageSrc;
        popupImage.alt = title;
        popupTitle.textContent = title;

        popup.classList.remove('hidden');
        popup.classList.add('flex');
        document.body.style.overflow = 'hidden';
    };

    window.closePopup = function() {
        const popup = document.getElementById('imagePopup');
        popup.classList.add('hidden');
        popup.classList.remove('flex');
        document.body.style.overflow = 'auto';
    };

    // Close popup when clicking outside the image
    const imagePopup = document.getElementById('imagePopup');
    if (imagePopup) {
        imagePopup.addEventListener('click', function(e) {
            if (e.target === this) {
                window.closePopup();
            }
        });
    }

    // Preload images function
    window.preloadImages = function() {
        const imageUrls = [
            'design/2835 copy.png',
            'design/dwdwdw.png',
            'newImages/COVER-COPIMA.png',
            'newImages/COVER-SESAME.png'
        ];

        imageUrls.forEach((url, index) => {
            const img = new Image();
            img.src = url;
            img.id = `preload-img${index + 1}`;
            img.style.display = 'none';
            document.body.appendChild(img);
        });
    };

    // Initialize preloading
    window.preloadImages();
    console.log('Images preloaded successfully');
}

// ==================== FOOTER FIX ====================
function initializeFooterFix() {
    console.log('Fixing footer interactivity...');

    // Force footer elements to be clickable
    const footer = document.querySelector('footer');
    if (footer) {
        footer.style.pointerEvents = 'auto';
        footer.style.zIndex = '100';
        footer.style.position = 'relative';

        console.log('Footer found and styled');

        // Fix all interactive elements in footer
        const interactiveElements = footer.querySelectorAll('a, button, iframe, [onclick]');
        console.log('Found', interactiveElements.length, 'interactive elements');

        interactiveElements.forEach((element, index) => {
            element.style.pointerEvents = 'auto';
            element.style.cursor = 'pointer';
            element.style.zIndex = '101';
            element.style.position = 'relative';

            // Special handling for Google Maps iframe
            if (element.tagName === 'IFRAME' && element.src.includes('google.com/maps')) {
                element.style.touchAction = 'auto';
                element.style.userSelect = 'auto';
                element.removeAttribute('loading');
                console.log('Google Maps iframe fixed');
            }

            // Add click event listener for debugging
            element.addEventListener('click', function(e) {
                console.log('Footer element clicked:', element);
            });
        });

        // Remove overlapping elements that might block clicks
        const overlayElements = document.querySelectorAll('[style*="position: absolute"], [style*="position: fixed"]');
        overlayElements.forEach(element => {
            if (element !== footer && !footer.contains(element)) {
                const rect = element.getBoundingClientRect();
                const footerRect = footer.getBoundingClientRect();

                if (rect.bottom > footerRect.top && rect.top < footerRect.bottom) {
                    element.style.pointerEvents = 'none';
                    console.log('Disabled pointer events for overlapping element:', element);
                }
            }
        });
    } else {
        console.error('Footer not found!');
    }

    // Additional fix on window load
    window.addEventListener('load', function() {
        setTimeout(() => {
            const footer = document.querySelector('footer');
            const iframe = footer?.querySelector('iframe');
            if (iframe) {
                iframe.style.pointerEvents = 'auto';
                iframe.style.touchAction = 'auto';
                console.log('Google Maps iframe re-enabled after load');
            }
        }, 1000);
    });
}

// ==================== LUCIDE ICONS ====================
function initializeLucideIcons() {
    // Initialize Lucide icons
    if (typeof lucide !== 'undefined') {
        lucide.createIcons();
        console.log('Lucide icons initialized');
    }
}
document.addEventListener('DOMContentLoaded', function() {
            // Initialize Lucide icons
            if (typeof lucide !== 'undefined') {
                lucide.createIcons();
            }

            // Category filtering with AJAX
            function setupCategoryFiltering() {
                const categoryButtons = document.querySelectorAll('.category-id-filter');

                categoryButtons.forEach(button => {
                    button.addEventListener('click', function(e) {
                        e.preventDefault();

                        const categoryId = this.getAttribute('data-category-id');
                        console.log('Category clicked:', categoryId);

                        // Update active button state immediately for better UX
                        updateCategoryButtonStates(categoryId);

                        // Make AJAX call
                        filterByCategory(categoryId);
                    });
                });
            }

            function updateCategoryButtonStates(activeCategoryId) {
                const buttons = document.querySelectorAll('.category-id-filter');
                buttons.forEach(btn => {
                    const btnCategoryId = btn.getAttribute('data-category-id');
                    if (btnCategoryId === activeCategoryId) {
                        btn.classList.remove('bg-slate-100', 'text-slate-700', 'hover:bg-slate-200');
                        btn.classList.add('bg-[#01acb6]', 'text-white', 'shadow-lg');
                    } else {
                        btn.classList.remove('bg-[#01acb6]', 'text-white', 'shadow-lg');
                        btn.classList.add('bg-slate-100', 'text-slate-700', 'hover:bg-slate-200');
                    }
                });
            }

            function filterByCategory(categoryId) {
                const loadingOverlay = document.getElementById('loading-overlay');
                if (loadingOverlay) {
                    loadingOverlay.classList.remove('hidden');
                }

                const baseUrl = window.location.origin + window.location.pathname.replace('/blogs.php', '');

                let apiUrl = baseUrl + '/admin/cms.php/Records?category_id=1&search_field=category id&search_value=' + encodeURIComponent(categoryId);
                fetch(apiUrl, {
                        method: 'GET',
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest',
                            'X-API-TOKEN': apikey,
                            'Accept': 'application/json'
                        }
                    })

                console.log('Making AJAX call to:', apiUrl);

                fetch('blogs.php?ajax=1&filter_category_id=' + encodeURIComponent(categoryId), {
                        method: 'GET',
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest',
                            'X-API-TOKEN': '2250c47a4397a41fe5ff8f2c2a92b240ddf46aaeded8ffddb4f29f02ffef8720',

                        }
                    })
                    .then(response => {
                        if (!response.ok) {
                            throw new Error('Network response was not ok');
                        }
                        return response.json();
                    })
                    .then(data => {
                        console.log('AJAX response:', data);

                        if (data.success) {
                            // Update the articles container
                            const postsContainer = document.getElementById('posts');
                            if (postsContainer && data.articles_html) {
                                postsContainer.innerHTML = data.articles_html;
                            }

                            // Update pagination if exists
                            const paginationContainer = document.querySelector('.pagination-container');
                            if (paginationContainer && data.pagination_html) {
                                paginationContainer.innerHTML = data.pagination_html;
                            }

                            // Update URL without page reload
                            const newUrl = new URL(window.location);
                            if (categoryId) {
                                newUrl.searchParams.set('filter_category_id', categoryId);
                            } else {
                                newUrl.searchParams.delete('filter_category_id');
                            }
                            newUrl.searchParams.delete('page'); // Reset to first page
                            window.history.pushState({}, '', newUrl);

                            // Re-initialize Lucide icons for new content
                            if (typeof lucide !== 'undefined') {
                                lucide.createIcons();
                            }
                        } else {
                            console.error('Filter request failed:', data);
                        }
                    })
                    .catch(error => {
                        console.error('Error filtering categories:', error);
                        // Show error message to user
                        const postsContainer = document.getElementById('posts');
                        if (postsContainer) {
                            postsContainer.innerHTML = '<div class="col-span-2 text-center py-12"><div class="text-red-500 mb-4"><i data-lucide="alert-circle" class="w-16 h-16 mx-auto mb-4"></i></div><h3 class="text-xl font-semibold text-red-600 mb-2">Erreur de chargement</h3><p class="text-slate-500">Une erreur est survenue lors du filtrage. Veuillez réessayer.</p></div>';
                        }
                    })
                    .finally(() => {
                        // Hide loading overlay
                        if (loadingOverlay) {
                            loadingOverlay.classList.add('hidden');
                        }
                    });
            }

            // Search functionality
            function setupSearchFunctionality() {
                const searchInput = document.getElementById('search-input');
                const searchBtn = document.getElementById('search-btn');
                const clearSearchBtn = document.getElementById('clear-search-btn');
                const sortSelect = document.getElementById('sort-select');

                if (searchBtn) {
                    searchBtn.addEventListener('click', performSearch);
                }

                if (searchInput) {
                    searchInput.addEventListener('keypress', function(e) {
                        if (e.key === 'Enter') {
                            e.preventDefault();
                            performSearch();
                        }
                    });
                }

                if (clearSearchBtn) {
                    clearSearchBtn.addEventListener('click', function() {
                        if (searchInput) {
                            searchInput.value = '';
                        }
                        performSearch();
                    });
                }

                if (sortSelect) {
                    sortSelect.addEventListener('change', performSearch);
                }
            }

            function performSearch() {
                const searchInput = document.getElementById('search-input');
                const sortSelect = document.getElementById('sort-select');
                const clearSearchBtn = document.getElementById('clear-search-btn');
                const searchSpinner = document.getElementById('search-spinner');

                const searchValue = searchInput ? searchInput.value.trim() : '';
                const sortValue = sortSelect ? sortSelect.value : 'new';

                // Show/hide clear button
                if (clearSearchBtn) {
                    if (searchValue) {
                        clearSearchBtn.classList.remove('hidden');
                    } else {
                        clearSearchBtn.classList.add('hidden');
                    }
                }

                const loadingOverlay = document.getElementById('loading-overlay');

                if (loadingOverlay) {
                    loadingOverlay.classList.remove('hidden');
                }
                if (searchSpinner) {
                    searchSpinner.classList.remove('hidden');
                }

                // Build URL parameters
                const params = new URLSearchParams();
                params.set('ajax', '1');
                if (searchValue) params.set('search', searchValue);
                if (sortValue && sortValue !== 'new') params.set('sort', sortValue);

                fetch('blogs.php?' + params.toString(), {
                        method: 'GET',
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest'
                        }
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            const postsContainer = document.getElementById('posts');
                            if (postsContainer && data.articles_html) {
                                postsContainer.innerHTML = data.articles_html;
                            }

                            // Update pagination
                            const paginationContainer = document.querySelector('.pagination-container');
                            if (paginationContainer && data.pagination_html) {
                                paginationContainer.innerHTML = data.pagination_html;
                            }

                            // Update URL
                            const newUrl = new URL(window.location);
                            if (searchValue) {
                                newUrl.searchParams.set('search', searchValue);
                            } else {
                                newUrl.searchParams.delete('search');
                            }
                            if (sortValue && sortValue !== 'new') {
                                newUrl.searchParams.set('sort', sortValue);
                            } else {
                                newUrl.searchParams.delete('sort');
                            }
                            newUrl.searchParams.delete('page');
                            window.history.pushState({}, '', newUrl);


                            // Re-initialize icons
                            if (typeof lucide !== 'undefined') {
                                lucide.createIcons();
                            }
                        }
                    })
                    .catch(error => {
                        console.error('Search error:', error);
                    })
                    .finally(() => {
                        if (loadingOverlay) {
                            loadingOverlay.classList.add('hidden');
                        }
                        if (searchSpinner) {
                            searchSpinner.classList.add('hidden');
                        }
                    });
            }

            // Tag filtering
            window.searchByTag = function(tag) {
                const searchInput = document.getElementById('search-input');
                if (searchInput) {
                    searchInput.value = tag;
                    performSearch();
                }
            };

            // Clear filters functions
            window.clearCategoryFilter = function() {
                filterByCategory('');
            };

            window.clearTagFilter = function() {
                const newUrl = new URL(window.location);
                newUrl.searchParams.delete('tag');
                window.location.href = newUrl.toString();
            };

            window.clearSearch = function() {
                const searchInput = document.getElementById('search-input');
                if (searchInput) {
                    searchInput.value = '';
                }
                performSearch();
            };

            // Pagination handling
            function setupPaginationHandling() {
                document.addEventListener('click', function(e) {
                    if (e.target.classList.contains('pagination-btn') || e.target.closest('.pagination-btn')) {
                        e.preventDefault();

                        const button = e.target.classList.contains('pagination-btn') ? e.target : e.target.closest('.pagination-btn');
                        const page = button.getAttribute('data-page');

                        if (page) {
                            // Get current filter parameters
                            const urlParams = new URLSearchParams(window.location.search);
                            const categoryId = urlParams.get('filter_category_id') || '';
                            const searchValue = urlParams.get('search') || '';
                            const sortValue = urlParams.get('sort') || 'new';

                            // Build new request
                            const params = new URLSearchParams();
                            params.set('ajax', '1');
                            params.set('page', page);
                            if (categoryId) params.set('filter_category_id', categoryId);
                            if (searchValue) params.set('search', searchValue);
                            if (sortValue && sortValue !== 'new') params.set('sort', sortValue);

                            const loadingOverlay = document.getElementById('loading-overlay');
                            if (loadingOverlay) {
                                loadingOverlay.classList.remove('hidden');
                            }

                            fetch('blogs.php?' + params.toString(), {
                                    method: 'GET',
                                    headers: {
                                        'X-Requested-With': 'XMLHttpRequest'
                                    }
                                })
                                .then(response => response.json())
                                .then(data => {
                                    if (data.success) {
                                        const postsContainer = document.getElementById('posts');
                                        if (postsContainer && data.articles_html) {
                                            postsContainer.innerHTML = data.articles_html;
                                        }

                                        const paginationContainer = document.querySelector('.pagination-container');
                                        if (paginationContainer && data.pagination_html) {
                                            paginationContainer.innerHTML = data.pagination_html;
                                        }

                                        // Update URL
                                        const newUrl = new URL(window.location);
                                        newUrl.searchParams.set('page', page);
                                        window.history.pushState({}, '', newUrl);

                                        // Scroll to top of articles
                                        const postsElement = document.getElementById('posts');
                                        if (postsElement) {
                                            postsElement.scrollIntoView({
                                                behavior: 'smooth',
                                                block: 'start'
                                            });
                                        }

                                        // Re-initialize icons
                                        if (typeof lucide !== 'undefined') {
                                            lucide.createIcons();
                                        }
                                    }
                                })
                                .catch(error => {
                                    console.error('Pagination error:', error);
                                })
                                .finally(() => {
                                    if (loadingOverlay) {
                                        loadingOverlay.classList.add('hidden');
                                    }
                                });
                        }
                    }
                });
            }

            // Initialize all functionality
            setupCategoryFiltering();
            setupSearchFunctionality();
            setupPaginationHandling();

            console.log('Blog filtering functionality initialized');
        });