<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: admin-login.php");
    exit();
}

// Handle logout
if (isset($_POST['logout'])) {
    session_destroy();
    header("Location: admin-login.php");
    exit();
}

// Initialize variables
$error = null;
$submissions = [];
$cities = [];
$total_records = 0;
$total_pages = 1;

// Get stored filters from session or initialize them
$date_from = isset($_SESSION['filters']['date_from']) ? $_SESSION['filters']['date_from'] : '';
$date_to = isset($_SESSION['filters']['date_to']) ? $_SESSION['filters']['date_to'] : '';
$email = isset($_SESSION['filters']['email']) ? $_SESSION['filters']['email'] : '';
$ville = isset($_SESSION['filters']['ville']) ? $_SESSION['filters']['ville'] : '';
$page = isset($_SESSION['filters']['page']) ? $_SESSION['filters']['page'] : 1;
$records_per_page = isset($_SESSION['rows_per_page']) ? $_SESSION['rows_per_page'] : 10;

// Process form submissions
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Store the rows per page
    if (isset($_POST['rows_per_page'])) {
        $records_per_page = (int)$_POST['rows_per_page'];
        $_SESSION['rows_per_page'] = $records_per_page;
    }
    
    // Reset filters if reset button was clicked
    if (isset($_POST['reset_filters'])) {
        $_SESSION['filters'] = [
            'date_from' => '',
            'date_to' => '',
            'email' => '',
            'ville' => '',
            'page' => 1
        ];
    } else {
        // Store filter parameters in session
        $_SESSION['filters'] = [
            'date_from' => isset($_POST['date_from']) ? trim($_POST['date_from']) : $date_from,
            'date_to' => isset($_POST['date_to']) ? trim($_POST['date_to']) : $date_to,
            'email' => isset($_POST['email']) ? trim($_POST['email']) : $email,
            'ville' => isset($_POST['ville']) ? trim($_POST['ville']) : $ville,
            'page' => isset($_POST['page']) ? (int)$_POST['page'] : 1
        ];
    }
    
    // Handle Excel export
    if (isset($_POST['export'])) {
        $_SESSION['export'] = true;
    } else {
        $_SESSION['export'] = false;
    }
    
    // Redirect to the same page to prevent form resubmission
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Update variables from session after redirect
$date_from = isset($_SESSION['filters']['date_from']) ? $_SESSION['filters']['date_from'] : '';
$date_to = isset($_SESSION['filters']['date_to']) ? $_SESSION['filters']['date_to'] : '';
$email = isset($_SESSION['filters']['email']) ? $_SESSION['filters']['email'] : '';
$ville = isset($_SESSION['filters']['ville']) ? $_SESSION['filters']['ville'] : '';
$page = isset($_SESSION['filters']['page']) ? $_SESSION['filters']['page'] : 1;

try {
    require_once 'config/connect.php';
    
    // Get all available cities for dropdown
    $cities_query = "SELECT DISTINCT ville FROM bo_contact WHERE ville IS NOT NULL AND ville != '' ORDER BY ville";
    $cities_stmt = $pdo->prepare($cities_query);
    $cities_stmt->execute();
    $cities = $cities_stmt->fetchAll(PDO::FETCH_COLUMN);

    // Handle export if session flag is set
    if (isset($_SESSION['export']) && $_SESSION['export']) {
        // Clear the export flag
        $_SESSION['export'] = false;
        
        // Set headers for Excel download
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment;filename="submissions_export.xls"');
        header('Cache-Control: max-age=0');
        
        // Print Excel content
        echo "ID\tNom\tEmail\tTéléphone\tVille\tCampagne\tDate d'ajout\n";
        
        // Get all records for export (without pagination)
        $where_conditions = [];
        $params = [];

        if ($email) {
            $where_conditions[] = "email LIKE ?";
            $params[] = "%$email%";
        }

        if ($ville) {
            $where_conditions[] = "ville LIKE ?";
            $params[] = "%$ville%";
        }

        if ($date_from) {
            $where_conditions[] = "date_ajout >= ?";
            $params[] = $date_from . " 00:00:00";
        }

        if ($date_to) {
            $where_conditions[] = "date_ajout <= ?";
            $params[] = $date_to . " 23:59:59";
        }

        $where_clause = $where_conditions ? "WHERE " . implode(" AND ", $where_conditions) : "";
        
        $query = "SELECT * FROM bo_contact $where_clause ORDER BY date_ajout DESC";
        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        
        while ($row = $stmt->fetch()) {
            echo implode("\t", [
                $row['id'],
                $row['nom'],
                $row['email'],
                $row['telephone'],
                $row['ville'],
                $row['compagne'] ?? '',
                date('d F Y H:i', strtotime($row['date_ajout']))
            ]) . "\n";
        }
        exit;
    }

    // Build query for normal view
    $where_conditions = [];
    $params = [];

    if ($email) {
        $where_conditions[] = "email LIKE ?";
        $params[] = "%$email%";
    }

    if ($ville) {
        $where_conditions[] = "ville = ?";
        $params[] = $ville;
    }

    if ($date_from) {
        $where_conditions[] = "date_ajout >= ?";
        $params[] = $date_from . " 00:00:00";
    }

    if ($date_to) {
        $where_conditions[] = "date_ajout <= ?";
        $params[] = $date_to . " 23:59:59";
    }

    $where_clause = $where_conditions ? "WHERE " . implode(" AND ", $where_conditions) : "";

    // Count total records for pagination
    $count_query = "SELECT COUNT(*) as total FROM bo_contact $where_clause";
    $stmt = $pdo->prepare($count_query);
    $stmt->execute($params);
    $total_records = $stmt->fetch()['total'];
    $total_pages = ceil($total_records / $records_per_page);
    
    // Ensure page is valid
    if ($page < 1) $page = 1;
    if ($page > $total_pages && $total_pages > 0) $page = $total_pages;
    
    $offset = ($page - 1) * $records_per_page;

    // Get records for current page
    $query = "SELECT * FROM bo_contact $where_clause ORDER BY date_ajout DESC LIMIT ? OFFSET ?";
    $stmt = $pdo->prepare($query);
    $params[] = $records_per_page;
    $params[] = $offset;
    $stmt->execute($params);
    $submissions = $stmt->fetchAll();

} catch(PDOException $e) {
    $error = "Échec de connexion: " . $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tableau de Bord Admin</title>
    <!-- Include Font Awesome from CDN -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Add Flatpickr CSS -->
    <link rel="stylesheet" href="/css/flatpickr.min.css">
    <!-- Add French locale for Flatpickr -->
    <script src="/js/fr.min.js"></script>
    
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f4f4f4;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background-color: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 20px;
            border-bottom: 1px solid #ddd;
        }
        .filters {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
            padding: 15px;
            background-color: #f8f9fa;
            border-radius: 4px;
        }
        .filter-group {
            display: flex;
            flex-direction: column;
        }
        .filter-group label {
            margin-bottom: 5px;
            color: #666;
        }
        .filter-group input,
        .filter-group select {
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        /* Flatpickr custom styles */
        .flatpickr-input {
            background-color: white !important;
            cursor: pointer;
        }
        .filter-buttons {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
            align-items: flex-end;
        }
        .btn {
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }
        .btn-primary {
            background-color: #4CAF50;
            color: white;
        }
        .btn-secondary {
            background-color: #6c757d;
            color: white;
        }
        .btn-danger {
            background-color: #dc3545;
            color: white;
        }
        .btn:hover {
            opacity: 0.9;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background-color: #f8f9fa;
            cursor: pointer;
        }
        th:hover {
            background-color: #e9ecef;
        }
        tr:hover {
            background-color: #f5f5f5;
        }
        .status {
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 0.9em;
        }
        .status-pending {
            background-color: #ffd700;
            color: #000;
        }
        .status-processed {
            background-color: #28a745;
            color: white;
        }
        .status-error {
            background-color: #dc3545;
            color: white;
        }
        .error {
            color: #dc3545;
            padding: 10px;
            margin-bottom: 20px;
            border: 1px solid #dc3545;
            border-radius: 4px;
            background-color: #f8d7da;
        }
        .pagination {
            display: flex;
            justify-content: center;
            gap: 10px;
            margin-top: 20px;
        }
        .pagination a {
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            text-decoration: none;
            color: #333;
        }
        .pagination a.active {
            background-color: #4CAF50;
            color: white;
            border-color: #4CAF50;
        }
        .pagination a:hover:not(.active) {
            background-color: #f5f5f5;
        }
        .status-badge {
            padding: 5px 10px;
            border-radius: 15px;
            font-size: 0.8em;
        }
        .action-buttons {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }
        .rows-per-page {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .rows-per-page select {
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .export-btn {
            background-color: #28a745;
            color: white;
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }
        .export-btn:hover {
            opacity: 0.9;
        }
        .stats-bar {
            display: flex;
            justify-content: space-between;
            background-color: #f8f9fa;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 4px;
        }
        .stat-item {
            text-align: center;
        }
        .stat-value {
            font-size: 1.5em;
            font-weight: bold;
            color: #4CAF50;
        }
        .stat-label {
            color: #666;
            font-size: 0.9em;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Soumissions de Formulaire</h1>
            <form method="POST" style="display: inline;">
                <button type="submit" name="logout" value="1" class="btn btn-danger">
                    <i class="fas fa-sign-out-alt"></i> Déconnexion
                </button>
            </form>
        </div>

        <?php if (isset($error)): ?>
            <div class="error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <div class="stats-bar">
            <div class="stat-item">
                <div class="stat-value"><?php echo $total_records; ?></div>
                <div class="stat-label">Total soumissions</div>
            </div>
        </div>

        <div style="display: flex; justify-content: space-between; align-items: center;" class="action-buttons">
            <div class="rows-per-page">
                <label for="rows_per_page">Lignes par page:</label>
                <form method="POST" id="rowsForm">
                    <select id="rows_per_page" name="rows_per_page" onchange="this.form.submit()">
                        <option value="10" <?php echo $records_per_page == 10 ? 'selected' : ''; ?>>10</option>
                        <option value="25" <?php echo $records_per_page == 25 ? 'selected' : ''; ?>>25</option>
                        <option value="50" <?php echo $records_per_page == 50 ? 'selected' : ''; ?>>50</option>
                        <option value="100" <?php echo $records_per_page == 100 ? 'selected' : ''; ?>>100</option>
                    </select>
                </form>
            </div>
            <form method="POST">
                <button type="submit" name="export" value="1" class="export-btn">
                    <i class="fas fa-file-excel"></i> Exporter vers Excel
                </button>
            </form>
        </div>

        <form method="POST" class="filters">
            <div class="filter-group">
                <label for="email"><i class="fas fa-envelope"></i> Email:</label>
                <input type="text" id="email" name="email" value="<?php echo htmlspecialchars($email); ?>" placeholder="Rechercher par email...">
            </div>
            <div class="filter-group">
                <label for="ville"><i class="fas fa-city"></i> Ville:</label>
                <select id="ville" name="ville">
                    <option value="">Toutes les villes</option>
                    <?php foreach ($cities as $city): ?>
                        <option value="<?php echo htmlspecialchars($city); ?>" <?php echo $ville === $city ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($city); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="filter-group">
                <label for="date_from"><i class="fas fa-calendar-alt"></i> Date début:</label>
                <input type="text" id="date_from" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>" class="flatpickr-date" placeholder="Sélectionner une date...">
            </div>
            <div class="filter-group">
                <label for="date_to"><i class="fas fa-calendar-alt"></i> Date fin:</label>
                <input type="text" id="date_to" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>" class="flatpickr-date" placeholder="Sélectionner une date...">
            </div>
            <div class="filter-buttons">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-filter"></i> Filtrer
                </button>
                <button type="submit" name="reset_filters" value="1" class="btn btn-secondary">
                    <i class="fas fa-undo"></i> Réinitialiser
                </button>
            </div>
        </form>

        <table>
            <thead>
                <tr>
                    <th><i class="fas fa-hashtag"></i> ID</th>
                    <th><i class="fas fa-user"></i> Nom</th>
                    <th><i class="fas fa-envelope"></i> Email</th>
                    <th><i class="fas fa-phone"></i> Téléphone</th>
                    <th><i class="fas fa-city"></i> Ville</th>
                    <th><i class="fas fa-bullhorn"></i> Campagne</th>
                    <th><i class="fas fa-calendar"></i> Date d'ajout</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($submissions) > 0): ?>
                    <?php foreach ($submissions as $submission): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($submission['id']); ?></td>
                            <td><?php echo htmlspecialchars($submission['nom']); ?></td>
                            <td><?php echo htmlspecialchars($submission['email']); ?></td>
                            <td><?php echo htmlspecialchars($submission['telephone']); ?></td>
                            <td><?php echo htmlspecialchars($submission['ville']); ?></td>
                            <td><?php echo htmlspecialchars($submission['compagne'] ?? ''); ?></td>
                            <td><?php 
                                setlocale(LC_TIME, 'fr_FR.UTF-8', 'fr_FR', 'fr');
                                echo htmlspecialchars(date('d F Y H:i', strtotime($submission['date_ajout']))); 
                            ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="7" style="text-align: center;">
                            <i class="fas fa-search"></i> Aucun résultat trouvé
                        </td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>

        <?php if ($total_pages > 1): ?>
            <div class="pagination">
                <?php if ($page > 1): ?>
                    <a href="#" onclick="submitPageForm(1); return false;">
                        <i class="fas fa-angle-double-left"></i>
                    </a>
                    <a href="#" onclick="submitPageForm(<?php echo ($page - 1); ?>); return false;">
                        <i class="fas fa-angle-left"></i>
                    </a>
                <?php endif; ?>

                <?php
                $start_page = max(1, $page - 2);
                $end_page = min($total_pages, $page + 2);

                for ($i = $start_page; $i <= $end_page; $i++): ?>
                    <a href="#" onclick="submitPageForm(<?php echo $i; ?>); return false;" 
                       class="<?php echo $i === $page ? 'active' : ''; ?>">
                        <?php echo $i; ?>
                    </a>
                <?php endfor; ?>

                <?php if ($page < $total_pages): ?>
                    <a href="#" onclick="submitPageForm(<?php echo ($page + 1); ?>); return false;">
                        <i class="fas fa-angle-right"></i>
                    </a>
                    <a href="#" onclick="submitPageForm(<?php echo $total_pages; ?>); return false;">
                        <i class="fas fa-angle-double-right"></i>
                    </a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- Page navigation form -->
    <form id="pageForm" method="POST" style="display: none;">
        <input type="hidden" id="page_input" name="page" value="1">
    </form>

    <!-- Add Flatpickr JavaScript -->
    <script src="/js/flatpickr.min.js"></script>
    
    <script>
    // Initialize Flatpickr for date inputs
    document.addEventListener('DOMContentLoaded', function() {
        flatpickr(".flatpickr-date", {
            dateFormat: "Y-m-d",
            locale: "fr",
            allowInput: true,
            altInput: true,
            altFormat: "d F Y",
            disableMobile: "true"
        });
    });
    
    function submitPageForm(page) {
        document.getElementById('page_input').value = page;
        document.getElementById('pageForm').submit();
    }
    </script>
</body>
</html>