// Canvas Animation System for MaxMind Hero Section
// Brand colors
const BRAND_COLORS = {
    primary: '#01acb6',
    secondary: '#EF234B',
    primaryRgb: [1, 172, 182],
    secondaryRgb: [239, 35, 75]
};

// Performance settings with device detection
const PERFORMANCE_SETTINGS = {
    maxParticles: 30,
    maxGeometryShapes: 4,
    maxFlowPoints: 30,
    animationSpeed: 1,
    reducedMotion: false,
    deviceType: 'desktop'
};





// Initialize device detection


// Particle System Class
class ParticleSystem {
    constructor(canvas) {
        this.canvas = canvas;
        this.ctx = canvas.getContext('2d');
        this.particles = [];
        this.mouse = { x: 0, y: 0 };
        this.isActive = true;
        
        this.setupCanvas();
        this.createParticles();
        this.bindEvents();
    }

    setupCanvas() {
        this.resizeCanvas();
        window.addEventListener('resize', () => this.resizeCanvas());
    }

    resizeCanvas() {
        const rect = this.canvas.getBoundingClientRect();
        this.canvas.width = rect.width * window.devicePixelRatio;
        this.canvas.height = rect.height * window.devicePixelRatio;
        this.ctx.scale(window.devicePixelRatio, window.devicePixelRatio);
        this.canvas.style.width = rect.width + 'px';
        this.canvas.style.height = rect.height + 'px';
    }

    createParticles() {
        const count = PERFORMANCE_SETTINGS.maxParticles;
        for (let i = 0; i < count; i++) {
            this.particles.push(new Particle(this.canvas.width / window.devicePixelRatio, this.canvas.height / window.devicePixelRatio));
        }
    }

    bindEvents() {
        this.canvas.addEventListener('mousemove', (e) => {
            const rect = this.canvas.getBoundingClientRect();
            this.mouse.x = e.clientX - rect.left;
            this.mouse.y = e.clientY - rect.top;
        });
    }

    animate() {
        if (!this.isActive) return;

        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
        
        // Update and draw particles
        this.particles.forEach(particle => {
            particle.update(this.mouse);
            particle.draw(this.ctx);
        });

        // Draw connections between nearby particles
        this.drawConnections();

        requestAnimationFrame(() => this.animate());
    }

    drawConnections() {
        for (let i = 0; i < this.particles.length; i++) {
            for (let j = i + 1; j < this.particles.length; j++) {
                const dx = this.particles[i].x - this.particles[j].x;
                const dy = this.particles[i].y - this.particles[j].y;
                const distance = Math.sqrt(dx * dx + dy * dy);

                if (distance < 120) {
                    const opacity = (120 - distance) / 120 * 0.3;
                    this.ctx.strokeStyle = `rgba(${BRAND_COLORS.primaryRgb.join(',')}, ${opacity})`;
                    this.ctx.lineWidth = 1;
                    this.ctx.beginPath();
                    this.ctx.moveTo(this.particles[i].x, this.particles[i].y);
                    this.ctx.lineTo(this.particles[j].x, this.particles[j].y);
                    this.ctx.stroke();
                }
            }
        }
    }

    start() {
        this.isActive = true;
        this.animate();
    }

    stop() {
        this.isActive = false;
    }
}

// Particle Class
class Particle {
    constructor(canvasWidth, canvasHeight) {
        this.x = Math.random() * canvasWidth;
        this.y = Math.random() * canvasHeight;
        this.vx = (Math.random() - 0.5) * 2 * PERFORMANCE_SETTINGS.animationSpeed;
        this.vy = (Math.random() - 0.5) * 2 * PERFORMANCE_SETTINGS.animationSpeed;
        this.radius = Math.random() * 3 + 1;
        this.color = Math.random() > 0.5 ? BRAND_COLORS.primary : BRAND_COLORS.secondary;
        this.opacity = Math.random() * 0.5 + 0.3;
        this.canvasWidth = canvasWidth;
        this.canvasHeight = canvasHeight;
        this.originalVx = this.vx;
        this.originalVy = this.vy;
    }

    update(mouse) {
        // Mouse interaction
        const dx = mouse.x - this.x;
        const dy = mouse.y - this.y;
        const distance = Math.sqrt(dx * dx + dy * dy);

        if (distance < 100) {
            const force = (100 - distance) / 100;
            this.vx -= (dx / distance) * force * 0.5;
            this.vy -= (dy / distance) * force * 0.5;
        } else {
            // Return to original velocity
            this.vx += (this.originalVx - this.vx) * 0.02;
            this.vy += (this.originalVy - this.vy) * 0.02;
        }

        // Update position
        this.x += this.vx;
        this.y += this.vy;

        // Bounce off edges
        if (this.x < 0 || this.x > this.canvasWidth) {
            this.vx *= -1;
            this.originalVx *= -1;
        }
        if (this.y < 0 || this.y > this.canvasHeight) {
            this.vy *= -1;
            this.originalVy *= -1;
        }

        // Keep within bounds
        this.x = Math.max(0, Math.min(this.canvasWidth, this.x));
        this.y = Math.max(0, Math.min(this.canvasHeight, this.y));
    }

    draw(ctx) {
        ctx.save();
        ctx.globalAlpha = this.opacity;
        ctx.fillStyle = this.color;
        ctx.beginPath();
        ctx.arc(this.x, this.y, this.radius, 0, Math.PI * 2);
        ctx.fill();
        
        // Add glow effect
        ctx.shadowColor = this.color;
        ctx.shadowBlur = 10;
        ctx.fill();
        ctx.restore();
    }
}

// Geometry Animation Class
class GeometryAnimation {
    constructor(canvas) {
        this.canvas = canvas;
        this.ctx = canvas.getContext('2d');
        this.shapes = [];
        this.time = 0;
        this.isActive = true;
        
        this.setupCanvas();
        this.createShapes();
    }

    setupCanvas() {
        this.resizeCanvas();
        window.addEventListener('resize', () => this.resizeCanvas());
    }

    resizeCanvas() {
        const rect = this.canvas.getBoundingClientRect();
        this.canvas.width = rect.width * window.devicePixelRatio;
        this.canvas.height = rect.height * window.devicePixelRatio;
        this.ctx.scale(window.devicePixelRatio, window.devicePixelRatio);
        this.canvas.style.width = rect.width + 'px';
        this.canvas.style.height = rect.height + 'px';
    }

    createShapes() {
        const count = PERFORMANCE_SETTINGS.maxGeometryShapes;
        for (let i = 0; i < count; i++) {
            this.shapes.push(new GeometricShape(this.canvas.width / window.devicePixelRatio, this.canvas.height / window.devicePixelRatio, i));
        }
    }

    animate() {
        if (!this.isActive) return;

        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
        this.time += 0.01 * PERFORMANCE_SETTINGS.animationSpeed;

        this.shapes.forEach(shape => {
            shape.update(this.time);
            shape.draw(this.ctx);
        });

        requestAnimationFrame(() => this.animate());
    }

    start() {
        this.isActive = true;
        this.animate();
    }

    stop() {
        this.isActive = false;
    }
}

// Geometric Shape Class
class GeometricShape {
    constructor(canvasWidth, canvasHeight, index) {
        this.x = Math.random() * canvasWidth;
        this.y = Math.random() * canvasHeight;
        this.size = Math.random() * 60 + 20;
        this.rotation = 0;
        this.rotationSpeed = (Math.random() - 0.5) * 0.02;
        this.color = index % 2 === 0 ? BRAND_COLORS.primary : BRAND_COLORS.secondary;
        this.opacity = Math.random() * 0.3 + 0.1;
        this.type = Math.floor(Math.random() * 5); // 0: triangle, 1: square, 2: hexagon, 3: circle, 4: diamond
        this.canvasWidth = canvasWidth;
        this.canvasHeight = canvasHeight;
        this.offsetX = Math.random() * 100;
        this.offsetY = Math.random() * 100;
    }

    update(time) {
        this.rotation += this.rotationSpeed;
        
        // Floating motion
        this.x += Math.sin(time + this.offsetX) * 0.5;
        this.y += Math.cos(time + this.offsetY) * 0.3;
        
        // Keep within bounds with padding
        if (this.x < -this.size) this.x = this.canvasWidth + this.size;
        if (this.x > this.canvasWidth + this.size) this.x = -this.size;
        if (this.y < -this.size) this.y = this.canvasHeight + this.size;
        if (this.y > this.canvasHeight + this.size) this.y = -this.size;
    }

    draw(ctx) {
        ctx.save();
        ctx.translate(this.x, this.y);
        ctx.rotate(this.rotation);
        ctx.globalAlpha = this.opacity;
        ctx.strokeStyle = this.color;
        ctx.lineWidth = 2;

        ctx.beginPath();
        
        switch (this.type) {
            case 0: // Triangle
                ctx.moveTo(0, -this.size / 2);
                ctx.lineTo(-this.size / 2, this.size / 2);
                ctx.lineTo(this.size / 2, this.size / 2);
                ctx.closePath();
                break;
            case 1: // Square
                ctx.rect(-this.size / 2, -this.size / 2, this.size, this.size);
                break;
            case 2: // Hexagon
                for (let i = 0; i < 6; i++) {
                    const angle = (i * Math.PI) / 3;
                    const x = Math.cos(angle) * this.size / 2;
                    const y = Math.sin(angle) * this.size / 2;
                    if (i === 0) ctx.moveTo(x, y);
                    else ctx.lineTo(x, y);
                }
                ctx.closePath();
                break;
            case 3: // Circle
                ctx.arc(0, 0, this.size / 2, 0, Math.PI * 2);
                break;
            case 4: // Diamond
                ctx.moveTo(0, -this.size / 2);
                ctx.lineTo(this.size / 2, 0);
                ctx.lineTo(0, this.size / 2);
                ctx.lineTo(-this.size / 2, 0);
                ctx.closePath();
                break;
        }
        
        ctx.stroke();
        ctx.restore();
    }
}

// Flow Animation Class
class FlowAnimation {
    constructor(canvas) {
        this.canvas = canvas;
        this.ctx = canvas.getContext('2d');
        this.flowPoints = [];
        this.time = 0;
        this.isActive = true;

        this.setupCanvas();
        this.createFlowPoints();
    }

    setupCanvas() {
        this.resizeCanvas();
        window.addEventListener('resize', () => this.resizeCanvas());
    }

    resizeCanvas() {
        const rect = this.canvas.getBoundingClientRect();
        this.canvas.width = rect.width * window.devicePixelRatio;
        this.canvas.height = rect.height * window.devicePixelRatio;
        this.ctx.scale(window.devicePixelRatio, window.devicePixelRatio);
        this.canvas.style.width = rect.width + 'px';
        this.canvas.style.height = rect.height + 'px';
    }

    createFlowPoints() {
        const count = PERFORMANCE_SETTINGS.maxFlowPoints;
        for (let i = 0; i < count; i++) {
            this.flowPoints.push(new FlowPoint(this.canvas.width / window.devicePixelRatio, this.canvas.height / window.devicePixelRatio, i));
        }
    }

    animate() {
        if (!this.isActive) return;

        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
        this.time += 0.005 * PERFORMANCE_SETTINGS.animationSpeed;

        // Draw flowing curves
        this.drawFlowingCurves();

        requestAnimationFrame(() => this.animate());
    }

    drawFlowingCurves() {
        // Update all flow points
        this.flowPoints.forEach(point => point.update(this.time));

        // Primary flowing lines
        this.ctx.strokeStyle = `rgba(${BRAND_COLORS.primaryRgb.join(',')}, 0.3)`;
        this.ctx.lineWidth = 2;
        this.ctx.lineCap = 'round';

        this.ctx.beginPath();
        for (let i = 0; i < this.flowPoints.length - 1; i++) {
            const current = this.flowPoints[i];
            const next = this.flowPoints[i + 1];

            if (i === 0) {
                this.ctx.moveTo(current.x, current.y);
            }

            // Create smooth curves between points
            const cpx = (current.x + next.x) / 2;
            const cpy = (current.y + next.y) / 2 + Math.sin(this.time * 2 + i) * 25;

            this.ctx.quadraticCurveTo(current.x, current.y, cpx, cpy);
        }
        this.ctx.stroke();

        // Secondary flowing lines with different pattern
        this.ctx.strokeStyle = `rgba(${BRAND_COLORS.secondaryRgb.join(',')}, 0.2)`;
        this.ctx.lineWidth = 1.5;

        for (let i = 0; i < this.flowPoints.length - 1; i += 2) {
            const current = this.flowPoints[i];
            const next = this.flowPoints[Math.min(i + 2, this.flowPoints.length - 1)];

            this.ctx.beginPath();
            this.ctx.moveTo(current.x, current.y);

            const cpx = (current.x + next.x) / 2 + Math.cos(this.time * 1.5 + i) * 40;
            const cpy = (current.y + next.y) / 2 + Math.sin(this.time * 1.2 + i) * 15;

            this.ctx.quadraticCurveTo(cpx, cpy, next.x, next.y);
            this.ctx.stroke();
        }

        // Tertiary subtle background waves
        this.ctx.strokeStyle = `rgba(${BRAND_COLORS.primaryRgb.join(',')}, 0.1)`;
        this.ctx.lineWidth = 3;

        this.ctx.beginPath();
        for (let i = 0; i < this.canvas.width / window.devicePixelRatio; i += 20) {
            const y = (this.canvas.height / window.devicePixelRatio) / 2 +
                     Math.sin((i * 0.01) + this.time * 3) * 60 +
                     Math.cos((i * 0.005) + this.time * 2) * 30;

            if (i === 0) {
                this.ctx.moveTo(i, y);
            } else {
                this.ctx.lineTo(i, y);
            }
        }
        this.ctx.stroke();
    }

    start() {
        this.isActive = true;
        this.animate();
    }

    stop() {
        this.isActive = false;
    }
}

// Flow Point Class
class FlowPoint {
    constructor(canvasWidth, canvasHeight, index) {
        this.baseX = (canvasWidth / PERFORMANCE_SETTINGS.maxFlowPoints) * index;
        this.baseY = canvasHeight / 2;
        this.x = this.baseX;
        this.y = this.baseY;
        this.amplitude = Math.random() * 100 + 50;
        this.frequency = Math.random() * 0.02 + 0.01;
        this.phase = Math.random() * Math.PI * 2;
        this.canvasWidth = canvasWidth;
        this.canvasHeight = canvasHeight;
    }

    update(time) {
        this.x = this.baseX + Math.sin(time * this.frequency + this.phase) * 30;
        this.y = this.baseY + Math.sin(time * this.frequency * 2 + this.phase) * this.amplitude;

        // Keep within canvas bounds
        this.x = Math.max(0, Math.min(this.canvasWidth, this.x));
        this.y = Math.max(0, Math.min(this.canvasHeight, this.y));
    }
}

// Initialize canvas animations when DOM is loaded
let particleSystem, geometryAnimation, flowAnimation;

function initializeCanvasAnimations() {
    const particleCanvas = document.getElementById('particleCanvas');
    const geometryCanvas = document.getElementById('geometryCanvas');
    const flowCanvas = document.getElementById('flowCanvas');

    console.log('Canvas elements found:', {
        particle: !!particleCanvas,
        geometry: !!geometryCanvas,
        flow: !!flowCanvas
    });

    if (particleCanvas) {
        particleSystem = new ParticleSystem(particleCanvas);
        window.particleSystem = particleSystem; // Make globally accessible for performance monitoring
        particleSystem.start();
        console.log('Particle system initialized with', PERFORMANCE_SETTINGS.maxParticles, 'particles');
    }

    if (geometryCanvas) {
        geometryAnimation = new GeometryAnimation(geometryCanvas);
        window.geometryAnimation = geometryAnimation; // Make globally accessible for performance monitoring
        geometryAnimation.start();
        console.log('Geometry animation initialized with', PERFORMANCE_SETTINGS.maxGeometryShapes, 'shapes');
    }

    if (flowCanvas) {
        flowAnimation = new FlowAnimation(flowCanvas);
        window.flowAnimation = flowAnimation; // Make globally accessible for performance monitoring
        flowAnimation.start();
        console.log('Flow animation initialized with', PERFORMANCE_SETTINGS.maxFlowPoints, 'flow points');
    }

    console.log('Canvas animations initialized successfully! 🎨');
}

// Advanced performance monitoring and optimization

// Frame rate limiting for better performance
class FrameRateLimiter {
    constructor(targetFPS = 60) {
        this.targetFPS = targetFPS;
        this.frameInterval = 1000 / targetFPS;
        this.lastFrameTime = 0;
    }

    shouldRender(currentTime) {
        if (currentTime - this.lastFrameTime >= this.frameInterval) {
            this.lastFrameTime = currentTime;
            return true;
        }
        return false;
    }
}

// Visibility API for pausing animations when tab is not active
function setupVisibilityControl() {
    let isVisible = !document.hidden;

    document.addEventListener('visibilitychange', () => {
        isVisible = !document.hidden;

        if (window.particleSystem) {
            window.particleSystem.isActive = isVisible;
            if (isVisible) window.particleSystem.animate();
        }

        if (window.geometryAnimation) {
            window.geometryAnimation.isActive = isVisible;
            if (isVisible) window.geometryAnimation.animate();
        }

        if (window.flowAnimation) {
            window.flowAnimation.isActive = isVisible;
            if (isVisible) window.flowAnimation.animate();
        }

        console.log('Canvas animations', isVisible ? 'resumed' : 'paused', 'due to tab visibility');
    });
}

// Test function to verify canvas integration
function testCanvasIntegration() {
    const heroSection = document.querySelector('.hero-section');
    const heroContent = document.querySelector('.hero-section [style*="z-index: 10"]');
    const canvases = document.querySelectorAll('#particleCanvas, #geometryCanvas, #flowCanvas');

    console.log('Canvas Integration Test:', {
        heroSectionFound: !!heroSection,
        heroContentFound: !!heroContent,
        canvasCount: canvases.length,
        heroContentZIndex: heroContent ? getComputedStyle(heroContent).zIndex : 'not found',
        canvasesPointerEvents: Array.from(canvases).map(canvas => getComputedStyle(canvas).pointerEvents),
        servicesbadges: document.querySelectorAll('.services-badges-container > div').length
    });

    // Test if hero content is clickable
    if (heroContent) {
        const buttons = heroContent.querySelectorAll('button');
        console.log('Hero buttons found:', buttons.length);
        console.log('Hero buttons are clickable:', Array.from(buttons).every(btn =>
            getComputedStyle(btn).pointerEvents !== 'none'
        ));
    }

    return {
        canvasesInitialized: canvases.length === 3,
        heroContentAccessible: !!heroContent,
        noInterference: true
    };
}

// Export for use in main.js
window.CanvasAnimations = {
    initialize: initializeCanvasAnimations,
   
    setupVisibilityControl: setupVisibilityControl,
    testIntegration: testCanvasIntegration,
    settings: PERFORMANCE_SETTINGS
};
