<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <?php
  // Server protection settings
  set_time_limit(10); // Maximum 10 seconds execution time
  ini_set('memory_limit', '32M'); // Limit memory to 32MB
  ini_set('max_execution_time', '10');

  $GLOBALS['API_TOKEN'] = "2250c47a4397a41fe5ff8f2c2a92b240ddf46aaeded8ffddb4f29f02ffef8720";
  $GLOBALS['MAX_LOOP_ITERATIONS'] = 500; // Prevent infinite loops
  $GLOBALS['MAX_SERVICES'] = 100; // Maximum services to process

  function getBaseUrl() {
      $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
      $host = $_SERVER['HTTP_HOST'];
      $script = $_SERVER['SCRIPT_NAME'];
      $basePath = dirname($script);
      return $protocol . $host . $basePath;
  }

  function truncateText($text, $maxLength = 35) {
      if (strlen($text) <= $maxLength) {
          return $text;
      }
      return substr($text, 0, $maxLength) . '...';
  }

  function fetchservices($category_id = 10) {
    // Validate category_id to prevent injection
    $category_id = filter_var($category_id, FILTER_VALIDATE_INT);
    if ($category_id === false || $category_id < 0) {
        return ['success' => false, 'data' => ['items' => [], 'pagination' => ['total' => 0]]];
    }

    $baseUrl = getBaseUrl();
    $adminPath = (strpos($baseUrl, 'localhost') !== false) ? '/admin' : 'admin';
    $url = "{$baseUrl}{$adminPath}/cms.php/Records?category_id={$category_id}&per_page=100";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 5); // Reduced from 30 to 5 seconds
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 3); // Connection timeout
    curl_setopt($ch, CURLOPT_MAXREDIRS, 3); // Limit redirects
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        "X-API-TOKEN: {$GLOBALS['API_TOKEN']}"
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    if (curl_error($ch)) {
        curl_close($ch);
        return ['success' => false, 'data' => ['items' => [], 'pagination' => ['total' => 0]]];
    }

    curl_close($ch);

    if ($httpCode === 200) {
        // Limit response size
        if (strlen($response) > 1048576) { // 1MB limit
            return ['success' => false, 'data' => ['items' => [], 'pagination' => ['total' => 0]]];
        }

        // Decode with depth limit to prevent recursion attacks
        $data = json_decode($response, true, 10);
        return $data ?: ['success' => false, 'data' => ['items' => [], 'pagination' => ['total' => 0]]];
    }

    return ['success' => false, 'data' => ['items' => [], 'pagination' => ['total' => 0]]];
  }

  // Fetch all services
  $servicesData = fetchservices(10);
  $allServices = $servicesData['success'] ? $servicesData['data']['items'] : [];

  // Limit the number of services to prevent resource exhaustion
  if (is_array($allServices)) {
    $allServices = array_slice($allServices, 0, $GLOBALS['MAX_SERVICES']);
  } else {
    $allServices = [];
  }

  // Category configurations with names and descriptions
  $categoryInfo = [
    '79' => [
      'title' => 'Publicité Digitale',
      'subtitle' => 'Atteignez votre public cible là où il passe son temps en ligne',
      'icon' => 'fas fa-bullhorn',
      'color' => '#01acb6'
    ],
    '80' => [
      'title' => 'Contenu & Communauté',
      'subtitle' => 'Créez des connexions authentiques grâce à un contenu convaincant et à l\'engagement communautaire',
      'icon' => 'fas fa-pen-fancy',
      'color' => '#EF234B'
    ],
    '81' => [
      'title' => 'Stratégie & Croissance',
      'subtitle' => 'Développez votre entreprise avec du marketing stratégique et de l\'automatisation',
      'icon' => 'fas fa-chart-line',
      'color' => '#01acb6'
    ],
    '82' => [
      'title' => 'Développement & SEO',
      'subtitle' => 'Construisez et optimisez votre présence en ligne pour une visibilité maximale',
      'icon' => 'fas fa-code',
      'color' => '#EF234B'
    ]
  ];

  // Group services by category with loop protection
  $categories = [];
  $loopCounter = 0;
  foreach ($allServices as $service) {
    // Prevent infinite loops
    if (++$loopCounter > $GLOBALS['MAX_LOOP_ITERATIONS']) {
      break;
    }

    // Validate service structure
    if (!is_array($service) || !isset($service['values']) || !is_array($service['values'])) {
      continue;
    }

    $catId = isset($service['values'][3]) ? trim($service['values'][3]) : '';

    if (!empty($catId) && isset($categoryInfo[$catId])) {
      if (!isset($categories[$catId])) {
        $categories[$catId] = $categoryInfo[$catId];
        $categories[$catId]['services'] = [];
      }
      $categories[$catId]['services'][] = $service;
    }
  }

  // Sort categories by ID
  ksort($categories);

  // Generate meta tags from all service slugs with loop protection
  $metaKeywords = [];
  $keywordCounter = 0;
  foreach ($allServices as $service) {
    // Prevent infinite loops
    if (++$keywordCounter > $GLOBALS['MAX_LOOP_ITERATIONS']) {
      break;
    }

    // Validate service structure
    if (!is_array($service)) {
      continue;
    }

    if (!empty($service['slug']) && is_string($service['slug'])) {
      $metaKeywords[] = str_replace(['-', '_'], ' ', substr($service['slug'], 0, 100));
    }
  }
  $metaKeywordsString = substr(implode(', ', $metaKeywords), 0, 500); // Limit keywords length

  ?>
  <title>MaxMind Services - Digital Solutions</title>
  <meta name="description" content="Stimulez la croissance avec des stratégies de marketing digital basées sur les données qui offrent des résultats mesurables">
  <meta name="keywords" content="<?php echo htmlspecialchars($metaKeywordsString); ?>">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
  <style>
    /* Override footer.php body classes */
    body {
      background: #ffffff !important;
      color: #1a1a1a !important;
    }

    body.bg-black {
      background: #ffffff !important;
    }

    body.text-white {
      color: #1a1a1a !important;
    }

    .geometric-bg {

      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      pointer-events: none;
      z-index: 0;
      overflow: hidden;
    }

    .geo-shape {
      position: absolute;
      opacity: 0.08;
    }

    .circle-1 {
      width: clamp(200px, 30vw, 300px);
      height: clamp(200px, 30vw, 300px);
      border: 3px solid #01acb6;
      border-radius: 50%;
      top: 10%;
      right: 10%;
      animation: float 20s ease-in-out infinite;
    }

    .circle-2 {
      width: clamp(150px, 20vw, 200px);
      height: clamp(150px, 20vw, 200px);
      border: 2px solid #EF234B;
      border-radius: 50%;
      bottom: 20%;
      left: 5%;
      animation: float 15s ease-in-out infinite reverse;
    }

    .square-1 {
      width: clamp(100px, 15vw, 150px);
      height: clamp(100px, 15vw, 150px);
      border: 3px solid #01acb6;
      top: 50%;
      left: 15%;
      animation: rotate 30s linear infinite;
      transform-origin: center;
    }

    .square-2 {
      width: clamp(80px, 12vw, 100px);
      height: clamp(80px, 12vw, 100px);
      border: 2px solid #EF234B;
      bottom: 10%;
      right: 20%;
      animation: rotate 25s linear infinite reverse;
      transform-origin: center;
    }

    .triangle {
      width: 0;
      height: 0;
      border-left: 75px solid transparent;
      border-right: 75px solid transparent;
      border-bottom: 130px solid rgba(1, 172, 182, 0.08);
      top: 30%;
      right: 30%;
      animation: pulse 10s ease-in-out infinite;
    }

    @keyframes float {
      0%, 100% {
        transform: translateY(0) translateX(0);
      }
      25% {
        transform: translateY(-30px) translateX(20px);
      }
      50% {
        transform: translateY(-60px) translateX(-20px);
      }
      75% {
        transform: translateY(-30px) translateX(20px);
      }
    }

    @keyframes rotate {
      from {
        transform: rotate(0deg);
      }
      to {
        transform: rotate(360deg);
      }
    }

    @keyframes pulse {
      0%, 100% {
        transform: scale(1);
        opacity: 0.1;
      }
      50% {
        transform: scale(1.1);
        opacity: 0.15;
      }
    }

    /* Container */
    .container {
      max-width: 1200px;
      margin: 0 auto;
      padding: 0 20px;
      position: relative;
      z-index: 1;
    }

    h1 {
      font-size: clamp(2rem, 5vw, 4rem);
      font-weight: 800;
      color: #01acb6;
      margin-bottom: 20px;
      letter-spacing: -2px;
    }

    .subtitle {
      font-size: 1.25rem;
      color: #666;
      max-width: 600px;
      margin: 0 auto 40px;
    }

    /* Services Grid */
    .services-grid {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
      gap: 30px;
      margin-bottom: 80px;
    }

    .service-card {
      background: white;
      border-radius: 20px;
      padding: 40px;
      position: relative;
      overflow: hidden;
      cursor: pointer;
      transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
      box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .service-card::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: linear-gradient(135deg, transparent 0%, rgba(255, 23, 68, 0.05) 100%);
      opacity: 0;
      transition: opacity 0.4s ease;
    }

    .service-card:hover {
      transform: translateY(-10px);
      box-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
    }

    .service-card:hover::before {
      opacity: 1;
    }

    .service-card.expanded {
      grid-column: span 2;
    }

    .service-number {
      position: absolute;
      top: 20px;
      right: 20px;
      font-size: 5rem;
      font-weight: 900;
      opacity: 0.05;
      transition: all 0.4s ease;
      pointer-events: none;
    }

    .service-card:hover .service-number {
      opacity: 0.1;
      transform: scale(1.1);
    }

    .service-icon {
      width: clamp(50px, 8vw, 60px);
      height: clamp(50px, 8vw, 60px);
      border-radius: 12px;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: clamp(1.25rem, 3vw, 1.75rem);
      margin-bottom: 20px;
      transition: transform 0.4s ease;
    }

    .service-icon i {
      color: white;
    }

    .service-card:hover .service-icon {
      transform: rotate(10deg) scale(1.1);
    }

    .service-card:nth-child(odd) .service-icon {
      background: #01acb6;
      color: white;
    }

    .service-card:nth-child(even) .service-icon {
      background: #EF234B;
      color: white;
    }

    h3 {
      font-size: 1.75rem;
      font-weight: 700;
      margin-bottom: 15px;
      color: #1a1a1a;
    }

    .service-description {
      color: #666;
      margin-bottom: 20px;
      line-height: 1.7;
    }

    .service-features {
      max-height: 0;
      overflow: hidden;
      transition: max-height 0.5s ease;
    }

    .service-card.expanded .service-features {
      max-height: 500px;
    }

    .service-features ul {
      list-style: none;
      margin-top: 20px;
      padding-top: 20px;
      border-top: 2px solid #f0f0f0;
    }

    .service-features li {
      padding: 10px 0;
      color: #555;
      position: relative;
      padding-left: 25px;
    }

    .service-features li::before {
      content: '✓';
      position: absolute;
      left: 0;
      color: #00bcd4;
      font-weight: bold;
    }

    .cta-button {
      display: inline-block;
      padding: 12px 30px;
      background: #01acb6;
      color: white;
      text-decoration: none;
      border-radius: 25px;
      font-weight: 600;
      margin-top: 20px;
      transition: all 0.3s ease;
      border: none;
      cursor: pointer;
      font-size: 1rem;
    }

    .cta-button:hover {
      transform: translateY(-2px);
      box-shadow: 0 10px 25px rgba(0, 188, 212, 0.3);
    }

    /* Stats Section */
    .stats-section {
      background: white;
      border-radius: 30px;
      padding: 60px 40px;
      margin-bottom: 80px;
      box-shadow: 0 10px 40px rgba(0, 0, 0, 0.08);
    }

    .stats-grid {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
      gap: 40px;
      text-align: center;
    }

    .stat-item {
      transition: transform 0.3s ease;
    }

    .stat-item:hover {
      transform: translateY(-5px);
    }

    .stat-number {
      font-size: 3rem;
      font-weight: 900;
      background: #EF234B;
      -webkit-background-clip: text;
      -webkit-text-fill-color: transparent;
      background-clip: text;
      margin-bottom: 10px;
    }

    .stat-label {
      color: #666;
      font-size: 1rem;
    }

    /* Process Section */
    .process-section {
      margin-bottom: 80px;
    }

    .process-section h2 {
      text-align: center;
      font-size: 3rem;
      font-weight: 800;
      margin-bottom: 60px;
      color: #1a1a1a;
    }

    .process-timeline {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
      gap: 30px;
    }

    .process-step {
      background: white;
      padding: 30px;
      border-radius: 20px;
      text-align: center;
      transition: all 0.3s ease;
      box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .process-step:hover {
      transform: translateY(-5px);
      box-shadow: 0 10px 30px rgba(0, 0, 0, 0.12);
    }

    .process-step-number {
      width: 60px;
      height: 60px;
      border-radius: 50%;
      background: #EF234B;
      color: white;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: 1.5rem;
      font-weight: 700;
      margin: 0 auto 20px;
    }

    .process-step h4 {
      font-size: 1.25rem;
      margin-bottom: 10px;
      color: #1a1a1a;
    }

    .process-step p {
      color: #666;
      font-size: 0.95rem;
    }

    /* CTA Section */
    .cta-section {
      text-align: center;
      padding: 80px 0;
      background: linear-gradient(135deg, #ff1744 0%, #9c27b0 50%, #00bcd4 100%);
      border-radius: 30px;
      margin-bottom: 80px;
      color: white;
    }

    .cta-section h2 {
      font-size: 3rem;
      font-weight: 800;
      margin-bottom: 20px;
    }

    .cta-section p {
      font-size: 1.25rem;
      margin-bottom: 30px;
      opacity: 0.9;
    }

    .cta-section .cta-button {
      background: white;
      color: #ff1744;
      font-size: 1.1rem;
      padding: 15px 40px;
    }

    .cta-section .cta-button:hover {
      transform: translateY(-3px);
      box-shadow: 0 15px 35px rgba(0, 0, 0, 0.2);
    }

    /* Responsive */
    @media (max-width: 768px) {
      h1 {
        font-size: 2.5rem;
      }

      .service-card.expanded {
        grid-column: span 1;
      }

      .services-grid {
        grid-template-columns: 1fr;
      }

      .cta-section h2 {
        font-size: 2rem;
      }

      .process-section h2 {
        font-size: 2rem;
      }
    }
  </style>
</head>
<body>
  <?php include 'header.php'; ?>
  <!-- Geometric Background -->
  <div class="geometric-bg">
    <div class="geo-shape circle-1"></div>
    <div class="geo-shape circle-2"></div>
    <div class="geo-shape square-1"></div>
    <div class="geo-shape square-2"></div>
    <div class="geo-shape triangle"></div>
  </div>

  <div class="container">
    <!-- Header -->

    <header style="margin-top: 100px;" class="mt-20 text-center">
      <h1 style="font-size: 40px;">Nos Services</h1>
      <p class="subtitle">  Transformez votre vision en performance digitale. Maxmind réunit développement, marketing, design et stratégie pour propulser votre marque à l’ère numérique — du code à la conversion, du clic à la fidélisation.</p>
    </header>

    <?php
    $serviceNumber = 1;
    $categoryCounter = 0;
    foreach ($categories as $catId => $category):
      // Prevent infinite loops in category rendering
      if (++$categoryCounter > 10) break; // Max 10 categories
      if (empty($category['services'])) continue;
    ?>

    <!-- Section: <?php echo htmlspecialchars($category['title']); ?> -->
    <div style="margin: <?php echo $catId == '79' ? '60px' : '80px'; ?> 0 40px;">
      <h2 style="font-size: clamp(1.75rem, 3vw, 2.25rem); font-weight: 700; color: #1a1a1a; margin-bottom: 8px; display: flex; align-items: center; gap: 12px;">
        <i class="<?php echo $category['icon']; ?>" style="color: <?php echo $category['color']; ?>;"></i>
        <?php echo htmlspecialchars($category['title']); ?>
      </h2>
      <p style="color: #666; font-size: 1.1rem;"><?php echo htmlspecialchars($category['subtitle']); ?></p>
    </div>

    <!-- Services Grid -->
    <div class="services-grid">
      <?php
      $serviceCounter = 0;
      foreach ($category['services'] as $service):
        // Prevent infinite loops in service rendering
        if (++$serviceCounter > 50) break; // Max 50 services per category
        // Values array structure: [title, description, features, category_id, icon]
        // Validate service structure
        if (!is_array($service) || !isset($service['values']) || !is_array($service['values'])) {
          continue;
        }

        $title = isset($service['values'][0]) ? substr(strval($service['values'][0]), 0, 200) : 'Service';
        $description = isset($service['values'][1]) ? substr(strval($service['values'][1]), 0, 500) : '';
        $features = isset($service['values'][2]) ? substr(strval($service['values'][2]), 0, 2000) : '';

        // Handle icon value - it might be a string, array, or empty
        $iconValue = '';
        if (isset($service['values'][4])) {
          if (is_string($service['values'][4])) {
            $iconValue = trim($service['values'][4]);
          } elseif (is_array($service['values'][4]) && !empty($service['values'][4])) {
            $iconValue = is_string($service['values'][4][0]) ? trim($service['values'][4][0]) : '';
          }
        }

        // Determine icon to display
        $defaultIcons = [
          'Facebook' => 'fab fa-facebook-f',
          'Instagram' => 'fab fa-instagram',
          'Google' => 'fab fa-google',
          'TikTok' => 'fab fa-tiktok',
          'SEO' => 'fas fa-search',
          'E-commerce' => 'fas fa-shopping-cart',
          'Développement' => 'fas fa-code',
          'Contenu' => 'fas fa-pen-fancy',
          'Communauté' => 'fas fa-users',
          'Stratégie' => 'fas fa-chess',
          'Leads' => 'fas fa-magnet',
          'Automatisation' => 'fas fa-robot'
        ];

        // Default icon
        $iconClass = 'fas fa-star';
        foreach ($defaultIcons as $keyword => $faIcon) {
          if (stripos($title, $keyword) !== false) {
            $iconClass = $faIcon;
            break;
          }
        }

        // Clean up the features HTML - keep only safe tags and limit length
        $features = strip_tags($features, '<ul><li><strong><span><p><div>');
        $features = substr($features, 0, 3000); // Additional safety limit
      ?>
      <div class="service-card" onclick="toggleCard(this)">
        <div class="service-number"><?php echo str_pad($serviceNumber++, 2, '0', STR_PAD_LEFT); ?></div>
        <div class="service-icon">
          <?php if (!empty($iconValue) && filter_var($iconValue, FILTER_VALIDATE_URL)): ?>
            <img src="<?php echo htmlspecialchars($iconValue); ?>" alt="<?php echo htmlspecialchars($title); ?>" style="width: 100%; height: 100%; object-fit: contain;">
          <?php elseif (!empty($iconValue) && strpos($iconValue, 'fa') !== false): ?>
            <i class="<?php echo htmlspecialchars($iconValue); ?>"></i>
          <?php else: ?>
            <i class="<?php echo $iconClass; ?>"></i>
          <?php endif; ?>
        </div>
        <h3><?php echo htmlspecialchars($title); ?></h3>
        <p class="service-description"><?php echo htmlspecialchars($description); ?></p>
        <div class="service-features">
          <?php echo $features; ?>
        </div>
      </div>
      <?php endforeach; ?>
    </div>

    <?php endforeach; ?>
  </div>

  <script>
    function toggleCard(card) {
      // Close all other cards
      const allCards = document.querySelectorAll('.service-card');
      allCards.forEach(c => {
        if (c !== card) {
          c.classList.remove('expanded');
        }
      });

      // Toggle current card
      card.classList.toggle('expanded');
    }

    // Add smooth scroll behavior
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
      anchor.addEventListener('click', function (e) {
        e.preventDefault();
        const target = document.querySelector(this.getAttribute('href'));
        if (target) {
          target.scrollIntoView({
            behavior: 'smooth'
          });
        }
      });
    });

    // Add parallax effect to geometric shapes
    document.addEventListener('mousemove', (e) => {
      const shapes = document.querySelectorAll('.geo-shape');
      const mouseX = e.clientX / window.innerWidth;
      const mouseY = e.clientY / window.innerHeight;

      shapes.forEach((shape, index) => {
        const speed = (index + 1) * 10;
        const x = (mouseX - 0.5) * speed;
        const y = (mouseY - 0.5) * speed;
        shape.style.transform = `translate(${x}px, ${y}px)`;
      });
    });
  </script>

  <?php include 'footer.php'; ?>

</body>
</html>
